//! This crate provides implementations of the [standards][standards] and [algorithms][algorithms] used with the IndieWeb.
//!
//! More information about what's available is in either the [algorithms][algorithms] or
//! [standards][standards] module. A required trait to use is the [HTTP Client][http::Client]
//! if you'd like to use your own networking stack that's compatible with [http][::http]. This
//! library also provides some [traits][traits] to extend common values with IndieWeb-adjacent
//! capabilities.
//!
//! # Quick Start
//!
//! ```
//! use indieweb::prelude::*;
//! ```
//!
#![warn(invalid_doc_attributes, unused, deprecated, clippy::perf)]
#![deny(rustdoc::broken_intra_doc_links, dead_code, unsafe_code)]

/// A collection of algorithms commonly used in the IndieWeb.
/// This module provides a collection of implementation of known
/// algorithms when working with the IndieWeb or adjacent tooling.
pub mod algorithms;

/// A representation of errors from the IndieWeb error.
pub mod error;

/// A facade for HTTP interactions when working with this library.
pub mod http;

/// A collection of standards that the IndieWeb can support.
///
/// View <https://spec.indieweb.org> for more information.
pub mod standards;

/// Traits to extend everyday functionality with IndieWeb-adjacent tooling.
pub mod traits;

/// A prelude module for ergonomic imports.
///
/// ```
/// use indieweb::prelude::*;
/// ```
pub mod prelude;

mod test;

// ============================================================================
// Re-exports for microformats (core to this library)
// ============================================================================
pub use microformats as mf2;
pub use mf2::parse as parser;
pub use mf2::types::{Document, Item, PropertyValue};

// ============================================================================
// Re-exports for HTTP clients (for simpler imports)
// 
// These provide flatter import paths:
//   - indieweb::ReqwestClient  (instead of indieweb::http::reqwest::Client)
//   - indieweb::BlockingClient (instead of indieweb::http::blocking::Client)
//   - indieweb::MiddlewareClient (instead of indieweb::http::MiddlewareClient)
// ============================================================================
#[cfg(feature = "reqwest")]
pub use http::reqwest::Client as ReqwestClient;

#[cfg(feature = "blocking")]
pub use http::blocking::Client as BlockingClient;

#[cfg(feature = "reqwest_middleware")]
pub use http::MiddlewareClient;

/// Result type alias for convenience
pub use error::Error;
pub type Result<T, E = Error> = std::result::Result<T, E>;
