---Copyright (C) 2022-2025 by Josef Friedrich <josef@friedrich.rocks>
------------------------------------------------------------------------
---
---This program is free software: you can redistribute it and/or modify it
---under the terms of the GNU General Public License as published by the
---Free Software Foundation, either version 2 of the License, or (at your
---option) any later version.
---
---This program is distributed in the hope that it will be useful, but
---WITHOUT ANY WARRANTY; without even the implied warranty of
---MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
---Public License for more details.
---
---You should have received a copy of the GNU General Public License along
---with this program. If not, see <https://www.gnu.org/licenses/>.
---@meta
---
---The token library provides means to intercept the input and deal with it at the
---Lua level. The library provides a basic scanner infrastructure that can be used
---to write macros that accept a wide range of arguments. This interface is on
---purpose kept general and as performance is quite ok. One can build additional
---parsers without too much overhead. It's up to macro package writers to see how
---they can benefit from this as the main principle behind LuaTeX is to provide a
---minimal set of tools and no solutions. The scanner functions are probably the
---most intriguing.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L2091-L2098](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L2091-L2098)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
token = {}

---
---10.6 The token library

---
---Scan and gobble a given keyword.
---
---As with the regular *TeX* keyword scanner this is case insensitive (and ASCII based).
---
---__Example:__
---
---```latex
---\def\scanner{\directlua{
---  print(token.scan_keyword('keyword'))
---}}
---\scanner keyword % true
---\scanner KEYWORD % true
---\scanner not the keyword % false
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L339-L353](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L339-L353)
---
---@param keyword string # An ASCII based keyword to scan for.
---
---@return boolean # True if the keyword could be gobbled up otherwise false.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_keyword(keyword) end

---
--- Scan and gobble a given case sensitive and *UTF-8* based keyword.
---
---__Example:__
---
---```tex
---\def\scanner{\directlua{
---  print(token.scan_keyword_cs('Keyword'))
---}}
---\scanner Keyword % true
---\scanner keyword % false
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L355-L369](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L355-L369)
---
---@param keyword string # A case sensitive and *UTF-8* based keyword
---
---@return boolean # True if the case sensitive and *UTF-8* based keyword could be gobbled up otherwise false.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_keyword_cs(keyword) end

---
---Scan and gobble a given integer.
---
---__Example:__
---
---```latex
---\def\scanner{\directlua{
---  print(token.scan_int())
---}}
---\scanner 1 % 1
---\scanner 1.1 % 1 (Scans only 1 not 1.1)
---\scanner -1 % -1
---\scanner 1234567890 % 1234567890
---\scanner string % Missing number, treated as zero
---\scanner 12345678901 % Number to big
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L391-L401](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L391-L401)
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_int() end

---
---Scan and gobble a floating point number that cannot have an exponent (`1E10` is scanned as `1.0`).
---
---__Example:__
---
---```tex
---\def\scan{\directlua{
---  print(token.scan_real())
---}}
---\scan 1E10 % 1.0 Does not scan “E10“
---\scan 1 % 1.0
---\scan 1.1 % 1.1
---\scan .1 % 0.1
---\scan - .1 % -0.1
---\scan -1 % -1.0
---\scan - 1 % -1.0
---\scan 1234567890 % 1234567890.0
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L530-L533](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L530-L533)
---
---@return number
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_real() end

---
---Scan and gobble a floating point number that can be provided with an exponent (e. g. `1E10`).
---
---__Example:__
---
---```tex
---\def\scan{\directlua{
---  print(token.scan_float())
---}}
---\scan 1E10 % 10000000000.0
---\scan .1e-10 % 1e-11
---\scan 1 % 1.0
---\scan 1.1 % 1.1
---\scan .1 % 0.1
---\scan - .1 % -0.1
---\scan -1 % -1.0
---\scan - 1 % -1.0
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L525-L528](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L525-L528)
---
---@return number
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_float() end

---
---Returns a number representing a dimension and or two numbers being the filler and order
---
---__Example:__
---
---Parameter `inf`:
---
---```tex
---\directlua{token.scan_dimen(true)}1fi % 1
---\directlua{token.scan_dimen(true)}1fil % 2
---\directlua{token.scan_dimen(true)}1fill % 3
---\directlua{token.scan_dimen(true)}1filll % 4
---
---```
---
---Parameter `mu`:
---
---```tex
---\directlua{token.scan_dimen(false, true)}1mu % 65536
---\directlua{token.scan_dimen(false, true)}1cm % Illegal unit of measure (mu inserted).
---```
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L535-L557](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L535-L557)
---
---@param inf? boolean # inf values allowed
---@param mu? boolean # mu (math units) units required
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_dimen(inf, mu) end

---
---returns a glue spec node
---
---__Example:__
---
---```tex
---\def\scan{\directlua{
---  local node = token.scan_glue()
---  print(node.width, node.stretch, node.stretch_order, node.shrink, node.shrink_order)
---}}
---\def\scanMu{\directlua{
---  local node = token.scan_glue(true)
---  print(node.width, node.stretch, node.stretch_order, node.shrink, node.shrink_order)
---}}
---\scan 1pt % 65536 0 0 0 0
---\scan 1pt plus 2pt % 65536 131072 0 0 0
---\scan 1pt minus 3pt % 65536 0 0 196608 0
---\scan 1pt plus 2pt minus 3pt % 65536 131072 0 196608 0
---\scan 1pt plus 2fi minus 3fi % 65536 131072 1 196608 1
---\scan 1pt plus 2fil minus 3fil % 65536 131072 2 196608 2
---\scan 1pt plus 2fill minus 3fill % 65536 131072 3 196608 3
---\scan 1pt plus 2filll minus 3filll % 65536 131072 4 196608 4
---\scan string % Missing number, treated as zero.
---\scanMu 3mu % 196608 0 0 0 0
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L559-L573](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L559-L573)
---
---@param mu_units? boolean
---
---@return GlueSpecNode
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_glue(mu_units) end

---
---Scan a list of tokens delimited by balanced braces.
---
---__Example:__
---
---```tex
---\directlua{
---  local t = token.scan_toks()
---  for id, tok in ipairs(t) do
---    print(id, tok, tok.command, tok.cmdname, tok.csname)
---  end
---}{Some text}
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L575-L602](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L575-L602)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)-
---
---@param definer? boolean `macro_def`, `\def`
---@param expand? boolean
---
---@return Token[]
function token.scan_toks(definer, expand) end

---
---Return a character if its category is in the given bitset (representing catcodes)
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L730-L756](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L730-L756)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_code(bitset) end

---
---Scan and gobble a string.
---
---The string scanner scans for something between curly braces and expands on the
---way, or when it sees a control sequence it will return its meaning. Otherwise it
---will scan characters with catcode `letter` or `other`.
---
---__Example:__
---
---```tex
---\def\scan{\directlua{
---  print(token.scan_string())
---}}
---\def\bar{bar}
---\def\foo{\bar}
---\scan \foo % bar
---\scan {\foo} % bar
---\scan {A string} % A string
---\scan A string % A
---\scan Word1 Word2 % Word1
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L604-L641](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L604-L641)
---
---@return string # A string given between `{` `}`, as `\macro` or as sequence of characters with catcode 11 or 12
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_string() end

---
---Scan and gobble an argument.
---
---This function is simular to `token.scan_string` but also accepts a `\cs`.
---It expands the given argument. When a braced
---argument is scanned, expansion can be prohibited by passing `false`
---(default is `true`). In case of a control sequence passing `false`
---will result in a one-level expansion (the meaning of the macro).
---
---__Example:__
---
---```latex
---\def\scan{\directlua{
---  print(token.scan_argument(true))
---}}
---\def\scanNoExpand{\directlua{
---  print(token.scan_argument(false))
---}}
---\def\foo{bar}
---\scan \foo % bar
---\scan { {\bf text} } % {\fam \bffam \tenbf text}
---\scanNoExpand { {\bf text} } % {\bf text}
---\scan c % c
---\scan \bf % \fam \bffam \tenbf
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L643-L702](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L643-L702)
---
---@param expand? boolean # When a braced argument is scanned, expansion can be prohibited by passing `false` (default is `true`)
---
---@return string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_argument(expand) end

---
---Return a sequence of characters with catcode `11` or `12` as a string.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L704-L728](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L704-L728)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_word() end

---
---Return `foo` after scanning `\foo`.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L371-L389](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L371-L389)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_csname() end

---
---Pick up a box specification and return a `[h|v]list` node.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L1100-L1111](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L1100-L1111)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_list() end

---
---Scan and gobble the next token.
---
---The different scanner functions of the `token` library look for a
---sequence of tokens. This function scans just the next token.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L2237-L2239](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L2237-L2239)
---* Corresponding C source code: [lnewtokenlib.c#L231-L239](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L231-L239)
---
---@return Token
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_next() end

---
---Use `scan_token`
---if you want to
---enforce expansion first you can.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L1090-L1098](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L1090-L1098)
---
---@return Token
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.scan_token() end

---
---Trigger expansion of the next token in the
---input.
---
---This can be quite unpredictable but when you call it you probably know
---enough about *TeX* not to be too worried about that. It basically is a call to
---the internal expand related function.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L764-L769](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L764-L769)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.expand() end

---
---__Reference:__
---
---* Corresponding C source code: [luatoken.c#L192-L351](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/luatoken.c#L192-L351)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias TokenCommandName
---| 'relax'
---| 'left_brace'
---| 'right_brace'
---| 'math_shift'
---| 'tab_mark'
---| 'car_ret'
---| 'mac_param'
---| 'sup_mark'
---| 'sub_mark'
---| 'endv'
---| 'spacer'
---| 'letter'
---| 'other_char'
---| 'par_end'
---| 'stop'
---| 'delim_num'
---| 'char_num'
---| 'math_char_num'
---| 'mark'
---| 'node'
---| 'xray'
---| 'make_box'
---| 'hmove'
---| 'vmove'
---| 'un_hbox'
---| 'un_vbox'
---| 'remove_item'
---| 'hskip'
---| 'vskip'
---| 'mskip'
---| 'kern'
---| 'mkern'
---| 'leader_ship'
---| 'halign'
---| 'valign'
---| 'no_align'
---| 'vrule'
---| 'hrule'
---| 'novrule'
---| 'nohrule'
---| 'insert'
---| 'vadjust'
---| 'ignore_spaces'
---| 'after_assignment'
---| 'after_group'
---| 'partoken_name'
---| 'break_penalty'
---| 'start_par'
---| 'ital_corr'
---| 'accent'
---| 'math_accent'
---| 'discretionary'
---| 'eq_no'
---| 'left_right'
---| 'math_comp'
---| 'limit_switch'
---| 'above'
---| 'math_style'
---| 'math_choice'
---| 'non_script'
---| 'vcenter'
---| 'case_shift'
---| 'message'
---| 'normal'
---| 'extension'
---| 'option'
---| 'lua_function_call'
---| 'lua_bytecode_call'
---| 'lua_call'
---| 'in_stream'
---| 'begin_group'
---| 'end_group'
---| 'omit'
---| 'ex_space'
---| 'boundary'
---| 'radical'
---| 'super_sub_script'
---| 'no_super_sub_script'
---| 'math_shift_cs'
---| 'end_cs_name'
---| 'char_ghost'
---| 'assign_local_box'
---| 'char_given'
---| 'math_given'
---| 'xmath_given'
---| 'last_item'
---| 'toks_register'
---| 'assign_toks'
---| 'assign_int'
---| 'assign_attr'
---| 'assign_dimen'
---| 'assign_glue'
---| 'assign_mu_glue'
---| 'assign_font_dimen'
---| 'assign_font_int'
---| 'assign_hang_indent'
---| 'set_aux'
---| 'set_prev_graf'
---| 'set_page_dimen'
---| 'set_page_int'
---| 'set_box_dimen'
---| 'set_tex_shape'
---| 'set_etex_shape'
---| 'def_char_code'
---| 'def_del_code'
---| 'extdef_math_code'
---| 'extdef_del_code'
---| 'def_family'
---| 'set_math_param'
---| 'set_font'
---| 'def_font'
---| 'def_lua_call'
---| 'register'
---| 'assign_box_direction'
---| 'assign_box_dir'
---| 'assign_direction'
---| 'assign_dir'
---| 'combinetoks'
---| 'advance'
---| 'multiply'
---| 'divide'
---| 'prefix'
---| 'let'
---| 'shorthand_def'
---| 'read_to_cs'
---| 'def'
---| 'set_box'
---| 'hyph_data'
---| 'set_interaction'
---| 'letterspace_font'
---| 'expand_font'
---| 'copy_font'
---| 'set_font_id'
---| 'undefined_cs'
---| 'expand_after'
---| 'no_expand'
---| 'input'
---| 'lua_expandable_call'
---| 'lua_local_call'
---| 'if_test'
---| 'fi_or_else'
---| 'cs_name'
---| 'convert'
---| 'variable'
---| 'feedback'
---| 'the'
---| 'top_bot_mark'
---| 'call'
---| 'long_call'
---| 'outer_call'
---| 'long_outer_call'
---| 'end_template'
---| 'dont_expand'
---| 'glue_ref'
---| 'shape_ref'
---| 'box_ref'
---| 'data'

---
---A token object has the properties of the `relax`
---primitive.
---
---Alternatively you can use a getter `get_<fieldname>` to access a property
---of a token.
---
---The numbers that represent a catcode are the same as in *TeX* itself, so using
---this information assumes that you know a bit about *TeX*'s internals. The other
---numbers and names are used consistently but are not frozen. So, when you use them
---for comparing you can best query a known primitive or character first to see the
---values.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L2285-L2310](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L2285-L2310)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class Token
---@field command integer # A number representing the internal command number (catcode in case of a character), for example: `1` (`left_brace`), `2` (`right_brace`), `3` (`math_shift`), `4` (`tab_mark`), `6` (`mac_param`), `7` (`sup_mark`), `8` (`sub_mark`), `10 ` (`spacer`), `11 ` (`letter`), `12 ` (`other`), `147` (`call`).
---@field cmdname TokenCommandName # The type of the command, for example `letter`.
---@field csname string|nil # The associated control sequence (if applicable), for example `bigskip`.
---@field id integer # The unique id of the token, for example `6876`.
---@field tok integer # The full token number as stored in TeX, for example `536883863`.
---@field active boolean # A boolean indicating the active state of the token, for example `true`.
---@field expandable boolean # A boolean indicating if the token (macro) is expandable, for example `true`.
---@field public protected boolean # A boolean indicating if the token (macro) is protected, for example `false`.
---@field mode integer # A number either representing a character or another entity, for example `1007`.
---@field index integer|nil # A number running from 0x0000 upto 0xFFFF indicating a `TeX` register index. In case of letters the unicode code point, for example `398` = `Ǝ`.

---
---Return the internal command number.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L870-L880](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L870-L880)
---
---@param t Token
---
---@return integer command # A number representing the internal command number, for example `147`.
---
---@see Token.command
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_command(t) end

---
---Return the type of the command (for instance the catcode in case of a character or the classifier that determines the internal treatment, for example `letter`.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L936-L943](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L936-L943)
---
---@param t Token
---
---@return TokenCommandName cmdname # The type of the command (for instance the catcode in case of a character or the classifier that determines the internal treatment, for example `letter`.
---
---@see Token.cmdname
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_cmdname(t) end

---
---Return the associated control sequence (if applicable), for example `bigskip`.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L945-L959](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L945-L959)
---
---@param t Token
---
---@return string|nil csname # The associated control sequence (if applicable), for example `bigskip`.
---
---@see Token.csname
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_csname(t) end

---
---Return the unique id of the token.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L961-L966](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L961-L966)
---
---@param t Token
---
---@return integer id # The unique id of the token, for example `6876`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_id(t) end

---
---Return the full token number as stored in TeX.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L968-L974](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L968-L974)
---
---@see Token.tok
---
---@param t Token
---
---@return integer tok # The full token number as stored in TeX, for example `536883863`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_tok(t) end

---
---Return a boolean indicating the active state of the token, for example `true`.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L976-L991](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L976-L991)
---
---@param t Token
---
---@return boolean active # A boolean indicating the active state of the token, for example `true`.
---
---@see Token.active
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_active(t) end

---
---Return a boolean indicating if the token (macro) is expandable.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L993-L1004](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L993-L1004)
---
---@param t Token
---
---@return boolean expandable # A boolean indicating if the token (macro) is expandable, for example `true`.
---
---@see Token.expandable
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_expandable(t) end

---
---Return a boolean indicating if the token (macro) is protected.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L1006-L1022](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L1006-L1022)
---
---@param t Token
---
---@return boolean protected # A boolean indicating if the token (macro) is protected, for example `false`.
---
---@see Token.protected
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_protected(t) end

---
---Return a number either representing a character or another entity.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L924-L934](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L924-L934)
---
---@param t Token
---
---@return integer mode # A number either representing a character or another entity, for example `1007`.
---
---@see Token.mode
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_mode(t) end

---
---Return a number running from `0x0000` upto `0xFFFF` indicating a TeX register index.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L882-L922](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L882-L922)
---
---@param t Token
---
---@return integer index # A number running from 0x0000 upto 0xFFFF indicating a TeX register index, for example `1007`.
---
---@see Token.index
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_index(t) end

---
---Get the content of a macro.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L1141-L1166](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L1141-L1166)
---
---@param name string # The name of the macro without the leading backslash.
---
---@return string # for example `foo #1 bar`.
---
---@see token.set_macro
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_macro(name) end

---
---Get the meaning of a macro including the argument
---specification (as usual in *TeX* separated by `->`).
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L1121-L1139](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L1121-L1139)
---
---@param name string # The name of the macro without the leading backslash.
---
---@return string # for example `->foo #1 bar`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.get_meaning(name) end

---
---Ask for a list of commands.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L1373-L1382](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L1373-L1382)
---
---@return table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.commands() end

---
---Return the id of a token class.
---
---@param cmdname TokenCommandName # for example `letter`
---
---@return integer|nil
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L201-L213](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L201-L213)
function token.command_id(cmdname) end

---
---Create a token.
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L826-L845](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L826-L845)
---
---@param chr integer
---@param cmd? integer
---
---@return Token
---
---@see token.new
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.create(chr, cmd) end

---
--- A variant that ignores the current catcode table is:
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L847-L854](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L847-L854)
---
---@param chr integer
---@param cmd integer
---
---@return Token
---
---@see token.create
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.new(chr, cmd) end

---
---__Example:__
---
---```tex
---\def\foo{bar}
---\directlua{
---  print(token.is_defined('foo')) % true
---  print(token.is_defined('nofoo')) % false
---  print(token.is_defined('bf')) % true
---}
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L807-L824](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L807-L824)
---
---@param cs string
---
---@return boolean
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.is_defined(cs) end

---
---__Example:__
---
---```lua
---print(token.biggest_char()) % 1114111
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L185-L189](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L185-L189)
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.biggest_char() end

---
---Create a macro.
---
---__Example:__
---
---```tex
---\directlua{
---  token.set_macro("test", "content")
---}
---\test
---```
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L2368-L2382](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L2368-L2382)
---* Corresponding C source code: [lnewtokenlib.c#L1218-L1342](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L1218-L1342)
---
---@param csname string
---@param content? string
---@param global? 'global'
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.set_macro(csname, content, global) end

---
---Create a macro.
---
---__Example:__
---
---```tex
---\directlua{
---  token.set_macro("test", "content")
---}
---\test
---```
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L2368-L2382](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L2368-L2382)
---* Corresponding C source code: [lnewtokenlib.c#L1218-L1342](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L1218-L1342)
---
---@param catcodetable integer # A catcodetable identifier.
---@param csname string
---@param content? string
---@param global? 'global'
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.set_macro(catcodetable, csname, content, global) end

---
---Do a `chardef` at the
---*Lua* end, where invalid assignments are silently ignored.
---
---__Example:__
---
---```tex
---\directlua{
---  token.set_char('myT', 84)
---  token.set_char('mye', 101)
---  token.set_char('myX', 88)
---}
---\myT\mye\myX % TeX
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L1344-L1371](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L1344-L1371)
---
---@param csname string
---@param number integer
---@param global? 'global'
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.set_char(csname, number, global) end

---
---Create a token that refers to a *Lua* function with an entry in the table
---that you can access with `lua.get_functions_table`. It is the companion
---to `luadef`.
---
---__Example:__
---
---```tex
---\directlua{
---  local t = lua.get_functions_table()
---
---  t[1] = function () end

---
---  local index = 1
---  while t[index] do
---    index = index + 1
---  end
---
---  t[index] = function(slot)
---    print(slot)
---  end
---  token.set_lua('mycode', index, 'protected', 'global')
---}
---
---\mycode
---
---\bye
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L1168-L1216](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L1168-L1216)
---
---@param name string
---@param id integer
---@param ... 'global'|'protected'
---
---@see lua.get_functions_table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.set_lua(name, id, ...) end

---
---Put the next token back in the input.
---
---__Example:__
---
---```lua
---local t1 = token.get_next()
---local t2 = token.get_next()
---local t3 = token.get_next()
---local t4 = token.get_next()
----- watch out, we flush in sequence
---token.put_next { t1, t2 }
----- but this one gets pushed in front
---token.put_next ( t3, t4 )
----- so when we get wxyz we put yzwx!
---```
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L2422-L2433](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L2422-L2433)
---* Corresponding C source code: [lnewtokenlib.c#L262-L337](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L262-L337)
---
---@param ... Token
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.put_next(...) end

---
---Check if the given argument is a token.
---
---__Example:__
---
---```tex
---\directlua{
---  local t = token.get_next()
---  print(token.is_token(t)) % true
---  print(token.is_token('text')) % false
---  print(token.is_token(true)) % false
---}Token
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L758-L762](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L758-L762)
---
---@param t any
---
---@return boolean
---
---@see token.type
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.is_token(t) end

---
---Return the string `token` if the given parameter is a token else `nil`.
---
---__Example:__
---
---```tex
---\directlua{
---  local t = token.get_next()
---  print(token.type(t)) % 'token'
---  print(token.type('text')) % nil
---  print(token.type(true)) % nil
---}Token
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnewtokenlib.c#L1080-L1088](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnewtokenlib.c#L1080-L1088)
---
---@param t any
---
---@return 'token'|nil
---
---@see token.is_token
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/token.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function token.type(t) end

---
---https://github.com/TeX-Live/luatex/tree/master/source/texk/web2c/luatexdir/luasocket/src/socket.lua
---Changes to upstream: global socket table

---
---The definitions are developed in this repository: https://github.com/LuaCATS/luasocket

---
---https://lunarmodules.github.io/luasocket/socket.html
---
---The socket namespace
---
---The `socket` namespace contains the core functionality of LuaSocket.
---
---To obtain the `socket` namespace, run:
---
---```lua
----- loads the socket module
---local socket = require("socket")
---```
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
---@class Socket
socket = {}

---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
socket.headers = {}

---
---The `socket.headers.canonic` table
---is used by the HTTP and SMTP modules to translate from
---lowercase field names back into their canonic
---capitalization. When a lowercase field name exists as a key
---in this table, the associated value is substituted in
---whenever the field name is sent out.
---```
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
socket.headers.canonic = {}

---
---This constant is set to `true` if the library was compiled
---with debug support.
---
---@type boolean
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
socket._DEBUG = true

---
---Default datagram size used by calls to
---`receive`and
---`receivefrom`.
---(Unless changed in compile time, the value is 8192.)
---
---@type integer
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
socket._DATAGRAMSIZE = 8192

---
---Return the UNIX time in seconds.
---
---You should subtract the values returned by this function
---to get meaningful values.
---
---__Example:__
---
---```lua
---t = socket.gettime()
----- do stuff
---print(socket.gettime() - t .. " seconds elapsed")
---```
---
---__Reference:__
---
---* Corresponding C source code: [timeout.c#L190-L194](https://github.com/lunarmodules/luasocket/blob/8c2ff7217e2a205eb107a6f48b04ff1b2b3090a1/src/timeout.c#L190-L194)
---
---@return number time # for example `1683526723.1653`
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.gettime() end

---
---Creates and returns a clean
---`try`
---function that allows for cleanup before the exception
---is  raised.
---
---Note: This idea saved a lotof work with the
---implementation of protocols in LuaSocket:
---
---```lua
---foo = socket.protect(function()
---    -- connect somewhere
---    local c = socket.try(socket.connect("somewhere", 42))
---    -- create a try function that closes 'c' on error
---    local try = socket.newtry(function() c:close() end)
---    -- do everything reassured c will be closed
---    try(c:send("hello there?\r\n"))
---    local answer = try(c:receive())
---    ...
---    try(c:send("good bye\r\n"))
---    c:close()
---end)
---```
---
---@param finalizer function # -`Finalizer` is a function that will be called before `try` throws the exception.
---
---@return function # The function returns your customized `try` function.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.newtry(finalizer) end

---
---Converts a function that throws exceptions into a safe function. This
---function only catches exceptions thrown by the `try`
---and `newtry`functions. It does not catch normal
---Lua errors.
---
---@param func function # `func` is a function that calls `try` (or `assert`, or `error`) to throw exceptions.
---
---@return function # Returns an equivalent function that instead of throwing exceptions in case of a failed `try`call, returns `nil` followed by an error message.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.protect(func) end

---
---Waits for a number of sockets to change status.
---
--- `Recvt` and `sendt` can also
---be empty tables or `nil`. Non-socket values (or values with
---non-numeric indices) in the arrays will be silently ignored.
---
---The function returns a list with the sockets ready for
---reading, a list with the sockets ready for writing and an error message.
---The error message is "`timeout`" if a timeout
---condition was met, "`select failed`" if the call
---to `select` failed, and
---`nil` otherwise. The returned tables are
---doubly keyed both by integers and also by the sockets
---themselves, to simplify the test if a specific socket has
---changed status.
---
---Note:`select` can monitor a limited number
---of sockets, as defined by the constant
---`socket._SETSIZE`. This
---number may be as high as 1024 or as low as 64 by default,
---depending on the system. It is usually possible to change this
---at compile time. Invoking `select` with a larger
---number of sockets will raise an error.
---
---Important note: a known bug in WinSock causes `select` to fail
---on non-blocking TCP sockets. The function may return a socket as
---writable even though the socket is notready for sending.
---
---Another important note: calling select with a server socket in the receive parameter before a call to accept does notguarantee
---`accept`will return immediately.
---Use the `settimeout`
---method or `accept` might block forever.
---
---Yet another note: If you close a socket and pass
---it to `select`, it will be ignored.
---
---Using select with non-socket objects: Any object that implements `getfd` and `dirty` can be used with `select`, allowing objects from other libraries to be used within a `socket.select` driven loop.
---
---@param recvt table|nil # `recvt` is an array with the sockets to test for characters available for reading.
---@param sendt table|nil # Sockets in the `sendt` array are watched to see if it is OK to immediately write on them.
---@param timeout? integer # `timeout` is the
---maximum amount of time (in seconds) to wait for a change in status.  A
---`nil`, negative or omitted `timeout` value allows the
---function to block indefinitely.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.select(recvt, sendt, timeout) end

---
---The maximum number of sockets that the select function can handle.
---
---@type integer
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
socket._SETSIZE = 0

---
---Creates an
---LTN12
---sink from a stream socket object.
---
---`Mode` defines the behavior of the sink. The following
---options are available:
---
---* `"http-chunked"`: sends data through socket after applying the
---chunked transfer coding, closing the socket when done;
---* `"close-when-done"`: sends all received data through the
---socket, closing the socket when done;
---* `"keep-open"`: sends all received data through the
---socket, leaving it open when done.
---
---`Socket` is the stream socket object used to send the data.
---
---The function returns a sink with the appropriate behavior.
---
---@param mode 'http-chunked'|'close-when-done'|'keep-open'
---@param socket any
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.sink(mode, socket) end

---
---Drops a number of arguments and returns the remaining.
---
---`D` is the number of arguments to drop. `Ret1` to
---`retN` are the arguments.
---
---The function returns `retd+1` to `retN`.
---
---Note: This function is useful to avoid creation of dummy variables:
---
---```
----- get the status code and separator from SMTP server reply
---local code, sep = socket.skip(2, string.find(line, "^(%d%d%d)(.?)"))
---```
---
---@param d integer
---@param ... any
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.skip(d, ...) end

---
---Freeze the program execution during a given amount of time.
---
---__Reference:__
---
---* Corresponding C source code: [timeout.c#L196-L226](https://github.com/lunarmodules/luasocket/blob/8c2ff7217e2a205eb107a6f48b04ff1b2b3090a1/src/timeout.c#L196-L226)
---
---@param time integer # `time` is the number of seconds to sleep for. If `time` is negative, the function returns immediately.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.sleep(time) end

---
---Creates an
---LTN12
---source from a stream socket object.
---
---`Mode` defines the behavior of the source. The following
---options are available:
---
---* `"http-chunked"`: receives data from socket and removes the
---chunked transfer codingbefore returning the data;
---* `"by-length"`: receives a fixed number of bytes from the
---socket. This mode requires the extra argument `length`;
---* `"until-closed"`: receives data from a socket until the other
---side closes the connection.
---
---`Socket` is the stream socket object used to receive the data.
---
---The function returns a source with the appropriate behavior.
---
---@param mode 'http-chunked'|'by-length'|'until-closed'
---@param socket any
---@param timeout integer
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.source(mode, socket, timeout) end

---
---The OS value for an invalid socket. This can be used with
---`tcp:getfd`and `tcp:setfd`methods.
---@type any
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
socket._SOCKETINVALID = nil

---
---Throws an exception in case `ret1` is falsy, using
---`ret2` as the error message. The exception is supposed to be caught
---by a `protect`ed function only.
---
---`Ret1` to `retN` can be arbitrary
---arguments, but are usually the return values of a function call
---nested with `try`.
---
---The function returns `ret`1 to `ret`N if
---`ret`1 is not `nil` or `false`.
---Otherwise, it calls `error` passing `ret`2 wrapped
---in a table with metatable used by `protect`to
---distinguish exceptions from runtime errors.
---
---```lua
----- connects or throws an exception with the appropriate error message
---c = socket.try(socket.connect("localhost", 80))
---```
---
---@param ... any
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.try(...) end

---
---This constant has a string describing the current LuaSocket version.
---
---@type string
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
socket._VERSION = ""

---
---{{{ Common Types
--#region Common Types

---
---A string representing a datagram
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
---@alias Datagram string

--> Note: In some functions, there could be more results than just a 1,
-->       but the documentation has no information as to what each of them are.
-->       Check: https://github.com/lunarmodules/luasocket/blob/master/src/udp.c#L253
-->       Sometimes its 1 and 2, others up to 3.

---
---A 1 in case of success
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
---@alias SocketReturnResult 1?

---
---Error message in case of failure
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
---@alias SocketReturnError string?

---
---A string with the family ("inet" or "inet6")
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
---@alias SocketFamily
---| "inet" IPv4
---| "inet6" IPv6
---| "unspec" Unspecified
---| "unknown" Unknown

--#endregion Common Types
---}}}

--{{{ TCP
--#region TCP

---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
---@class TCPSocketMaster
local tcp_master = {}

---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
---@class TCPSocketServer : TCPSocketMaster
local tcp_server = {}

---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
---@class TCPSocketClient : TCPSocketMaster
local tcp_client = {}

---
---Waits for a remote connection on the server object and returns a client object representing that connection.
---
---If a connection is successfully initiated, a client object is returned. If a timeout condition is met, the method returns `nil` followed by the error string 'timeout'. Other errors are reported by `nil` followed by a message describing the error.
---
---Note: calling `socket.select` with a server object in the recvt parameter before a call to accept does not guarantee accept will return immediately. Use the `settimeout` method or accept might block until another client shows up.
---
---@return TCPSocketClient | nil, "timeout" | string
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_server:accept() end

---
---Binds a master object to address and port on the local host.
---
---Address can be an IP address or a host name. Port must be an integer number in the range [0..64K). If address is `'*'`, the system binds to all local interfaces using the `INADDR_ANY` constant or `IN6ADDR_ANY_INIT`, according to the family. If `port` is `0`, the system automatically chooses an ephemeral port.
---
---In case of success, the method returns `1`. In case of error, the method returns `nil` followed by an error message.
---
---On success, the type changes to `TCPSocketServer`, and you should cast it as such.
---
---Note: The function `socket.bind` is available and is a shortcut for the creation of server sockets.
---
---@param address string
---@param port integer
---
---@return 1|nil, nil|string
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_master:bind(address, port) end

---
---Closes a TCP object.
---The internal socket used by the object is closed and the local address to which the object was bound is made available to other applications.
---No further operations (except for further calls to the close method) are allowed on a closed socket.
---
---Note: It is important to close all used sockets once they are not needed, since, in many systems,
---each socket uses a file descriptor, which are limited system resources.
---Garbage-collected objects are automatically closed before destruction, though.
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_master:close() end

---
---Attempts to connect a master object to a remote host, transforming it into a client object.
---Client objects support methods `send`, `receive`, `getsockname`, `getpeername`, `settimeout`, and `close`.
---
---On success, the type changes to `TCPClient`, and you should cast it as such.
---
---Note: The function `socket.connect` is available and is a shortcut for the creation of client sockets.
---Note: Starting with LuaSocket 2.0, the `settimeout` method affects the behavior of connect, causing it to return with an error in case of a timeout. If that happens, you can still call socket.select with the socket in the sendt table. The socket will be writable when the connection is established.
---Note: Starting with LuaSocket 3.0, the host name resolution depends on whether the socket was created by socket.tcp, socket.tcp4 or socket.tcp6. Addresses from the appropriate family (or both) are tried in the order returned by the resolver until the first success or until the last failure. If the timeout was set to zero, only the first address is tried.
---
---@param address string # IP address or a host name
---@param port integer # TCP port, in the range [1..64K)
---
---@return nil | 1, nil | string # In case of error, the method returns nil followed by a string describing the error. In case of success, the method returns 1.
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_master:connect(address, port) end

---
---Check the read buffer status.
---
---**Note: This is an internal method, use at your own risk.**
---
---@return boolean # `true` if there is any data in the read buffer, `false` otherwise.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_master:dirty() end

---
---Get the underling socket descriptor or handle associated to the object.
---
---**Note: This is an internal method. Unlikely to be portable. use at your own risk.**
---
---@return integer # The descriptor or handle. `-1` if the object has been closed. `_SOCKETINVALID` if it is an invalid socket.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_master:getfd() end

---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
---@alias TCPOption
---| "keepalive" Setting this option to `true` enables the periodic transmission of messages on a connected socket. Should the connected party fail to respond to these messages, the connection is considered broken and processes using the socket are notified;
---| "linger" Controls the action taken when unsent data are queued on a socket and a close is performed. The value is a table with a boolean entry `on` and a numeric entry for the time interval `timeout` in seconds. If the `on` field is set to `true`, the system will block the process on the close attempt until it is able to transmit the data or until `timeout` has passed. If `on` is `false` and a `close` is issued, the system will process the `close` in a manner that allows the process to continue as quickly as possible. I do not advise you to set this to anything other than zero;
---| "reuseaddr" Setting this option indicates that the rules used in validating addresses supplied in a call to bind should allow reuse of local addresses;
---| "tcp-nodelay" Setting this option to `true` disables the Nagle's algorithm for the connection;
---| "tcp-keepidle" value in seconds for `TCP_KEEPIDLE` **Linux only!!**
---| "tcp-keepcnt" value for `TCP_KEEPCNT` **Linux only!!**
---| "tcp-keepintvl" value for `TCP_KEEPINTVL` **Linux only!!**
---| "tcp-defer-accept" value for `TCP_DEFER_ACCEPT` **Linux only!!**
---| "tcp-fastopen" value for `TCP_FASTOPEN` **Linux only!!**
---| "tcp-fastopen-connect" value for `TCP_FASTOPEN_CONNECT` **Linux only!!**
---| "ipv6-v6only" Setting this option to `true` restricts an inet6 socket to sending and receiving only IPv6 packets.
--- The descriptions above come from the man pages.

---
---Gets options for the TCP object. See `setoption` for description of the option names and values.
---@param option TCPOption
---
---@return any | nil, nil | string # The option value in case of success, or nil followed by an error message otherwise.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_client:getoption(option) end

---
---Gets options for the TCP object. See `setoption` for description of the option names and values.
---@param option TCPOption
---
---@return any | nil, nil | string # The option value in case of success, or nil followed by an error message otherwise.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_server:getoption(option) end

---
---Returns information about the remote side of a connected client object.
---
---Note: It makes no sense to call this method on server objects.
---
---@return string | nil, integer | nil, SocketFamily | nil # The IP address of the peer, the port number that the peer is using for the connection, and the family. In case of error, returns `nil`.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_client:getpeername() end

---
---Returns the local address information associated to the object.
---
---@return string | nil, integer | nil, SocketFamily | nil # The IP address of the peer, the port number that the peer is using for the connection, and the family. In case of error, returns `nil`.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_master:getsockname() end

---
---Returns the current block timeout followed by the current total timeout.
---
---@return number, number # Current block timeout, current total timeout.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_master:gettimeout() end

---
---Specifies the socket is willing to receive connections, transforming the object into a server object. Server objects support the `accept`, `getsockname`, `setoption`, `settimeout`, and `close` methods.
---
---On success, the type changes to `TCPSocketServer`, and you should cast it as such.
---
---@param backlog integer The number number of client connections that can be queued waiting for service. If the queue is full and another client attempts connection, the connection is refused.
---
---@return 1 | nil, nil | string # Returns 1 on success, nil and an error on failure.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_master:listen(backlog) end

---
---@alias SocketError
---| "timeout" If there was a timeout during the operation.
---| "closed" If the connection was closed before the transmission was completed.

-- HACK: For some reason number doesn't show up in the completion if it's
--       defined as part of this union. It should be defined in-place.

---
---@alias ReceivePatternMode
---| '*a' # Reads from the socket until the connection is closed. No end-of-line translation is performed
---| '*l' # Reads a line of text from the socket. The line is terminated by a LF character (ASCII 10), optionally preceded by a CR character (ASCII 13). The CR and LF characters are not included in the returned line. In fact, all CR characters are ignored by the pattern.

---
---Reads data from a client object, according to the specified read pattern. Patterns follow the Lua file I/O format, and the difference in performance between all patterns is negligible.
---
---**Important note:** This function was changed severely. It used to support multiple patterns (but I have never seen this feature used) and now it doesn't anymore. Partial results used to be returned in the same way as successful results. This last feature violated the idea that all functions should return nil on error. Thus it was changed too.
---
---@param pattern? ReceivePatternMode | number The default is "*l"
---@param prefix? string Optional string to be concatenated to the beginning of any received data before return.
---
---@return string | nil, SocketError | string | nil # Returns the received pattern when successful, otherwise nil and an error message.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_client:receive(pattern, prefix) end

---
---Sends data through client object.
---
---The optional arguments `i` and `j` work exactly like the standard `string.sub` Lua function to allow the selection of a substring to be sent.
---
---**Note:** Output is not buffered. For small strings, it is always better to concatenate them in Lua (with the '..' operator) and send the result in one call instead of calling the method several times.
---
---@param data string The string to be sent.
---@param i? integer
---@param j? integer
---
---@return integer | nil, SocketError | string | nil, integer | nil # On success the number of bytes sent, otherwise nil followed by an error message, followed by the index of the last byte within `[i, j]` that has been sent. You might want to try again from the byte following that.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_client:send(data, i, j) end

---
---Sets options for the TCP object. Options are only needed by low-level or time-critical applications. You should only modify an option if you are sure you need it.
---
---@param option TCPOption A string with the option name
---@param value? any Depends on the option being set
---@return 1 | nil, string | nil # 1 on success, nil and an error message otherwise.
function tcp_client:setoption(option, value) end

---
---Sets options for the TCP object. Options are only needed by low-level or time-critical applications. You should only modify an option if you are sure you need it.
---
---@param option TCPOption A string with the option name
---@param value? any Depends on the option being set
---
---@return 1 | nil, string | nil # 1 on success, nil and an error message otherwise.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_server:setoption(option, value) end

---
---Resets accounting information on the socket, useful for throttling of bandwidth.
---
---@param received number Bytes received
---@param sent number Byte sent
---@param age number Age in seconds
---@return 1 | nil # 1 on success, nil otherwise.
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_master:setstats(received, sent, age) end

---
---@alias TCPTimeoutMode
---| "b" block timeout. Specifies the upper limit on the amount of time LuaSocket can be blocked by the operating system while waiting for completion of any single I/O operation.
---| "t" total timeout. Specifies the upper limit on the amount of time LuaSocket can block a Lua script before returning from a call.

---
---Changes the timeout values for the object. By default, all I/O operations are blocking. That is, any call to the methods `send`, `receive`, and `accept` will block indefinitely, until the operation completes. The `settimeout` method defines a limit on the amount of time the I/O methods can block. When a timeout is set and the specified amount of time has elapsed, the affected methods give up and fail with an error code.
---
---**Note:** although timeout values have millisecond precision in LuaSocket, large blocks can cause I/O functions not to respect timeout values due to the time the library takes to transfer blocks to and from the OS and to and from the Lua interpreter. Also, function that accept host names and perform automatic name resolution might be blocked by the resolver for longer than the specified timeout value.
---
---**Note:** The old timeout method is deprecated. The name has been changed for sake of uniformity, since all other method names already contained verbs making their imperative nature obvious.
---
---@param value number | nil Time to wait, in seconds. Use `nil` or negative to block indefinitely.
---@param mode? TCPTimeoutMode The default mode is "b"
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_master:settimeout(value, mode) end

---
---@alias TCPShutdownMode
---| "both" Disallow further sends and receives on the object.
---| "send" Disallow further sends on the object.
---| "receive" Disallow further receives on the object.

---
---Shuts down part of a full-duplex connection.
---@param mode TCPShutdownMode Determines which way of the connection should be shut down.
---
---@return 1 # Always returns 1.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_client:shutdown(mode) end

---
---Sets the underling socket descriptor or handle associated to the object. The current one is simply replaced, not closed, and no other change to the object state is made. To set it as invalid use `_SOCKETINVALID`.
---
---**Note: This is an internal method. Unlikely to be portable. Use at your own risk.**
---
---@param fd integer
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function tcp_master:setfd(fd) end

-- The following are in the socket namespace --

---
---This function is a shortcut that creates and returns a TCP server object bound to a local `address` and `port`, ready to accept client connections. Optionally, user can also specify the `backlog` argument to the `listen` method.
---
---Note: The server object returned will have the option `reuseaddr` set to `true`.
---
---@param address string
---@param port integer
---@param backlog? integer Defaults to 32.
---
---@return TCPSocketServer
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.bind(address, port, backlog) end

---
---This function is a shortcut that creates and returns a TCP client object connected to a remote `address` at a given `port`. Optionally, the user can also specify the local address and port to bind (`locaddr` and `locport`), or restrict the socket family to "`inet`" or "`inet6`".
---
---Two variations of connect are defined as simple helper functions that restrict the `family`, `socket.connect4` and `socket.connect6`.
---
---@param address string
---@param port integer
---@param locaddr? string The local address
---@param locport? integer The local port
---@param family? 'inet'|'inet6' If not specified, the family depends on your system configuration.
---
---@return TCPSocketClient | nil, nil | string # Returns the client on success, or `nil` and an error message on failure.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.connect(address, port, locaddr, locport, family) end

---
---This function is a shortcut that creates and returns an IPv4 TCP client object connected to a remote `address` at a given `port`. Optionally, the user can also specify the local address and port to bind (`locaddr` and `locport`)
---
---@param address string
---@param port integer
---@param locaddr? string The local address
---@param locport? integer The local port
---
---@return TCPSocketClient | nil, nil | string # Returns the client on success, or `nil` and an error message on failure.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.connect4(address, port, locaddr, locport) end

---
---This function is a shortcut that creates and returns an IPv6 TCP client object connected to a remote `address` at a given `port`. Optionally, the user can also specify the local address and port to bind (`locaddr` and `locport`)
---
---@param address string
---@param port integer
---@param locaddr? string The local address
---@param locport? integer The local port
---
---@return TCPSocketClient | nil, nil | string # Returns the client on success, or `nil` and an error message on failure.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.connect6(address, port, locaddr, locport) end

---
---Creates and returns an TCP master object. A master object can be transformed into a server object with the method `listen` (after a call to `bind`) or into a client object with the method `connect`. The only other method supported by a master object is the close method.
---
---Note: The choice between IPv4 and IPv6 happens during a call to `bind` or `connect`, depending on the address family obtained from the resolver.
---
---Note: Before the choice between IPv4 and IPv6 happens, the internal socket object is invalid and therefore `setoption` will fail.
---
---@return TCPSocketMaster | nil, nil | string # New master object if successful, otherwise `nil` followed by an error message.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.tcp() end

---
---Creates and returns an IPv4 TCP master object. A master object can be transformed into a server object with the method `listen` (after a call to `bind`) or into a client object with the method connect. The only other method supported by a master object is the `close` method.
---
---@return TCPSocketMaster | nil, nil | string # New master object if successful, otherwise `nil` followed by an error message.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.tcp4() end

---
---Creates and returns an IPv6 TCP master object. A master object can be transformed into a server object with the method `listen` (after a call to `bind`) or into a client object with the method `connect`. The only other method supported by a master object is the `close` method.
---
---Note: The TCP object returned will have the option "ipv6-v6only" set to `true`.
---
---@return TCPSocketMaster | nil, nil | string # New master object if successful, otherwise `nil` followed by an error message.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.tcp6() end

--#endregion
--}}}

--{{{ UDP
--#region UDP

--> Generic is my way to avoid stating the same thing twice.
--> Means that both connected and unconnected have it.

---
---@class UDPSocketGeneric
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
local udp_generic = {}

---
---@class UDPSocketConnected : UDPSocketGeneric
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
local udp_connected = {}

---
---@class UDPSocketUnconnected : UDPSocketGeneric
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
local udp_unconnected = {}

---
---Closes a UDP object. The internal socket used by the object is closed and the local address to which the object was bound is made available to other applications. No further operations (except for further calls to the close method) are allowed on a closed socket.
---
---Note: It is important to close all used sockets once they are not needed, since, in many systems, each socket uses a file descriptor, which are limited system resources. Garbage-collected objects are automatically closed before destruction, though.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_generic:close() end

---
---The option descriptions come from the official documentation, which come from the man pages.
---@alias UDPOption
---| 'dontroute': Indicates that outgoing messages should bypass the standard routing facilities. Receives a boolean value;
---| 'broadcast': Requests permission to send broadcast datagrams on the socket. Receives a boolean value;
---| 'reuseaddr': Indicates that the rules used in validating addresses supplied in a bind() call should allow reuse of local addresses. Receives a boolean value;
---| 'reuseport': Allows completely duplicate bindings by multiple processes if they all set 'reuseport' before binding the port. Receives a boolean value;
---| 'ip-multicast-loop': Specifies whether or not a copy of an outgoing multicast datagram is delivered to the sending host as long as it is a member of the multicast group. Receives a boolean value;
---| 'ipv6-v6only': Specifies whether to restrict inet6 sockets to sending and receiving only IPv6 packets. Receive a boolean value;
---| 'ip-multicast-if': Sets the interface over which outgoing multicast datagrams are sent. Receives an IP address;
---| 'ip-multicast-ttl': Sets the Time To Live in the IP header for outgoing multicast datagrams. Receives a number;
---| 'ip-add-membership': Joins the multicast group specified. Receives a table with fields multiaddr and interface, each containing an IP address;
---| 'ip-drop-membership': Leaves the multicast group specified. Receives a table with fields multiaddr and interface, each containing an IP address.

---
---Gets an option value from the UDP object.
---
---Option is a string with the option name.
---
---The method returns the option value in case of success, or nil followed by an error message otherwise.
---
---@param option UDPOption
---
---@return any?, SocketReturnError
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_generic:getoption(option) end

--> Note: This is not very satisfying, since one could check
-->       if the first argument is nil. But LuaLS will want to
-->       check the rest of the arguments anyway.

---
---Retrieves information about the peer associated with a connected UDP object.
---
---Returns a string with the IP address of the peer, the port number that peer is using for the connection, and a string with the family ("inet" or "inet6"). In case of error, the method returns nil.
---
---Note: It makes no sense to call this method on unconnected objects.
---
---@return string?, number?, SocketFamily?
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_connected:getpeername() end

---
---Returns the local address information associated to the object.
---
---The method returns a string with local IP address, a number with the local port, and a string with the family ("inet" or "inet6"). In case of error, the method returns nil.
---
---Note: UDP sockets are not bound to any address until the `setsockname` or the `sendto` method is called for the first time (in which case it is bound to an ephemeral port and the wild-card address).
---
---@return string?, number?, SocketFamily?
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_generic:getsockname() end

---
---Returns the current timeout value.
---
---@return number timeout
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_generic:gettimeout() end

---
---Receives a datagram from the UDP object. If the UDP object is connected, only datagrams coming from the peer are accepted. Otherwise, the returned datagram can come from any host.
---
---The optional size parameter specifies the maximum size of the datagram to be retrieved. If there are more than size bytes available in the datagram, the excess bytes are **discarded**.
---If there are less then size bytes available in the current datagram, the available bytes are returned. If size is omitted, the compile-time constant `socket._DATAGRAMSIZE` is used
---(it defaults to 8192 bytes). Larger sizes will cause a temporary buffer to be allocated for the operation.
---
---In case of success, the method returns the received datagram. In case of timeout, the method returns nil followed by the string `'timeout'`.
---
---Note: don't call this from unconnected sockets. Since you wont be able to know where it came from. Use `receivefrom` instead.
---
---@param size number?
---
---@return Datagram? datagram
---
---@return 'timeout'? err # `'timeout'` in case of timeout
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_generic:receive(size) end

---
---Works exactly as the `receive` method, except it returns the IP address and port as extra return values (and is therefore slightly less efficient)
---@param size number?
---
---@return Datagram? datagram
---@return string | 'timeout' ip_or_err # IP address or `'timeout'` error string
---@return number port
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_unconnected:receivefrom(size) end

---
---Sends a datagram to the UDP peer of a connected object.
---
---Datagram is a string with the datagram contents. The maximum datagram size for UDP is 64K minus IP layer overhead. However datagrams larger than the link layer packet size will be fragmented, which may deteriorate performance and/or reliability.
---
---If successful, the method returns `1`. In case of error, the method returns `nil` followed by an `error message`.
---
---Note: In UDP, the send method never blocks and the only way it can fail is if the underlying transport layer refuses to send a message to the specified address (i.e. no interface accepts the address).
---@param datagram Datagram
---
---@return SocketReturnResult, SocketReturnError
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_connected:send(datagram) end

---
---Sends a datagram to the specified IP address and port number.
---
---Datagram is a string with the datagram contents. The maximum datagram size for UDP is 64K minus IP layer overhead. However datagrams larger than the link layer packet size will be fragmented, which may deteriorate performance and/or reliability. Ip is the IP address of the recipient. Host names are not allowed for performance reasons. Port is the port number at the recipient.
---
---If successful, the method returns 1. In case of error, the method returns nil followed by an error message.
---
---Note: In UDP, the send method never blocks and the only way it can fail is if the underlying transport layer refuses to send a message to the specified address (i.e. no interface accepts the address).
---
---@param datagram Datagram
---@param ip string
---@param port number
---
---@return SocketReturnResult, SocketReturnError
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_unconnected:sendto(datagram, ip, port) end

---
---Sets options for the UDP object. Options are only needed by low-level or time-critical applications. You should only modify an option if you are sure you need it.
---
---Option is a string with the option name, and value depends on the option being set
---
---The method returns 1 in case of success, or nil followed by an error message otherwise.
---
---@param option UDPOption
---@param value any?
---
---@return SocketReturnResult, SocketReturnError
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_generic:setoption(option, value) end

--> `setpeername` sets the peer for an unconnected socket
--> when it's connected, only a `'*'` can be set. Which will make it unconnected again

---
---Changes the peer of a UDP object. This method turns an unconnected UDP object into a connected UDP object or vice versa.
---
---For connected objects, outgoing datagrams will be sent to the specified peer, and datagrams received from other peers will be discarded by the OS. Connected UDP objects must use the send and receive methods instead of sendto and receivefrom.
---
---Address can be an IP address or a host name. Port is the port number. If address is '*' and the object is connected, the peer association is removed and the object becomes an unconnected object again. In that case, the port argument is ignored.
---
---In case of error the method returns nil followed by an error message. In case of success, the method returns 1.
---
---Note: Since the address of the peer does not have to be passed to and from the OS, the use of connected UDP objects is recommended when the same peer is used for several transmissions and can result in up to 30% performance gains.
---
---Note: Starting with LuaSocket 3.0, the host name resolution depends on whether the socket was created by socket.udp or socket.udp6. Addresses from the appropriate family are tried in succession until the first success or until the last failure.
---
---@param address string # can be a host name
---@param port number
---
---@return SocketReturnResult, SocketReturnError
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_generic:setpeername(address, port) end

---
---Changes the peer of a UDP object. This method turns an unconnected UDP object into a connected UDP object or vice versa.
---
---For connected objects, outgoing datagrams will be sent to the specified peer, and datagrams received from other peers will be discarded by the OS. Connected UDP objects must use the send and receive methods instead of sendto and receivefrom.
---
---Address can be an IP address or a host name. Port is the port number. If address is '*' and the object is connected, the peer association is removed and the object becomes an unconnected object again. In that case, the port argument is ignored.
---
---In case of error the method returns nil followed by an error message. In case of success, the method returns 1.
---
---Note: Since the address of the peer does not have to be passed to and from the OS, the use of connected UDP objects is recommended when the same peer is used for several transmissions and can result in up to 30% performance gains.
---
---Note: Starting with LuaSocket 3.0, the host name resolution depends on whether the socket was created by socket.udp or socket.udp6. Addresses from the appropriate family are tried in succession until the first success or until the last failure.
---
---@param address "*" # will turn it unconnected
---
---@return SocketReturnResult, SocketReturnError
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_generic:setpeername(address) end

---
---Binds the UDP object to a local address.
---
---Address can be an IP address or a host name. If address is '*' the system binds to all local interfaces using the constant INADDR_ANY. If port is 0, the system chooses an ephemeral port.
---
---If successful, the method returns 1. In case of error, the method returns nil followed by an error message.
---
---Note: This method can only be called before any datagram is sent through the UDP object, and only once. Otherwise, the system automatically binds the object to all local interfaces and chooses an ephemeral port as soon as the first datagram is sent. After the local address is set, either automatically by the system or explicitly by setsockname, it cannot be changed.
---
---@return SocketReturnResult, SocketReturnError
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_generic:setsockname(address, port) end

---
---Sets the timeout value for the socket
---
---Sets the time to blocks the process when calling `receive` and `receivefrom`
---until the timeout is reached or a packet arrives.
---
---@param timeout number
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function udp_generic:settimeout(timeout) end

---
---Creates and returns an unconnected UDP object. Unconnected objects support the sendto, receive, receivefrom, getoption, getsockname, setoption, settimeout, setpeername, setsockname, and close. The setpeername is used to connect the object.
---
---In case of success, a new unconnected UDP object returned. In case of error, nil is returned, followed by an error message.
---
---Note: The choice between IPv4 and IPv6 happens during a call to sendto, setpeername, or sockname, depending on the address family obtained from the resolver.
---
---Note: Before the choice between IPv4 and IPv6 happens, the internal socket object is invalid and therefore setoption will fail.
---
---### To mark a socket as connected or unconnected use `---@cast`. Example:
---```lua
---local server, err = socket.udp()
---local client, err = socket.udp()
------@cast server UDPSocketUnconnected
---server:setsockname("127.0.0.1", 12345)
------@cast client UDPSocketConnected
---client:setpeername("127.0.0.1", 12345)
---```
---
---@return (UDPSocketConnected | UDPSocketUnconnected)?, SocketReturnError
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.udp() end

---
---Creates and returns an unconnected IPv4 UDP object. Unconnected objects support the sendto, receive, receivefrom, getoption, getsockname, setoption, settimeout, setpeername, setsockname, and close. The setpeername is used to connect the object.
---
---In case of success, a new unconnected UDP object returned. In case of error, nil is returned, followed by an error message.
---
---### To mark a socket as connected or unconnected use `---@cast`. Example:
---```lua
---local server, err = socket.udp4()
---local client, err = socket.udp4()
------@cast server UDPSocketUnconnected
---server:setsockname("127.0.0.1", 12345)
------@cast client UDPSocketConnected
---client:setpeername("127.0.0.1", 12345)
---```
---
---@return (UDPSocketConnected | UDPSocketUnconnected)?, SocketReturnError
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.udp4() end

---
---Creates and returns an unconnected IPv6 UDP object. Unconnected objects support the sendto,
---
---In case of success, a new unconnected UDP object returned. In case of error, nil is returned, followed by an error message.
---
---Note: The UDP object returned will have the option "ipv6-v6only" set to true.
---
---### To mark a socket as connected or unconnected use `---@cast`. Example:
---```lua
---local server, err = socket.udp6()
---local client, err = socket.udp6()
------@cast server UDPSocketUnconnected
---server:setsockname("127.0.0.1", 12345)
------@cast client UDPSocketConnected
---client:setpeername("127.0.0.1", 12345)
---```
---
---@return (UDPSocketConnected | UDPSocketUnconnected)?, SocketReturnError
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/socket.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function socket.udp6() end

--#endregion
--}}}

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex = {}

---
---The integer parameters accept and return Lua numbers. These are read-write:
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L284-350](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L284-350)
---@alias ReadWriteIntegerParameter
---| 'adjdemerits' # Penalty for adjacent visually incompatible lines. Default: `10000`.
---| 'binoppenalty' # Penalty for breaking after a binary operator not enclosed in a subformula. Default: `700`.
---| 'brokenpenalty' # Additional penalty for breaking a page after a hyphenated line. Default: `100`
---| 'catcodetable' # Catcode tables are a new feature that allows you to switch to a predefined catcode regime in a single statement.
---| 'clubpenalty' # Extra penalty for breaking a page after the first line of a paragraph. In Plain TeX this is `150`. This amount, and the following penalties, are added to the `\interlinepenalty`, and a penalty of the resulting size is inserted after the `\hbox` containing the first line of a paragraph instead of the `\interlinepenalty`.
---| 'day' # The day of the current job.
---| 'defaulthyphenchar' # Value of `\hyphenchar` when a font is loaded. Default: `‘\-`.
---| 'defaultskewchar' # Value of `\skewchar` when a font is loaded.
---| 'delimiterfactor' # `1000` times the fraction of a delimited formula that should be covered by a delimiter. Default: `901`.
---| 'displaywidowpenalty' # Extra penalty for breaking a page before the last line above a display formula. The default value in Plain TeX is `50`.
---| 'doublehyphendemerits' # Penalty for consecutive lines ending with a hyphen. Default: `10000`.
---| 'endlinechar' # The character code of the end-of-line character appended to input lines. IniTEX default: `13`.
---| 'errorcontextlines' # (TeX3 only) Number of additional context lines shown in error messages.
---| 'escapechar' # Number of the character that is to be used for the escape character when control sequences are being converted into character tokens. IniTEX default: `92` (`\`).
---| 'exhyphenpenalty' # Penalty for breaking a horizontal line at a discretionary item in the special case where the prebreak text is empty. Default: `50`.
---| 'fam' # The number of the current font family.
---| 'finalhyphendemerits' # Penalty added when the penultimate line of a paragraph ends with a hyphen. Default: `5000`.
---| 'floatingpenalty' # Penalty added when an insertion is split.
---| 'globaldefs' # Parameter for overriding \global prefixes. IniTEX default: `0`.
---| 'hangafter' # If positive, this denotes the number of lines before indenting starts; if negative, the absolute value of this is the number of indented lines starting with the first line of the paragraph. Default: `1`.
---| 'hbadness' # Amount of tolerance before TEX reports an underfull or overfull horizontal box.
---| 'holdinginserts' # (TeX3 only) If this is positive, insertions are not placed in their boxes at output time.
---| 'hyphenpenalty' # Penalty associated with break at a discretionary item in the general case. Default: `50`.
---| 'interlinepenalty' # Penalty for breaking a page between lines of a paragraph. Default: `0`
---| 'language' # Choose a set of hyphenation patterns and exceptions.
---| 'lastlinefit' # A readable and writable integer parameter that accepts and returns Lua numbers.
---| 'lefthyphenmin' # (TeX3 only) Minimal number of characters before a hyphenation. Default: `2`.
---| 'linepenalty' # Penalty value associated with each line break. Default: `10`.
---| 'localbrokenpenalty' # Local broken penalty (from `\localbrokenpenalty`).
---| 'localinterlinepenalty' # Local interline penalty (from `\localinterlinepenalty`).
---| 'looseness' # Number of lines by which this paragraph has to be made longer than it would be ideally.
---| 'mag' # `1000` times the magnification of the document.
---| 'maxdeadcycles' # The maximum number of times that the output routine is allowed to be called without a `\shipout` occurring. IniTEX default: `25`.
---| 'month' # The month of the current job.
---| 'newlinechar' # Number of the character that triggers a new line in `\write` statements.
---| 'outputpenalty' # Value of the penalty at the current page break, or `10000` if the break was not at a penalty.
---| 'pausing' # Specify that TEX should pause after each line that is read from a file.
---| 'postdisplaypenalty' # Penalty placed in the vertical list below a display. Default: `0`.
---| 'predisplaydirection' # An internal read/write integer, to indicate the direction of the last partial paragraph before a display; it is used to control the placement of elements such as equation numbers, and can be explictly set to affect this placement.
---| 'predisplaypenalty' # Penalty placed in the vertical list above a display. Default: `10000`.
---| 'pretolerance' # Tolerance value for a paragraph without hyphenation. Default: `100`.
---| 'relpenalty' # Penalty for breaking after a binary relation not enclosed in a subformula. Default: `500`.
---| 'righthyphenmin' # (TeX3 only) Minimum number of characters after a hyphenation. Default: `3`.
---| 'savinghyphcodes' # When a `\patterns` command is executed and \savinghyphcodes has a positive value, the current `\lccode` values are saved as hyphenation codes for the current language.
---| 'savingvdiscards' # When ε-TEX’s parameter `\savingvdiscards` has been assigned a positive value, these ‘discarded items’ are saved in two lists and can be recovered by the commands `\pagediscards` and `\splitdiscards` that act like ‘unvboxing’ hypothetical box registers containing a vbox with the discarded items.
---| 'showboxbreadth' # Number of successive elements on each level that are shown when boxes are displayed.
---| 'showboxdepth' # The number of levels of box dump that are shown when boxes are displayed.
---| 'time' # Number of minutes after midnight that the current job started.
---| 'tolerance' # Tolerance value for lines in a paragraph with hyphenation. Default: `200`.
---| 'tracingassigns' # When the program is compiled with the code for collecting statistics and `\tracingassigns` has a value of `1` or more, all assignments subject to TeX’s grouping mechanism are traced.
---| 'tracingcommands' # If this is `1` TEX displays primitive commands executed; if this is `2` or more the outcome of conditionals is also recorded.
---| 'tracinggroups' # When `\tracinggroups` has a value of `1` or more, the start and end of each save group is traced, together with the starting line and grouping level.
---| 'tracingifs' # When `\tracingifs` has a value of `1` or more, all conditionals (including `\unless`, `\or`, `\else`, and `\fi`) are traced, together with the starting line and nesting level.
---| 'tracinglostchars' # If this parameter is positive, TeX gives diagnostic messages whenever a character is accessed that is not present in a font. Plain default: `1`.
---| 'tracingmacros' # If this is `1`, the log file shows expansion of macros that are performed and the actual values of the arguments; if this is `2` or more token parameters such as `\output` and `\everypar` are also traced.
---| 'tracingnesting' # When `\tracingnesting` has a value of `1` or more, these anomalies are shown; when `\tracingnesting` has a value of `2` or more, the current context (traceback) is shown as well.
---| 'tracingonline' # If this parameter is positive, TeX will write trace information to the terminal in addition to the log file.
---| 'tracingoutput' # If this is positive, the log file shows a dump of boxes that are shipped to the dvi file.
---| 'tracingpages' # If this parameter is positive, TeX generates a trace of the page breaking algorithm.
---| 'tracingparagraphs' # If this parameter is positive, TeX generates a trace of the line breaking algorithm.
---| 'tracingrestores' # If this parameter is positive, TEX will report all values that are restored when a group ends.
---| 'tracingscantokens' # When `\tracingscantokens` has a value of `1` or more, the opening and closing of pseudo-files (generated by `\scantokens`) is recorded as for any other file, with ‘␣’ as filename.
---| 'tracingstats' # If this parameter is `1`, TeX reports at the end of the job the usage of various internal arrays; if it is `2`, the memory demands are given whenever a page is shipped out.
---| 'uchyph' # Positive to allow hyphenation of words starting with a capital letter. Default: `1`.
---| 'vbadness' # Amount of tolerance before TEX reports an underfull or overfull vertical box.
---| 'widowpenalty' # Additional penalty for breaking a page before the last line of a paragraph. Default: `150`.
---| 'year' # The year of the current job.

---
---The integer parameters accept and return Lua numbers. These are read-only:
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L358-366](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L358-366)
---@alias ReadOnlyIntegerParameter
---| 'deadcycles' # Counter that keeps track of how many times the output routine has been called without a `\shipout` taking place.
---| 'insertpenalties' # Total of penalties for split insertions. Inside the output routine, the number of held-over insertions.
---| 'parshape' # Command for general paragraph shapes.
---| 'interlinepenalties' # An extension to `\interlinepenalty` that allows different penalties to be added after each line. It takes a space separated list of numbers as value. If `n` is positive interlinepenalties `n i_1 ... i_n` specifies the penalty to be used after each of the lines in a paragraph, with the last value being repeated as often as needed if the paragraph has more then `n` lines.
---| 'clubpenalties' # An extension to `\clubpenalty` that allows different penalties to be added after each line. It takes a space separated list of numbers as value. If `n` is positive clubpenalties `n i_1 ... i_n` specifies the penalty to be used after each of the lines in a paragraph, with the last value being repeated as often as needed if the paragraph has more then `n` lines.
---| 'widowpenalties' # An extension to `\widowpenalty` that allows different penalties to be added after each line. It takes a space separated list of numbers as value. If `n` is positive widowpenalties `n i_1 ... i_n` specifies the penalty to be used after each of the lines in a paragraph, with the last value being repeated as often as needed if the paragraph has more then `n` lines. Lines are counted backwards from the last line in this case.
---| 'displaywidowpenalties' # An extension to `\widowpenalty` that allows different penalties to be added after each line. It takes a space separated list of numbers as value. If `n` is positive displaywidowpenalties `n i_1 ... i_n` specifies the penalty to be used after each of the lines in a paragraph, with the last value being repeated as often as needed if the paragraph has more then `n` lines. Lines counted backwards from each display math within a paragraph.
---| 'prevgraf' # The number of lines in the paragraph last added to the vertical list.
---| 'spacefactor' # `1000` times the ratio by which the stretch (shrink) component of the interword glue should be multiplied (divided).

---
---The dimension parameters accept Lua numbers (signifying scaled points) or
---strings (with included dimension). The result is always a number in scaled
---points. These are read-write:
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L378-407](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L378-407)
---@alias ReadWriteDimensionParameter
---| 'boxmaxdepth' # Maximum allowed depth of boxes. Default: `\maxdimen`.
---| 'delimitershortfall' # Size of the part of a delimited formula that is allowed to go uncovered by a delimiter. Default: `5pt`.
---| 'displayindent' # Distance by which the box, in which the display is centred, is indented owing to hanging indentation.
---| 'displaywidth' # Width of the box in which the display is centred.
---| 'emergencystretch' # (TeX3 only) Assumed extra stretchability in lines of a paragraph.
---| 'hangindent' # If positive, this indicates indentation from the left margin; if negative, this is the negative of the indentation from the right margin.
---| 'hfuzz' # Excess size that TeX tolerates before it considers a horizontal box overfull.
---| 'hoffset' # Distance by which the page is shifted right with respect to the reference point.
---| 'hsize' # Line width used for typesetting a paragraph. Default: `6.5in`.
---| 'lineskiplimit' # Distance to be maintained between the bottom and top of neighbouring boxes on a vertical list. Default: `0pt`.
---| 'mathsurround' # Kern amount placed before and after in-line formulas.
---| 'maxdepth' # Maximum depth of the page box. Default: `4pt`.
---| 'nulldelimiterspace' # Width taken for empty delimiters. Default: `1.2pt`.
---| 'overfullrule' # Width of the rule that is printed to indicate overfull horizontal boxes.
---| 'pagebottomoffset' # To set the bottom margin in non-TLT text.
---| 'pageheight' # The page height of the PDF output (the screen, the paper, etc.).
---| 'pageleftoffset' # To set the left margin in non-TLT text.
---| 'pagerightoffset' # To set the right margin in non-TLT text.
---| 'pagetopoffset' # To set the top margin in non-TLT text.
---| 'pagewidth' # The page width of the PDF output (the screen, the paper, etc.).
---| 'parindent' # Width of the indentation box added in front of a paragraph. Default: `20pt`.
---| 'predisplaysize' # Effective width of the line preceding the display.
---| 'scriptspace' # Extra space after subscripts and superscripts. Default: `0.5pt`.
---| 'splitmaxdepth' # Maximum depth of a box split off by a `\vsplit` operation. Default: `\maxdimen`.
---| 'vfuzz' # Excess size that TeX tolerates before it considers a vertical box overfull.
---| 'voffset' # Distance by which the page is shifted right/down with respect to the reference point.
---| 'vsize' # Height of the page box. Default: `8.9in`.
---| 'prevdepth' # Depth of the last box added to a vertical list as it is perceived by TeX.
---| 'prevgraf' # The number of lines in the paragraph last added to the vertical list.
---| 'spacefactor' # `1000` times the ratio by which the stretch (shrink) component of the interword glue should be multiplied (divided).

---
---The dimension parameters accept Lua numbers (signifying scaled points) or
---strings (with included dimension). The result is always a number in scaled
---points. These are read-only:
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L415-422](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L415-422)
---@alias ReadOnlyDimensionParameter
---| 'pagedepth' # Depth of the current page.
---| 'pagefilllstretch' # Accumulated third-order stretch of the current page.
---| 'pagefillstretch' # Accumulated second-order stretch of the current page.
---| 'pagefilstretch' # Accumulated first-order stretch of the current page.
---| 'pagegoal' # Goal height of the page box. This starts at `\vsize`, and is diminished by heights of insertion items.
---| 'pageshrink' # Accumulated shrink of the current page.
---| 'pagestretch' # Accumulated zeroth-order stretch of the current page.
---| 'pagetotal' # Accumulated natural height of the current page.

---
---The direction parameters are read-only and return a Lua string.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L465-469](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L465-469)
---@alias DirectionParameter
---| 'bodydir' # The direction for vboxes, including the main body of text. Default: `TLT`.
---| 'mathdir' # The direction for mathematics. Default: `TLT`.
---| 'pagedir' # The direction for pages (`\shipout`). Default: `TLT`.
---| 'pardir' # The direction for paragraphs. Default: `TLT`.
---| 'textdir' # The direction for text and hboxes. Default: `TLT`.

---
---The glue parameters accept and return a userdata object that
---represents a `glue_spec` node.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L479-493](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L479-493)
---@alias GlueParameter
---| 'abovedisplayshortskip' # Glue above a display if the line preceding the display was short. Default: `0pt plus 3pt`.
---| 'abovedisplayskip' # Glue above a display. Default: `12pt plus 3pt minus 9pt`.
---| 'baselineskip' # The ‘ideal’ baseline distance between neighbouring boxes on a vertical list. Default: `12pt`.
---| 'belowdisplayshortskip' # Glue above/below a display if the line preceding the display was short. Default: `0pt plus 3pt` and `7pt plus 3pt minus 4pt` respectively.
---| 'belowdisplayskip' # Glue below a display. Default: `12pt plus 3pt minus 9pt`.
---| 'leftskip' # Glue that is placed to the left of all lines of a paragraph.
---| 'lineskip' # Glue added if the distance between bottom and top of neighbouring boxes is less than `\lineskiplimit`. Default: `1pt`.
---| 'parfillskip' # Glue that is placed between the last element of the paragraph and the line end. Default: `0pt plus 1fil`.
---| 'parskip' # Amount of glue added to the surrounding vertical list when a paragraph starts. Default: `0pt plus 1pt`.
---| 'rightskip' # Glue that is placed to the right of all lines of a paragraph.
---| 'spaceskip' # Interword glue if non-zero.
---| 'splittopskip' # Minimum distance between the top of what remains after a `\vsplit` operation, and the first item in that box. Default: `10pt`.
---| 'tabskip' # Amount of glue in between columns (rows) of an `\halign` (`\valign`).
---| 'topskip' # Minimum distance between the top of the page box and the baseline of the first box on the page. Default: `10pt`.
---| 'xspaceskip' # Interword glue if non-zero and `\spacefactor` ≥ 2000.

---
---All muglue (math units) parameters are to be used read-only and return a Lua string.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L503-505](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L503-505)
---@alias MuglueParameter
---| 'medmuskip' # Medium amount of mu (math units) glue.
---| 'thickmuskip' # Large amount of mu (math units) glue.
---| 'thinmuskip' # Small amount of mu (math units) glue.

---
---The tokenlist parameters accept and return Lua strings. Lua strings are
---converted to and from token lists using `\the\toks` style expansion:
---all category codes are either space (10) or other (12). It follows that assigning
---to some of these, like `tex.output`, is actually useless, but it feels bad
---to make exceptions in view of a coming extension that will accept full-blown
---token strings.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L520-529](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L520-529)
---@alias TokenlistParameter
---| 'errhelp' # Tokens that will be displayed if the user asks further help after an `\errmessage`.
---| 'everycr' # Token list inserted after every `\cr` or non-redundant `\crcr`.
---| 'everydisplay' # Token list inserted at the start of a display.
---| 'everyeof' # The content of this token list is injected when a file ends.
---| 'everyhbox' # Token list inserted at the start of a horizontal box.
---| 'everyjob' # Token list that is inserted at the start of each new job.
---| 'everymath' # Token list inserted at the start of non-display math.
---| 'everypar' # Token list inserted in front of paragraph text.
---| 'everyvbox' # Token list inserted at the start of a vertical box.
---| 'output' # Token list with instructions for shipping out pages.

---
---@alias InternalParameter
---| ReadWriteIntegerParameter
---| ReadOnlyIntegerParameter
---| ReadWriteDimensionParameter
---| ReadOnlyDimensionParameter
---| DirectionParameter
---| GlueParameter
---| MuglueParameter
---| TokenlistParameter

---
---Set the given TeX parameter globally.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1714-1813](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/80a00d7131ae569d4a49f374e988226fe20de0d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1714-1813)
---
---@param global 'global' # It is possible to use `global` as the first argument to `tex.set`; this makes the assignment global instead of local.
---@param parameter InternalParameter # The name of the TeX parameter.
---@param ... any
function tex.set(global, parameter, ...) end

---
---Set the given TeX parameter.
---
---When you set a glue quantity you can either pass a `glue_spec` or upto five numbers.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1714-1813](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/80a00d7131ae569d4a49f374e988226fe20de0d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1714-1813)
---
---@param parameter InternalParameter # The name of the TeX parameter.
---@param ... any
function tex.set(parameter, ...) end

---
---Set the given glue parameter globally by specifying the `width`, `stretch`,
---`stretch_order`, `shrink` and `shrink_order` of the glue.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1714-1813](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/80a00d7131ae569d4a49f374e988226fe20de0d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1714-1813)
---
---@param global 'global' # It is possible to use `global` as the first argument to `tex.set`; this makes the assignment global instead of local.
---@param parameter GlueParameter # The name of the glue parameter.
---@param width integer # The horizontal or vertical displacement.
---@param stretch? integer # An extra (positive) displacement or stretch amount.
---@param stretch_order? integer # Factor applied to stretch amount.
---@param shrink? integer # An extra (negative) displacement or shrink amount.
---@param shrink_order? integer # Factor applied to shrink amount.
function tex.set(global, parameter, width, stretch, stretch_order, shrink, shrink_order) end

---
---Set the given glue parameter by specifying the `width`, `stretch`,
---`stretch_order`, `shrink` and `shrink_order` of the glue.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1714-1813](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/80a00d7131ae569d4a49f374e988226fe20de0d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1714-1813)
---
---@param parameter GlueParameter # The name of the glue parameter.
---@param width integer # The horizontal or vertical displacement.
---@param stretch? integer # An extra (positive) displacement or stretch amount.
---@param stretch_order? integer # Factor applied to stretch amount.
---@param shrink? integer # An extra (negative) displacement or shrink amount.
---@param shrink_order? integer # Factor applied to shrink amount.
function tex.set(parameter, width, stretch, stretch_order, shrink, shrink_order) end

---
---Set the given glue parameter globally using a `glue_spec` node.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1714-1813](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/80a00d7131ae569d4a49f374e988226fe20de0d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1714-1813)
---
---@param global 'global' # It is possible to use `global` as the first argument to `tex.set`; this makes the assignment global instead of local.
---@param parameter GlueParameter # The name of the glue parameter.
---@param glue_spec GlueSpecNode # A glue spec node.
function tex.set(global, parameter, glue_spec) end

---
---Set the given glue parameter using a `glue_spec` node.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1714-1813](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/80a00d7131ae569d4a49f374e988226fe20de0d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1714-1813)
---
---@param parameter GlueParameter # The name of the glue parameter.
---@param glue_spec GlueSpecNode # A glue spec node.
function tex.set(parameter, glue_spec) end

---
---Query the given TeX parameter.
---
---The exact return values differ depending on the actual parameter.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L2120-L2200](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2120-L2200)
---@param parameter InternalParameter
---@param opts? boolean
---
---@return any ...
function tex.get(parameter, opts) end

---
---Query the given glue parameter.
---
---The exact return values differ depending on the actual parameter.
---
---Glue is kind of special:
---The return value
---is a `glue_spec` node but when you pass `false` as last argument to
---`tex.get` you get the width of the glue and when you pass `true` you
---get all five values. Otherwise you get a node which is a copy of the internal
---value so you are responsible for its freeing at the *Lua* end.
---
---If
---you pass `true` to `get` you get 5 values returned for a glue and
---when you pass `false` you only get the width returned.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L2120-L2200](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2120-L2200)

---
---@param parameter GlueParameter
---
---@return GlueSpecNode
function tex.get(parameter) end

---
---Query the given glue parameter.
---
---The exact return values differ depending on the actual parameter.
---
---Glue is kind of special:
---The return value
---is a `glue_spec` node but when you pass `false` as last argument to
---`tex.get` you get the width of the glue and when you pass `true` you
---get all five values. Otherwise you get a node which is a copy of the internal
---value so you are responsible for its freeing at the *Lua* end.
---
---If
---you pass `true` to `get` you get 5 values returned for a glue and
---when you pass `false` you only get the width returned.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L2120-L2200](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2120-L2200)
---@param parameter GlueParameter
---@param all true
---
---@return integer width  # The horizontal or vertical displacement.
---@return integer stretch  # An extra (positive) displacement or stretch amount.
---@return integer stretch_order  # Factor applied to stretch amount.
---@return integer shrink  # An extra (negative) displacement or shrink amount.
---@return integer shrink_order  # Factor applied to shrink amount.
function tex.get(parameter, all) end

---
---Query the given glue parameter.
---
---The exact return values differ depending on the actual parameter.
---
---Glue is kind of special:
---The return value
---is a `glue_spec` node but when you pass `false` as last argument to
---`tex.get` you get the width of the glue and when you pass `true` you
---get all five values. Otherwise you get a node which is a copy of the internal
---value so you are responsible for its freeing at the *Lua* end.
---
---If
---you pass `true` to `get` you get 5 values returned for a glue and
---when you pass `false` you only get the width returned.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L2120-L2200](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2120-L2200)
---@param parameter GlueParameter
---@param all false
---
---@return integer width  # The horizontal or vertical displacement.
function tex.get(parameter, all) end

---
---`\adjdemerits`: Penalty for adjacent visually incompatible lines.
---Default: `10000`.
---
---__Reference:__
---
---* [TeX by Topic, page 175](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.adjdemerits = 0

---
---`\binoppenalty`: Penalty for breaking after a binary operator not
---enclosed in a subformula. Default: `700`
---
---__Reference:__
---
---* [TeX by Topic, page 201](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.binoppenalty = 0

---
---`\brokenpenalty`: Additional penalty for breaking a page after a
---hyphenated line. Default: `100`.
---
---__Reference:__
---
---* [TeX by Topic, page 226](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Type definition and documentation](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 💁🏻 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.brokenpenalty = 0

---
---Catcode tables are a new feature that allows you to switch to a
---predefined catcode regime in a single statement.
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.catcodetable = 0

---
---`\clubpenalty`: Extra penalty for breaking a page after the first
---line of a paragraph. In Plain TeX this is `150`. This amount, and the
---following penalties, are added to the `\interlinepenalty`, and a
---penalty of the resulting size is inserted after the `\hbox`
---containing the first line of a paragraph instead of the
---`\interlinepenalty`.
---
---__Reference:__
---
---* [TeX by Topic, page 227](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.clubpenalty = 0

---
---`\day`: The day of the current job.
---
---__Reference:__
---
---* [TeX by Topic, page 257](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.day = 0

---
---`\defaulthyphenchar`: Value of `\hyphenchar` when a font is loaded.
---Default: `‘\-`.
---
---__Reference:__
---
---* [TeX by Topic, page 52](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.defaulthyphenchar = 0

---
---`\defaultskewchar`: Value of `\skewchar` when a font is loaded.
---
---__Reference:__
---
---* [TeX by Topic, page 190](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.defaultskewchar = 0

---
---`\delimiterfactor`: `1000` times the fraction of a delimited formula
---that should be covered by a delimiter. Default: `901`.
---
---__Reference:__
---
---* [TeX by Topic, page 190](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.delimiterfactor = 0

---
---`\displaywidowpenalty`: Extra penalty for breaking a page before the
---last line above a display formula. The default value in Plain TeX
---is `50`.
---
---__Reference:__
---
---* [TeX by Topic, page 229](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.displaywidowpenalty = 0

---
---`\doublehyphendemerits`: Penalty for consecutive lines ending with a
---hyphen. Default: `10000`.
---
---__Reference:__
---
---* [TeX by Topic, page 175](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.doublehyphendemerits = 0

---
---`\endlinechar`: The character code of the end-of-line character
---appended to input lines. IniTEX default: `13`.
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.endlinechar = 0

---
---`\errorcontextlines`: (TeX3 only) Number of additional context lines
---shown in error messages.
---
---__Reference:__
---
---* [TeX by Topic, page 272](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.errorcontextlines = 0

---
---`\escapechar`: Number of the character that is to be used for the
---escape character when control sequences are being converted into
---character tokens. IniTEX default: `92` (`\`).
---
---__Reference:__
---
---* [TeX by Topic, page 43](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.escapechar = 0

---
---`\exhyphenpenalty`: Penalty for breaking a horizontal line at a
---discretionary item in the special case where the prebreak text is
---empty. Default: `50`.
---
---__Reference:__
---
---* [TeX by Topic, page 175](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.exhyphenpenalty = 0

---
---`\fam`: The number of the current font family.
---
---__Reference:__
---
---* [TeX by Topic, page 196](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.fam = 0

---
---`\finalhyphendemerits`: Penalty added when the penultimate line of a
---paragraph ends with a hyphen. Default: `5000`.
---
---__Reference:__
---
---* [TeX by Topic, page 175](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.finalhyphendemerits = 0

---
---`\floatingpenalty`: Penalty added when an insertion is split.
---
---__Reference:__
---
---* [TeX by Topic, page 240](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.floatingpenalty = 0

---
---`\globaldefs`: Parameter for overriding \global prefixes.
---IniTEX default: `0`.
---
---`\globaldefs` is a TeX internal parameter. Normally set to `0`. If
---set equal to a positive number, all following definitions and
---assignments are treated as if preceded by `\global`. If set equal to
---a negative number, none of the following definitions or assignments
---are global, meaning that an explicit `\global` is ignored. This
---remains in effect until `\globaldefs` is explicitly set to `0`
---again, or the group containing the `\globaldefs` reassignment ends
---(assuming it was not made global as well).
---
---__Reference:__
---
---* [TeX by Topic, page 105](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---* [Wikibook/TeX](https://en.wikibooks.org/wiki/TeX/globaldefs)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.globaldefs = 0

---
---`\hangafter`: If positive, this denotes the number of lines before
---indenting starts; if negative, the absolute value of this is the
---number of indented lines starting with the first line of the
---paragraph. Default: `1`.
---
---__Reference:__
---
---* [TeX by Topic, page 170](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.hangafter = 0

---
---`\hbadness`: Amount of tolerance before TEX reports an underfull or
---overfull horizontal box.
---
---__Reference:__
---
---* [TeX by Topic, page 57](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.hbadness = 0

---
---`\holdinginserts` (TeX3 only) If this is positive, insertions are
---not placed in their boxes at output time.
---
---__Reference:__
---
---* [TeX by Topic, page 240](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.holdinginserts = 0

---
---`\hyphenpenalty`: Penalty associated with break at a discretionary
---item in the general case. Default: `50`.
---
---__Reference:__
---
---* [TeX by Topic, page 175](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.hyphenpenalty = 0

---
---`\interlinepenalty`: Penalty for breaking a page between lines of a
---paragraph. Default: `0`.
---
---__Reference:__
---
---* [TeX by Topic, page 57](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.interlinepenalty = 0

---
---`\language`: Choose a set of hyphenation patterns and exceptions.
---
---__Reference:__
---
---* [TeX by Topic, page 176](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.language = 0

---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lastlinefit = 0

---
---`\lefthyphenmin` (TeX3 only) Minimal number of characters before a
---hyphenation. Default: `2`.
---
---__Reference:__
---
---* [TeX by Topic, page 175](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lefthyphenmin = 0

---
---`\linepenalty`: Penalty value associated with each line break.
---Default: `10`.
---
---__Reference:__
---
---* [TeX by Topic, page 175](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.linepenalty = 0

---
---Local broken penalty (from `\localbrokenpenalty`)
---
------@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.localbrokenpenalty = 0

---
---Local interline penalty (from `\localinterlinepenalty`)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.localinterlinepenalty = 0

---
---`\looseness`: Number of lines by which this paragraph has to be made
---longer than it would be ideally.
---
---__Reference:__
---
---* [TeX by Topic, page 175](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.looseness = 0

---
---`\mag`: `1000` times the magnification of the document.
---
---__Reference:__
---
---* [TeX by Topic, page 257](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.mag = 0

---
---`\maxdeadcycles`: The maximum number of times that the output
---routine is allowed to be called without a `\shipout` occurring.
---IniTEX default: `25`.
---
---__Reference:__
---
---* [TeX by Topic, page 288](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.maxdeadcycles = 0

---
---`\month`: The month of the current job.
---
---__Reference:__
---
---* [TeX by Topic, page 257](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.month = 0

---
---`\newlinechar`: Number of the character that triggers a new line
---in `\write` statements.
---
---__Reference:__
---
---* [TeX by Topic, page 246](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.newlinechar = 0

---
---`\outputpenalty`: Value of the penalty at the current page break,
---or `10000` if the break was not at a penalty.
---
---__Reference:__
---
---* [TeX by Topic, page 289](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.outputpenalty = 0

---
---`\pausing`: Specify that TEX should pause after each line that is
---read from a file.
---
---__Reference:__
---
---* [TeX by Topic, page 246](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pausing = 0

---
---`\postdisplaypenalty`: Penalty placed in the vertical list below a
---display. Default: `0`
---
---__Reference:__
---
---* [TeX by Topic, page 210](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.postdisplaypenalty = 0

---
---`\predisplaydirection`: An internal read/write integer, to indicate
---the direction of the last partial paragraph before a display; it is
---used to control the placement of elements such as equation numbers,
---and can be explictly set to affect this placement.
---
---__Reference:__
---
---* [The e-TeX Short Reference Manual](https://ftp.fau.de/ctan/systems/doc/etex/etex_ref.html#predisplaydirection)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.predisplaydirection = 0

---
---`\predisplaypenalty`: Penalty placed in the vertical list above a
---display. Default: `10000`.
---
---__Reference:__
---
---* [TeX by Topic, page 210](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.predisplaypenalty = 0

---
---`\pretolerance`: Tolerance value for a paragraph without
---hyphenation. Default: `100`.
---
---__Reference:__
---
---* [TeX by Topic, page 175](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pretolerance = 0

---
---`\relpenalty`: Penalty for breaking after a binary relation not
---enclosed in a subformula. Default: `500`
---
---__Reference:__
---
---* [TeX by Topic, page 201](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.relpenalty = 0

---
---`\righthyphenmin`: (TeX3 only) Minimum number of characters after a
---hyphenation. Default: `3`.
---
---__Reference:__
---
---* [TeX by Topic, page 201](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.righthyphenmin = 0

---
---When a `\patterns` command is executed and \savinghyphcodes has a
---positive value, the current `\lccode` values are saved as
---hyphenation codes for the current language.
---
---__Reference:__
---
---* The ε-TEX manual Version 2, February 1998 (updated March 2024), page 11
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.savinghyphcodes = 0

---
---When ε-TEX’s parameter `\savingvdiscards` has been assigned a
---positive value, these ‘discarded items’ are saved in two lists and
---can be recovered by the commands `\pagediscards` and `\splitdiscards`
---that act like ‘unvboxing’ hypothetical box registers containing a
---vbox with the discarded items.
---
---__Reference:__
---
---* The ε-TEX manual Version 2, February 1998 (updated March 2024), page 12
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.savingvdiscards = 0

---
---`\showboxbreadth`: Number of successive elements on each level that
---are shown when boxes are displayed.
---
---__Reference:__
---
---* [TeX by Topic, page 267](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.showboxbreadth = 0

---
---`\showboxdepth`: The number of levels of box dump that are shown when
---boxes are displayed.
---
---__Reference:__
---
---* [TeX by Topic, page 267](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.showboxdepth = 0

---
---`\time`: Number of minutes after midnight that the current job
---started.
---
---__Reference:__
---
---* [TeX by Topic, page 257](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.time = 0

---
---`\tolerance`: Tolerance value for lines in a paragraph with
---hyphenation. Default: `200`.
---
---__Reference:__
---
---* [TeX by Topic, page 175](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tolerance = 0

---
---When the program is compiled with the code for collecting statistics and `\tracingassigns` has a value of `1` or more, all assignments subject to TeX’s grouping mechanism are traced.
---
---__Reference:__
---
---* The ε-TEX manual Version 2, February 1998 (updated March 2024), page 4
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracingassigns = 0

---
---`\tracingcommands`: If this is `1` TEX displays primitive commands
---executed; if this is `2` or more the outcome of conditionals is also
---recorded.
---
---__Reference:__
---
---* [TeX by Topic, page 267](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracingcommands = 0

---
---When `\tracinggroups` has a value of `1` or more, the start and end
---of each save group is traced, together with the starting line and
---grouping level.
---
---__Reference:__
---
---* The ε-TEX manual Version 2, February 1998 (updated March 2024), page 4
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracinggroups = 0

---
---When `\tracingifs` has a value of `1` or more, all conditionals
---(including `\unless`, `\or`, `\else`, and `\fi`) are traced,
---together with the starting line and nesting level.
---
---__Reference:__
---
---* The ε-TEX manual Version 2, February 1998 (updated March 2024), page 4
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracingifs = 0

---
---`\tracinglostchars`: If this parameter is positive, TeX gives
---diagnostic messages whenever a character is accessed that is not
---present in a font. Plain default: `1`.
---
---__Reference:__
---
---* [TeX by Topic, page 268](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracinglostchars = 0

---
---`\tracingmacros`: If this is `1`, the log file shows expansion of
---macros that are performed and the actual values of the arguments;
---if this is `2` or more token parameters such as `\output` and
---`\everypar` are also traced.
---
---__Reference:__
---
---* [TeX by Topic, page 293](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracingmacros = 0

---
---When `\tracingnesting` has a value of `1` or more, these anomalies
---are shown; when `\tracingnesting` has a value of `2` or more, the
---current context (traceback) is shown as well.
---
---__Reference:__
---
---* The ε-TEX manual Version 2, February 1998 (updated March 2024), page 5
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracingnesting = 0

---
---`\tracingonline`: If this parameter is positive, TeX will write
---trace information to the terminal in addition to the log file.
---
---__Reference:__
---
---* [TeX by Topic, page 267](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracingonline = 0

---
---`\tracingoutput`: If this is positive, the log file shows a dump of
---boxes that are shipped to the dvi file.
---
---__Reference:__
---
---* [TeX by Topic, page 267](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracingoutput = 0

---
---`\tracingpages`: If this parameter is positive, TeX generates a
---trace of the page breaking algorithm.
---
---__Reference:__
---
---* [TeX by Topic, page 57](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracingpages = 0

---
---`\tracingparagraphs`: If this parameter is positive, TeX
---generates a trace of the line breaking algorithm.
---
---__Reference:__
---
---* [TeX by Topic, page 57](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracingparagraphs = 0

---
---`\tracingrestores`: If this parameter is positive, TeX will report
---all values that are restored when a group ends.
---
---__Reference:__
---
---* [TeX by Topic, page 57](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracingrestores = 0

---
---When `\tracingscantokens` has a value of `1` or more, the opening
---and closing of pseudo-files (generated by `\scantokens`) is recorded
---as for any other file, with ‘␣’ as filename.
---
---__Reference:__
---
---* The ε-TEX manual Version 2, February 1998 (updated March 2024), page 5
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracingscantokens = 0

---
---`\tracingstats`: If this parameter is `1`, TeX reports at the end
---of the job the usage of various internal arrays; if it is `2`, the
---memory demands are given whenever a page is shipped out.
---
---__Reference:__
---
---* [TeX by Topic, page 268](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tracingstats = 0

---
---`\uchyph`: Positive to allow hyphenation of words starting with a
---capital letter. Default: `1`.
---
---__Reference:__
---
---* [TeX by Topic, page 57](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.uchyph = 0

---
---`\vbadness`: Amount of tolerance before TeX reports an underfull or
---overfull vertical box.
---
---__Reference:__
---
---* [TeX by Topic, page 57](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.vbadness = 0

---
---`\widowpenalty`: Additional penalty for breaking a page before the
---last line of a paragraph. Default: `150`
---
---__Reference:__
---
---* [TeX by Topic, page 226](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.widowpenalty = 0

---
---`\year`: The year of the current job.
---
---__Reference:__
---
---* [TeX by Topic, page 257](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable integer parameter that accepts and returns Lua numbers.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.year = 0

---
---`\deadcycles`: Counter that keeps track of how many times the output
---routine has been called without a `\shipout` taking place.
---
---__Reference:__
---
---* [TeX by Topic, page 234](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A read-only integer parameter that returns a Lua number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.deadcycles = 0

---
---`\insertpenalties`: Total of penalties for split insertions.
---Inside the output routine, the number of held-over insertions.
---
---__Reference:__
---
---* [TeX by Topic, page 240](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A read-only integer parameter that returns a Lua number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.insertpenalties = 0

---
---`\parshape`: Command for general paragraph shapes.
---
---__Reference:__
---
---* [TeX by Topic, page 170](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A read-only integer parameter that returns a Lua number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.parshape = 0

---
---An extension to `\interlinepenalty` that allows different penalties
---to be added after each line. It takes a space separated list of
---numbers as value. If `n` is positive interlinepenalties
---`n i_1 ... i_n` specifies the penalty to be used after each of the
---lines in a paragraph, with the last value being repeated as often as
---needed if the paragraph has more then `n` lines.
---
---__Reference:__
---
---* [Stackoverflow: What are penalties and which ones are defined?](https://tex.stackexchange.com/a/51264)
---* The ε-TEX manual Version 2, February 1998 (updated March 2024), page 10
---
---@type integer # A read-only integer parameter that returns a Lua number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.interlinepenalties = 0

---
---An extension to `\clubpenalty` that allows different penalties to be
---added after each line. It takes a space separated list of numbers as
---value. If `n` is positive clubpenalties `n i_1 ... i_n` specifies the
---penalty to be used after each of the lines in a paragraph, with the
---last value being repeated as often as needed if the paragraph has
---more then `n` lines.
---
---__Reference:__
---
---* [Stackoverflow: What are penalties and which ones are defined?](https://tex.stackexchange.com/a/51264)
---* The ε-TEX manual Version 2, February 1998 (updated March 2024), page 10
---
---@type integer # A read-only integer parameter that returns a Lua number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.clubpenalties = 0

---
---An extension to `\widowpenalty` that allows different penalties to be
---added after each line. It takes a space separated list of numbers as
---value. If `n` is positive widowpenalties `n i_1 ... i_n` specifies
---the penalty to be used after each of the lines in a paragraph, with
---the last value being repeated as often as needed if the paragraph
---has more then `n` lines. Lines are counted backwards from the last
---line in this case.
---
---__Reference:__
---
---* [Stackoverflow: What are penalties and which ones are defined?](https://tex.stackexchange.com/a/51264)
---* The ε-TEX manual Version 2, February 1998 (updated March 2024), page 10
---
---@type integer # A read-only integer parameter that returns a Lua number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.widowpenalties = 0

---
---An extension to `\widowpenalty` that allows different penalties to be
---added after each line. It takes a space separated list of numbers as
---value. If `n` is positive displaywidowpenalties `n i_1 ... i_n`
---specifies the penalty to be used after each of the lines in a
---paragraph, with the last value being repeated as often as needed if
---the paragraph has more then `n` lines. Lines counted backwards from
---each display math within a paragraph.
---
---__Reference:__
---
---* [Stackoverflow: What are penalties and which ones are defined?](https://tex.stackexchange.com/a/51264)
---* The ε-TEX manual Version 2, February 1998 (updated March 2024), page 10
---
---@type integer # A read-only integer parameter that returns a Lua number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.displaywidowpenalties = 0

---
---`\prevgraf`: The number of lines in the paragraph last added to the
---vertical list.
---
---__Reference:__
---
---* [TeX by Topic, page 175](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A read-only integer parameter that returns a Lua number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.prevgraf = 0

---
---`\spacefactor`: `1000` times the ratio by which the stretch (shrink)
---component of the interword glue should be multiplied (divided).
---
---__Reference:__
---
---* [TeX by Topic, page 185](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A read-only integer parameter that returns a Lua number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.spacefactor = 0

---
---`\boxmaxdepth`: Maximum allowed depth of boxes. Default: `\maxdimen`.
---
---__Reference:__
---
---* [TeX by Topic, page 57](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.boxmaxdepth = 0

---
---`\delimitershortfall`: Size of the part of a delimited formula that
---is allowed to go uncovered by a delimiter. Default: `5pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 190](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.delimitershortfall = 0

---
---`\displayindent`: Distance by which the box, in which the display is
---centred, is indented owing to hanging indentation.
---
---__Reference:__
---
---* [TeX by Topic, page 210](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.displayindent = 0

---
---`\displaywidth`: Width of the box in which the display is centred.
---
---__Reference:__
---
---* [TeX by Topic, page 210](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.displaywidth = 0

---
---`\emergencystretch` (TeX3 only) Assumed extra stretchability in
---lines of a paragraph.
---
---__Reference:__
---
---* [TeX by Topic, page 175](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.emergencystretch = 0

---
---`\hangindent`: If positive, this indicates indentation from the
---left margin; if negative, this is the negative of the indentation
---from the right margin.
---
---__Reference:__
---
---* [TeX by Topic, page 170](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.hangindent = 0

---
---`\hfuzz`: Excess size that TeX tolerates before it considers a
---horizontal box overfull.
---
---__Reference:__
---
---* [TeX by Topic, page 57](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.hfuzz = 0

---
---`\hoffset`:  Distance by which the page is shifted right with respect
---to the reference point.
---
---__Reference:__
---
---* [TeX by Topic, page 224](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.hoffset = 0

---
---`\hsize`: Line width used for typesetting a paragraph.
---Default: `6.5in`.
---
---__Reference:__
---
---* [TeX by Topic, page 170](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.hsize = 0

---
---`\lineskiplimit`: Distance to be maintained between the bottom and
---top of neighbouring boxes on a vertical list. Default: `0pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 201](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lineskiplimit = 0

---
---`\mathsurround`: Kern amount placed before and after in-line
---formulas.
---
---__Reference:__
---
---* [TeX by Topic, page 201](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.mathsurround = 0

---
---`\maxdepth`: Maximum depth of the page box. Default: `4pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 224](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.maxdepth = 0

---
---`\nulldelimiterspace`: Width taken for empty delimiters.
---Default: `1.2pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 190](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.nulldelimiterspace = 0

---
---`\overfullrule`: Width of the rule that is printed to indicate
---overfull horizontal boxes.
---
---__Reference:__
---
---* [TeX by Topic, page 57](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.overfullrule = 0

---
---`\pagebottomoffset`: To set the bottom margin in non-TLT text.
---
---__Reference:__
---
---* https://mirrors.aliyun.com/CTAN/systems/doc/aleph/News
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pagebottomoffset = 0

---
---`\pageheight`: The page height of the PDF output (the screen, the
---paper, etc.).
---
---__Reference:__
---
---* pdfTeX-Manual, page 25
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pageheight = 0

---
---`\pageleftoffset`: To set the left margin in non-TLT text.
---
---__Reference:__
---
---* https://mirrors.aliyun.com/CTAN/systems/doc/aleph/News
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pageleftoffset = 0

---
---`\pagerightoffset`: To set the right margin in non-TLT text.
---
---__Reference:__
---
---* https://mirrors.aliyun.com/CTAN/systems/doc/aleph/News
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pagerightoffset = 0

---
---`\pagetopoffset`: To set the top margin in non-TLT text.
---
---__Reference:__
---
---* https://mirrors.aliyun.com/CTAN/systems/doc/aleph/News
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pagetopoffset = 0

---
---`\pagewidth`: The page width of the PDF output (the screen,
---the paper, etc.).
---
---__Reference:__
---
---* pdfTeX-Manual, page 25
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pagewidth = 0

---
---`\parindent`: Width of the indentation box added in front of a
---paragraph. Default: `20pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 170](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.parindent = 0

---
---`\predisplaysize`: Effective width of the line preceding the display.
---
---__Reference:__
---
---* [TeX by Topic, page 210](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.predisplaysize = 0

---
---`\scriptspace`: Extra space after subscripts and superscripts.
---Default: `0.5pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 200](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.scriptspace = 0

---
---`\splitmaxdepth`: Maximum depth of a box split off by a `\vsplit`
---operation. Default: `\maxdimen`.
---
---__Reference:__
---
---* [TeX by Topic, page 224](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.splitmaxdepth = 0

---
---`\vfuzz`: Excess size that TeX tolerates before it considers a
---vertical box overfull.
---
---__Reference:__
---
---* [TeX by Topic, page 57](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.vfuzz = 0

---
---`\voffset`: Distance by which the page is shifted right/down with
---respect to the reference point.
---
---__Reference:__
---
---* [TeX by Topic, page 224](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.voffset = 0

---
---`\vsize`: Height of the page box. Default: `8.9in`.
---
---__Reference:__
---
---* [TeX by Topic, page 224](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.vsize = 0

---
---`\prevdepth`: Depth of the last box added to a vertical list as
---it is perceived by TeX.
---
---__Reference:__
---
---* [TeX by Topic, page 157](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.prevdepth = 0

---
---`\prevgraf`: The number of lines in the paragraph last added to the
---vertical list.
---
---__Reference:__
---
---* [TeX by Topic, page 290](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.prevgraf = 0

---
---`\spacefactor`: `1000` times the ratio by which the stretch (shrink)
---component of the interword glue should be multiplied (divided).
---
---__Reference:__
---
---* [TeX by Topic, page 185](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---* [The TexBook: page 76. Also: 76, 271, 285, 363, 433.](https://ctan.org/tex-archive/systems/knuth/dist/tex/texbook.tex)
---
---@type integer # A readable and writable dimension parameter that accepts a Lua number (signifying scaled points) or a string (with included dimension). The result is always a number in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.spacefactor = 0

---
---`\pagedepth`: Depth of the current page.
---
---__Reference:__
---
---* [TeX by Topic, page 226](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---A read only dimension parameter. The result is always a number in scaled points.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pagedepth = 0

---
---`\pagefilllstretch`: Accumulated third-order stretch of the current
---page.
---
---__Reference:__
---
---* [TeX by Topic, page 226](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---A read only dimension parameter. The result is always a number in scaled points.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pagefilllstretch = 0

---
---`\pagefillstretch`: Accumulated second-order stretch of the current
---page.
---
---__Reference:__
---
---* [TeX by Topic, page 226](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---A read only dimension parameter. The result is always a number in scaled points.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pagefillstretch = 0

---
---`\pagefilstretch`: Accumulated first-order stretch of the current
---page.
---
---__Reference:__
---
---* [TeX by Topic, page 226](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---A read only dimension parameter. The result is always a number in scaled points.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pagefilstretch = 0

---
---`\pagegoal`: Goal height of the page box. This starts at `\vsize`,
---and is diminished by heights of insertion items.
---
---__Reference:__
---
---* [TeX by Topic, page 226](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---A read only dimension parameter. The result is always a number in scaled points.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pagegoal = 0

---
---`\pageshrink`: Accumulated shrink of the current page.
---
---__Reference:__
---
---* [TeX by Topic, page 226](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
------
---A read only dimension parameter. The result is always a number in scaled points.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pageshrink = 0

---
---`\pagestretch`: Accumulated zeroth-order stretch of the current page.
---
---__Reference:__
---
---* [TeX by Topic, page ](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---A read only dimension parameter. The result is always a number in scaled points.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pagestretch = 0

---
---`\pagetotal`: Accumulated natural height of the current page.
---
---__Reference:__
---
---* [TeX by Topic, page ](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---A read only dimension parameter. The result is always a number in scaled points.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pagetotal = 0

---
---The direction for vboxes, including the main body of text.
---Default: `TLT`.
---
---__Reference:__
---
---* [README of the Omega engine](https://mirrors.ctan.org/obsolete/systems/omega/README)
---
---@type DirectionSpecifier A read-only direction parameter that returns a *Lua* string.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.bodydir = "TLT"

---
---The direction for mathematics. Default: `TLT`.
---
---__Reference:__
---
---* [README of the Omega engine](https://mirrors.ctan.org/obsolete/systems/omega/README)
---
---@type DirectionSpecifier A read-only direction parameter that returns a *Lua* string.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.mathdir = "TLT"

---
---The direction for pages (`\shipout`). Default: `TLT`.
---
---__Reference:__
---
---* [README of the Omega engine](https://mirrors.ctan.org/obsolete/systems/omega/README)
---
---@type DirectionSpecifier A read-only direction parameter that returns a *Lua* string.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pagedir = "TLT"

---
---The direction for paragraphs. Default: `TLT`.
---
---__Reference:__
---
---* [README of the Omega engine](https://mirrors.ctan.org/obsolete/systems/omega/README)
---
---@type DirectionSpecifier A read-only direction parameter that returns a *Lua* string.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.pardir = "TLT"

---
---The direction for text and hboxes. Default: `TLT`.
---
---__Reference:__
---
---* [README of the Omega engine](https://mirrors.ctan.org/obsolete/systems/omega/README)
---
---@type DirectionSpecifier A read-only direction parameter that returns a *Lua* string.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.textdir = "TLT"

---
---`\abovedisplayshortskip`: Glue above a display if the line preceding
---the display was short. Default: `0pt plus 3pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 210](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.abovedisplayshortskip = nil

---
---`\abovedisplayskip`: Glue above a display.
---Default: `12pt plus 3pt minus 9pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 210](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.abovedisplayskip = nil

---
---`\baselineskip`: The ‘ideal’ baseline distance between neighbouring
---boxes on a vertical list. Default: `12pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 157](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.baselineskip = nil

---
---`\belowdisplayshortskip`: Glue above/below a display if the line
---preceding the display was short. Default: `0pt plus 3pt` and
---`7pt plus 3pt minus 4pt` respectively.
---
---__Reference:__
---
---* [TeX by Topic, page 210](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.belowdisplayshortskip = nil

---
---`\belowdisplayskip`: Glue below a display.
---Default: `12pt plus 3pt minus 9pt`.
---
---__Reference:__
---
---* [TeX by Topic, page ](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.belowdisplayskip = nil

---
---`\leftskip`: Glue that is placed to the left of all lines of a
---paragraph.
---
---__Reference:__
---
---* [TeX by Topic, page 170](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.leftskip = nil

---
---`\lineskip`: Glue added if the distance between bottom and top of
---neighbouring boxes is less than `\lineskiplimit`. Default: `1pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 157](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lineskip = nil

---
---`\parfillskip`: Glue that is placed between the last element of the
---paragraph and the line end. Default: `0pt plus 1fil`.
---
---__Reference:__
---
---* [TeX by Topic, page 166](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.parfillskip = nil

---
---`\parskip`: Amount of glue added to the surrounding vertical list
---when a paragraph starts. Default: `0pt plus 1pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 161](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.parskip = nil

---
---`\rightskip`: Glue that is placed to the right of all lines of a
---paragraph.
---
---__Reference:__
---
---* [TeX by Topic, page 170](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.rightskip = nil

---
---`\spaceskip`: Interword glue if non-zero.
---
---__Reference:__
---
---* [TeX by Topic, page 185](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.spaceskip = nil

---
---`\splittopskip`: Minimum distance between the top of what remains
---after a `\vsplit` operation, and the first item in that box.
---Default: `10pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 226](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.splittopskip = nil

---
---`\tabskip`: Amount of glue in between columns (rows) of an
---`\halign` (`\valign`).
---
---__Reference:__
---
---* [TeX by Topic, page 215](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.tabskip = nil

---
---`\topskip`: Minimum distance between the top of the page box and the
---baseline of the first box on the page. Default: `10pt`.
---
---__Reference:__
---
---* [TeX by Topic, page 224](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.topskip = nil

---
---`\xspaceskip`: Interword glue if non-zero and `\spacefactor` ≥ 2000.
---
---__Reference:__
---
---* [TeX by Topic, page 185](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type GlueSpecNode # A readable and writeable glue parameter that accepts and returns a userdata object which in turn represents a `glue_spec` node.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.xspaceskip = nil

---
---`\medmuskip`: Medium amount of mu (math units) glue.
---
---__Reference:__
---
---* [TeX by Topic, page 201](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type string # A read-only `muglue` parameter that returns a *Lua* string.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.medmuskip = ""

---
---`\thickmuskip`: Large amount of mu (math units) glue.
---
---__Reference:__
---
---* [TeX by Topic, page 210](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type string # A read-only `muglue` parameter that returns a *Lua* string.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.thickmuskip = ""

---
---`\thinmuskip`: Small amount of mu (math units) glue.
---
---__Reference:__
---
---* [TeX by Topic, page 201](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type string # A read-only `muglue` parameter that returns a *Lua* string.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.thinmuskip = ""

---
---`\errhelp`: Tokens that will be displayed if the user asks further
---help after an `\errmessage`.
---
---A `tokenlist` parameters that accepts and returns a *Lua* string.
---
---The *Lua* string is
---converted to and from a token list using `\the` `\toks` style expansion:
---all category codes are either space (10) or other (12).
---
---__Reference:__
---
---* [TeX by Topic, page 272](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.errhelp = ""

---
---`\everycr`: Token list inserted after every `\cr` or
---non-redundant `\crcr`.
---
---A `tokenlist` parameters that accepts and returns a *Lua* string.
---
---The *Lua* string is
---converted to and from a token list using `\the` `\toks` style expansion:
---all category codes are either space (10) or other (12).
---
---__Reference:__
---
---* [TeX by Topic, page 215](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
-----
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.everycr = ""

---
---`\everydisplay`: Token list inserted at the start of a display.
---
---A `tokenlist` parameters that accepts and returns a *Lua* string.
---
---The *Lua* string is
---converted to and from a token list using `\the` `\toks` style expansion:
---all category codes are either space (10) or other (12).
---
---__Reference:__
---
---* [TeX by Topic, page 210](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
-----
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.everydisplay = ""

---
---`\everyeof`: The content of this token list is injected when a
---file ends.
---
---A `tokenlist` parameters that accepts and returns a *Lua* string.
---
---The *Lua* string is
---converted to and from a token list using `\the` `\toks` style expansion:
---all category codes are either space (10) or other (12).
---
---__Reference:__
---
---* LuaMetaTeX-Manual, page 115
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.everyeof = ""

---
---`\everyhbox`: Token list inserted at the start of a horizontal box.
---
---A `tokenlist` parameters that accepts and returns a *Lua* string.
---
---The *Lua* string is
---converted to and from a token list using `\the` `\toks` style expansion:
---all category codes are either space (10) or other (12).
---
---__Reference:__
---
---* [TeX by Topic, page 284](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.everyhbox = ""

---
---`\everyjob`: Token list that is inserted at the start of each new
---job.
---
---A `tokenlist` parameters that accepts and returns a *Lua* string.
---
---The *Lua* string is
---converted to and from a token list using `\the` `\toks` style expansion:
---all category codes are either space (10) or other (12).
---
---__Reference:__
---
---* [TeX by Topic, page 254](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.everyjob = ""

---
---`\everymath`: Token list inserted at the start of non-display math.
---
---A `tokenlist` parameters that accepts and returns a *Lua* string.
---
---The *Lua* string is
---converted to and from a token list using `\the` `\toks` style expansion:
---all category codes are either space (10) or other (12).
---
---__Reference:__
---
---* [TeX by Topic, page 284](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.everymath = ""

---
---`\everypar`: Token list inserted in front of paragraph text.
---
---A `tokenlist` parameters that accepts and returns a *Lua* string.
---
---The *Lua* string is
---converted to and from a token list using `\the` `\toks` style expansion:
---all category codes are either space (10) or other (12).
---
---__Reference:__
---
---* [TeX by Topic, page 284](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.everypar = ""

---
---`\everyvbox`: Token list inserted at the start of a vertical box.
---
---A `tokenlist` parameters that accepts and returns a *Lua* string.
---
---The *Lua* string is
---converted to and from a token list using `\the` `\toks` style expansion:
---all category codes are either space (10) or other (12).
---
---__Reference:__
---
---* [TeX by Topic, page 284](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.everyvbox = ""

---
---`\output`: Token list with instructions for shipping out pages.
---
---A `tokenlist` parameters that accepts and returns a *Lua* string.
---
---The *Lua* string is
---converted to and from a token list using `\the` `\toks` style expansion:
---all category codes are either space (10) or other (12).
---
---__Reference:__
---
---* [TeX by Topic, page 289](http://mirrors.ctan.org/info/texbytopic/TeXbyTopic.pdf)
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.output = ""

---
---@type string # for example `2.2`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.eTeXVersion = ""

---
---@type string # for example `2.2`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.eTeXrevision = ""

---
---@type string # for example `luatex`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.formatname = ""

---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.jobname = ""

---
---@type string # for example `This is LuaTeX, Version 1.15.1 (TeX Live 2023/dev)`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.luatexbanner = ""

---
---@type string # for example `1`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.luatexrevision = ""

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lastpenalty = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lastkern = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lastskip = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lastnodetype = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.inputlineno = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lastxpos = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lastypos = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.randomseed = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.luatexversion = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.eTeXminorversion = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.eTeXversion = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.currentgrouplevel = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.currentgrouptype = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.currentiflevel = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.currentiftype = 0

---
---All “last item” commands are read-only and return a number.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.currentifbranch = 0

---
---The attribute registers accept and return Lua numbers.
---
---see `LuaTeX` manual: 10.3.5 Accessing registers: `set*`, `get*` and `is*`
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3698](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3698)
---
---@type table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.attribute = {}

---
---* Corresponding C source code: [ltexlib.c#L1087-L1102](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1087-L1102)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param register string|integer # A register number or a predefined csname string from `\attributedef`.
---@param value integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setattribute(global, register, value) end

---
---* Corresponding C source code: [ltexlib.c#L1087-L1102](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1087-L1102)
---
---@param register string|integer # A register number or a predefined csname string from `\attributedef`.
---@param value integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setattribute(register, value) end

---
---* Corresponding C source code: [ltexlib.c#L1104-L1114](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1104-L1114)
---
---@param register string|integer # A register number or a predefined csname string from `\attributedef`.
---
---@return integer value
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getattribute(register) end

---
---* Corresponding C source code: [ltexlib.c#L1080-L1083](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1080-L1083)
---* Corresponding C source code: [ltexlib.c#L350-L358](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L350-L358)
---
---@param register string|integer # A register number or a predefined csname string from `\attributedef`.
---
---@return false|integer
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.isattribute(register) end

---
---The count registers accept and return *Lua* numbers.
---
---see `LuaTeX` manual: 10.3.5 Accessing registers: `set*`, `get*` and `is*`
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3704](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3704)
---
---@type table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.count = {}

---
---* Corresponding C source code: [ltexlib.c#L1051-L1066](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1051-L1066)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param register string|integer # A register number or a predefined csname string from `\countdef`.
---@param value integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setcount(global, register, value) end

---
---* Corresponding C source code: [ltexlib.c#L1051-L1066](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1051-L1066)
---
---@param register string|integer # A register number or a predefined csname string from `\countdef`.
---@param value integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setcount(register, value) end

---
---* Corresponding C source code: [ltexlib.c#L1068-L1078](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1068-L1078)
---
---@param register string|integer # A register number or a predefined csname string from `\countdef`.
---
---@return integer value
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getcount(register) end

---
---* Corresponding C source code: [ltexlib.c#L1046-L1049](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1046-L1049)
---* Corresponding C source code: [ltexlib.c#L350-L358](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L350-L358)
---
---@param register string|integer # A register number or a predefined csname string from `\countdef`.
---
---@return false|integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.iscount(register) end

---
---The dimension registers accept *Lua* numbers (in scaled points) or
---strings (with an included absolute dimension; `em` and `ex`
---and `px` are forbidden). The result is always a number in scaled
---points.
---
---see `LuaTeX` manual: 10.3.5 Accessing registers: `set*`, `get*` and `is*`
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3703](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3703)
---
---@type table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.dimen = {}

---
---* Corresponding C source code: [ltexlib.c#L818-L834](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L818-L834)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param register string|integer # A register number or a predefined csname string from `\dimendef`.
---@param value integer|string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setdimen(global, register, value) end

---
---* Corresponding C source code: [ltexlib.c#L836-L846](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L836-L846)
---
---@param register string|integer # A register number or a predefined csname string from `\dimendef`.
---
---@return integer value
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getdimen(register) end

---
---* Corresponding C source code: [ltexlib.c#L813-L816](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L813-L816)
---* Corresponding C source code: [ltexlib.c#L350-L358](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L350-L358)
---
---@param register string|integer # A register number or a predefined csname string from `\dimendef`.
---
---@return false|integer
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.isdimen(register) end

---
---The glue registers are just skip registers but instead of userdata
---are verbose.
---
---see `LuaTeX` manual: 10.3.5 Accessing registers: `set*`, `get*` and `is*`
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3700](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3700)
---
---@type table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.glue = {}

---
---* Corresponding C source code: [ltexlib.c#L884-L903](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L884-L903)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param register string|integer # A register number or a predefined csname string from `\skipdef`.
---@param width? number
---@param stretch? number
---@param shrink? number
---@param stretch_order? integer
---@param shrink_order? integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setglue(global, register, width, stretch, shrink, stretch_order, shrink_order) end

---
---* Corresponding C source code: [ltexlib.c#L884-L903](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L884-L903)
---
---@param register string|integer # A register number or a predefined csname string from `\skipdef`.
---@param width? number
---@param stretch? number
---@param shrink? number
---@param stretch_order? integer
---@param shrink_order? integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setglue(register, width, stretch, shrink, stretch_order, shrink_order) end

---
---* Corresponding C source code: [ltexlib.c#L905-L949](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L905-L949)
---
---@param register string|integer # A register number or a predefined csname string from `\skipdef`.
---
---@return integer width
---@return integer stretch
---@return integer shrink
---@return integer stretch_order
---@return integer shrink_order
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getglue(register) end

---
---Alias of tex.isskip()
---
---* Corresponding C source code: [ltexlib.c#L848-L851](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L848-L851)
---
---* Corresponding C source code: [ltexlib.c#L350-L358](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L350-L358)
---
---@param register string|integer # A register number or a predefined csname string from `\skipdef`.
---
---@return false|integer
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.isglue(register) end

---
---see `LuaTeX` manual: 10.3.5 Accessing registers: `set*`, `get*` and `is*`
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3702](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3702)
---
---@type table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.muglue = {}

---
---* Corresponding C source code: [ltexlib.c#L979-L998](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L979-L998)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param register string|integer # A register number or a predefined csname string from `\muskipdef`.
---@param width? number
---@param stretch? number
---@param shrink? number
---@param stretch_order? integer
---@param shrink_order? integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setmuglue(global, register, width, stretch, shrink, stretch_order, shrink_order) end

---
---* Corresponding C source code: [ltexlib.c#L979-L998](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L979-L998)
---
---@param register string|integer # A register number or a predefined csname string from `\muskipdef`.
---@param width? number
---@param stretch? number
---@param shrink? number
---@param stretch_order? integer
---@param shrink_order? integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setmuglue(register, width, stretch, shrink, stretch_order, shrink_order) end

---
---* Corresponding C source code: [ltexlib.c#L1000-L1044](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1000-L1044)
---
---@param register string|integer # A register number or a predefined csname string from `\muskipdef`.
---
---@return integer width
---@return integer stretch
---@return integer shrink
---@return integer stretch_order
---@return integer shrink_order
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getmuglue(register) end

---
---Alias of tex.ismuskip()
---
---* Corresponding C source code: [ltexlib.c#L951-L954](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L951-L954)
---* Corresponding C source code: [ltexlib.c#L350-L358](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L350-L358)
---
---@param register string|integer # A register number or a predefined csname string from `\muskipdef`.
---
---@return false|integer
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.ismuglue(register) end

---
---see `LuaTeX` manual: 10.3.5 Accessing registers: `set*`, `get*` and `is*`
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3701](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3701)
---
---@type table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.muskip = {}

---
---* Corresponding C source code: [ltexlib.c#L956-L965](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L956-L965)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param register string|integer # A register number or a predefined csname string from `\muskipdef`.
---@param value Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setmuskip(global, register, value) end

---
---* Corresponding C source code: [ltexlib.c#L956-L965](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L956-L965)
---
---@param register string|integer # A register number or a predefined csname string from `\muskipdef`.
---@param value Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setmuskip(register, value) end

---
---* Corresponding C source code: [ltexlib.c#L967-L977](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L967-L977)
---
---@param register string|integer # A register number or a predefined csname string from `\muskipdef`.
---
---@return Node value
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getmuskip(register) end

---
---* Corresponding C source code: [ltexlib.c#L951-L954](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L951-L954)
---* Corresponding C source code: [ltexlib.c#L350-L358](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L350-L358)
---
---@param register string|integer # A register number or a predefined csname string from `\muskipdef`.
---
---@return false|integer
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.ismuskip(register) end

---
---The skip registers accept and return `glue_spec` userdata node
---objects.
---
---see `LuaTeX` manual: 10.3.5 Accessing registers: `set*`, `get*` and `is*`
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3699](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3699)
---
---@type table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.skip = {}

---
---* Corresponding C source code: [ltexlib.c#L853-L866](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L853-L866)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param register string|integer # A register number or a predefined csname string from `\skipdef`.
---@param skip GlueSpecNode
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setskip(global, register, skip) end

---
---* Corresponding C source code: [ltexlib.c#L853-L866](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L853-L866)
---
---@param register string|integer # A register number or a predefined csname string from `\skipdef`.
---@param skip GlueSpecNode
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setskip(register, skip) end

---
---* Corresponding C source code: [ltexlib.c#L868-L882](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L868-L882)
---
---@param register string|integer # A register number or a predefined csname string from `\skipdef`.
---
---@return GlueSpecNode
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getskip(register) end

---
---* Corresponding C source code: [ltexlib.c#L848-L851](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L848-L851)
---* Corresponding C source code: [ltexlib.c#L350-L358](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L350-L358)
---
---@param register string|integer # A register number or a predefined csname string from `\skipdef`.
---
---@return false|integer
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.isskip(register) end

---
---The token registers accept and return *Lua* strings. *Lua* strings are
---converted to and from token lists using `\the` `\toks` style
---expansion: all category codes are either space (10) or other (12).
---
---see `LuaTeX` manual: 10.3.5 Accessing registers: `set*`, `get*` and `is*`
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3705](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3705)
---
---@type table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.toks = {}

---
---Set a toks register. Also accepts a predefined csname string.
---
---* Corresponding C source code: [ltexlib.c#L1125-L1158](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1125-L1158)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param register string|integer # A register number or a predefined csname string from `\toksdef`.
---@param toks string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.settoks(global, register, toks) end

---
---Get a toks register. Also accepts a predefined csname string.
---
---* Corresponding C source code: [ltexlib.c#L1197-L1209](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1197-L1209)
---
---@param register string|integer # A register number or a predefined csname string from `\toksdef`.
---
---@return string toks
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.gettoks(register) end

---
---* Corresponding C source code: [ltexlib.c#L1120-L1123](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1120-L1123)
---* Corresponding C source code: [ltexlib.c#L350-L358](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L350-L358)
---
---@param register string|integer # A register number or a predefined csname string from `\toksdef`.
---
---@return false|integer
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.istoks(register) end

---
---For tokens registers we have an alternative where a catcode table is specified:
---
---```lua
---tex.scantoks(0,3,"$e=mc^2$")
---tex.scantoks("global",0,"$\\int\\limits^1_2$")
---```
---
---* Corresponding C source code: [ltexlib.c#L1160-L1195](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1160-L1195)
---
---@param global 'global'
---@param register integer
---@param catcodetable integer
---@param toks string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.scantoks(global, register, catcodetable, toks) end

---
---When no arguments are given the current maximum number of classes is returned.
---
---* Corresponding C source code: [ltexlib.c#L1211-L1242](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1211-L1242)
---
---@param position? 'top'|'bottom'|'first'|'splitbottom'|'splitfirst' # one of top, bottom, first, splitbottom or splitfirst
---@param class? integer marks class number.
---
---@return integer|string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getmark(position, class) end

---
---The *TeX*'s character code table `lccode` (lower case code) can be accessed and written to using
---a virtual subtable of the `tex` table.
---
---```lua
---for i = 0, 1024 do
--- -- Exclude C0 and C1 control codes
---  if i > 31 and not (i >= 127 and i <= 159) then
---    print(i, utf8.char(i), tex.lccode[i], utf8.char(tex.lccode[i]))
---  end
---end
---```
---
---Output:
---
---```
---...
---65	A	97	a
---66	B	98	b
---67	C	99	c
---...
---90	Z	122	z
---...
---97	a	97	a
---98	b	98	b
---99	c	99	c
---...
---122	z	122	z
---...
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3708](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3708)
---* Corresponding plain TeX control sequence: [\lccode](https://www.tug.org/utilities/plain/cseq.html#lccode-rp)
---
---@type table<integer, integer>
tex.lccode = {}

---
---Set the `lccode` (lower case code) and additionally
---the associated sibling for a character code at the same time.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1369-L1397](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1369-L1397)
---* Corresponding plain TeX control sequence: [\lccode](https://www.tug.org/utilities/plain/cseq.html#lccode-rp)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param char_code integer # A Unicode code point for which a lowercase variant of the character is to be set, for example `65` stands for an uppercase `A` and `97` for a lowercase `a`.
---@param lower_case integer # The Unicode code point for the lowercase variant of the character.
---@param upper_case? integer # The Unicode code point for the uppercase variant of the character.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setlccode(global, char_code, lower_case, upper_case) end

---
---Set the `lccode` (lower case code) and additionally
---the associated sibling for a character code at the same time.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1369-L1397](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1369-L1397)
---* Corresponding plain TeX control sequence: [\lccode](https://www.tug.org/utilities/plain/cseq.html#lccode-rp)
---
---@param char_code integer # A Unicode code point for which a lowercase variant of the character is to be set, for example `65` stands for an uppercase `A` and `97` for a lowercase `a`.
---@param lower_case integer # The Unicode code point for the lowercase variant of the character.
---@param upper_case? integer # The Unicode code point for the uppercase variant of the character.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setlccode(char_code, lower_case, upper_case) end

---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1404-L1410](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1404-L1410)
---* Corresponding plain TeX control sequence: [\lccode](https://www.tug.org/utilities/plain/cseq.html#lccode-rp)
---
---@param char_code integer # A Unicode code point for which a lowercase variant of the character should be returned, for example `65` stands for an uppercase `A` and `97` for a lowercase `a`.
---
---@return integer lower_case # The Unicode code point for the lowercase variant of the character.
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getlccode(char_code) end

---
---*TeX*'s character code table `uccode` (upper case code) can be accessed and written to using
---a virtual subtable of the `tex` table.
---
---```lua
---for i = 0, 1024 do
--- -- Exclude C0 and C1 control codes
---  if i > 31 and not (i >= 127 and i <= 159) then
---    print(i, utf8.char(i), tex.uccode[i], utf8.char(tex.uccode[i]))
---  end
---end
---```
---
---Output:
---
---```
---...
---65	A	65	A
---66	B	66	B
---67	C	67	C
---...
---90	Z	90	Z
---...
---97	a	65	A
---98	b	66	B
---99	c	67	C
---...
---122	z	90	Z
---...
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3709](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3709)
---* Corresponding plain TeX control sequence: [\uccode](https://www.tug.org/utilities/plain/cseq.html#uccode-rp)
---
---@type table<integer, integer>
tex.uccode = {}

---
---The function call interface for `uccode` (upper case code) additionally
---allows you to set the associated sibling at the same time.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1369-L1397](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1369-L1397)
---* Corresponding plain TeX control sequence: [\uccode](https://www.tug.org/utilities/plain/cseq.html#uccode-rp)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param char_code integer # A Unicode code point for which a uppercase variant of the character is to be set, for example `65` stands for an uppercase `A` and `97` for a lowercase `a`.
---@param upper_case integer # The Unicode code point for the uppercase variant of the character.
---@param lower_case? integer # The Unicode code point for the lowercase variant of the character.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setuccode(global, char_code, upper_case, lower_case) end

---
---The function call interface for `uccode` (upper case code) additionally
---allows you to set the associated sibling at the same time.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1369-L1397](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1369-L1397)
---* Corresponding plain TeX control sequence: [\uccode](https://www.tug.org/utilities/plain/cseq.html#uccode-rp)
---
---@param char_code integer # A Unicode code point for which a uppercase variant of the character is to be set, for example `65` stands for an uppercase `A` and `97` for a lowercase `a`.
---@param upper_case integer # The Unicode code point for the uppercase variant of the character.
---@param lower_case? integer # The Unicode code point for the lowercase variant of the character.
---
---@see tex.getuccode
---@see tex.setlccode
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setuccode(char_code, upper_case, lower_case) end

---
---Corresponding plain TeX control sequence: [\uccode](https://www.tug.org/utilities/plain/cseq.html#uccode-rp)
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1417-L1423](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1417-L1423)
---* Corresponding plain TeX control sequence: [\uccode](https://www.tug.org/utilities/plain/cseq.html#uccode-rp)
---
---@param char_code integer # A Unicode code point for which a uppercase variant of the character should be returned, for example `65` stands for an uppercase `A` and `97` for a lowercase `a`.
---
---@return integer upper_case # The Unicode code point for the uppercase variant of the character.
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getuccode(char_code) end

---
---Each character in a font has a space factor code that is an integer
---between `0` and `32767`. The code is used to adjust the space factor
---in a horizontal list. The uppercase letters `A-Z` have space factor
---code `999`. Most other characters have code `1000` [Donald E. Knuth, The TeXbook, page 76]. However,
---Plain TeX makes `)', `'', and `]' have space factor code `0`.
---Also, the `\frenchspacing` and `\nonfrenchspacing` modes in Plain
---TeX work by changing the `\sfcode` for: `.`, `?`, `!`, `:`, `;`,
---and `,` [Donald E. Knuth, The TeXbook, 351].
---
---*TeX*'s character code table `sfcode` (space factor code) can be
---accessed and written to using a virtual subtable of the `tex` table.
---
---```lua
---for i = 0, 1024 do
---  -- Exclude C0 and C1 control codes
---  if i > 31 and not (i >= 127 and i <= 159) then
---    print(i, utf8.char(i), tex.sfcode[i])
---  end
---end
---```
---
---Output:
---
---```
---32	 	1000
---33	!	3000
---34	"	1000
---35	#	1000
---36	$	1000
---37	%	1000
---38	&	1000
---39	'	0
---40	(	1000
---41	)	0
---42	*	1000
---43	+	1000
---44	,	1250
---45	-	1000
---46	.	3000
---47	/	1000
---48	0	1000
---49	1	1000
---50	2	1000
---51	3	1000
---52	4	1000
---53	5	1000
---54	6	1000
---55	7	1000
---56	8	1000
---57	9	1000
---58	:	2000
---59	;	1500
---60	<	1000
---61	=	1000
---62	>	1000
---63	?	3000
---64	@	1000
---65	A	999
---...
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3707](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3707)
---* Corresponding plain TeX control sequence: [\sfcode](https://www.tug.org/utilities/plain/cseq.html#sfcode-rp)
---
---@type table<integer, integer>
tex.sfcode = {}

---
---Set the space factor code for a character globally.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1425-L1428](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1425-L1428)
---* Corresponding plain TeX control sequence: [\sfcode](https://www.tug.org/utilities/plain/cseq.html#sfcode-rp)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param char_code integer # A Unicode code point, for example `65` stands for an uppercase `A` and `97` for a lowercase `a`.
---@param space_factor integer # The code (between `0` and `32767`) to adjust the space factor in a horizontal list. The uppercase letters `A-Z` have space factor code `999`. Most other characters have code `1000`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setsfcode(global, char_code, space_factor) end

---
---Set the space factor code for a character.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1425-L1428](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1425-L1428)
---* Corresponding plain TeX control sequence: [\sfcode](https://www.tug.org/utilities/plain/cseq.html#sfcode-rp)
---
---@param char_code integer # A Unicode code point, for example `65` stands for an uppercase `A` and `97` for a lowercase `a`.
---@param space_factor integer # The code (between `0` and `32767`) to adjust the space factor in a horizontal list. The uppercase letters `A-Z` have space factor code `999`. Most other characters have code `1000`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setsfcode(char_code, space_factor) end

---
---Get the space factor code of a character.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1430-L1436](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1430-L1436)
---* Corresponding plain TeX control sequence: [\sfcode](https://www.tug.org/utilities/plain/cseq.html#sfcode-rp)
---
---@param char_code integer # A Unicode code point, for example `65` stands for an uppercase `A` and `97` for a lowercase `a`.
---
---@return integer space_factor # The code (between `0` and `32767`) to adjust the space factor in a horizontal list. The uppercase letters `A-Z` have space factor code `999`. Most other characters have code `1000`.
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getsfcode(char_code) end

---
---A virtual subtable of the `tex` table called `catcode` (category code) can be used to access and write to TeX's character code table.
---
---The category codes are indexed by the character code.
---
---```lua
---for i=33,126 do
---  print(i, utf8.char(i), tex.catcode[i])
---end
---```
---
---## Output:
---
---```
---33	!	12
---34	"	12
---35	#	6
---36	$	3
---37	%	14
---38	&	4
---39	'	12
---40	(	12
---41	)	12
---42	*	12
---43	+	12
---44	,	12
---45	-	12
---46	.	12
---47	/	12
---48	0	12
---49	1	12
---50	2	12
---51	3	12
---52	4	12
---53	5	12
---54	6	12
---55	7	12
---56	8	12
---57	9	12
---58	:	12
---59	;	12
---60	<	12
---61	=	12
---62	>	12
---63	?	12
---64	@	12
---65	A	11
---66	B	11
---67	C	11
---68	D	11
---69	E	11
---70	F	11
---71	G	11
---72	H	11
---73	I	11
---74	J	11
---75	K	11
---76	L	11
---77	M	11
---78	N	11
---79	O	11
---80	P	11
---81	Q	11
---82	R	11
---83	S	11
---84	T	11
---85	U	11
---86	V	11
---87	W	11
---88	X	11
---89	Y	11
---90	Z	11
---91	[	12
---92	\	0
---93	]	12
---94	^	7
---95	_	8
---96	`	12
---97	a	11
---98	b	11
---99	c	11
---100	d	11
---101	e	11
---102	f	11
---103	g	11
---104	h	11
---105	i	11
---106	j	11
---107	k	11
---108	l	11
---109	m	11
---110	n	11
---111	o	11
---112	p	11
---113	q	11
---114	r	11
---115	s	11
---116	t	11
---117	u	11
---118	v	11
---119	w	11
---120	x	11
---121	y	11
---122	z	11
---123	{	1
---124	|	12
---125	}	2
---126	~	13
---```
---
---## Category codes:
---
---* `0`: Escape character, normally `\`
---* `1`: Begin grouping, normally `{`
---* `2`: End grouping, normally `}`
---* `3`: Math shift, normally `$`
---* `4`: Alignment tab, normally `&`
---* `5`: End of line, normally `<return>`
---* `6`: Parameter, normally `#`
---* `7`: Superscript, normally `^`
---* `8`: Subscript, normally `_`
---* `9`: Ignored character, normally `<null>`
---* `10`: Space, normally `<space>` and `<tab>`
---* `11`: Letter, normally only contains the letters `a,...,z` and `A,...,Z`. These characters can be used in command names
---* `12`: Other, normally everything else not listed in the other categories
---* `13`: Active character, for example `~`
---* `14`: Comment character, normally `%`
---* `15`: Invalid character, normally `<delete>`
---
---* Corresponding C source code: [ltexlib.c#L3710](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3710)
---
---@type table<integer, integer>
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.catcode = {}

---
---Specify a
---category table to use on assignment or on query (default in both cases is the
---current one).
---
---* Corresponding C source code: [ltexlib.c#L1438-L1464](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1438-L1464)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param char_code integer # The character code, for example ASCII or UNICODE character code.
---@param cat_code integer # The category code (`0` stands for the escape character, normally `\`, `1` stands for begin grouping, normally `{` and so on).
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setcatcode(global, char_code, cat_code) end

---
---Specify a
---category table to use on assignment or on query (default in both cases is the
---current one).
---
---## Category codes:
---
---* `0`: Escape character, normally `\`
---* `1`: Begin grouping, normally `{`
---* `2`: End grouping, normally `}`
---* `3`: Math shift, normally `$`
---* `4`: Alignment tab, normally `&`
---* `5`: End of line, normally `<return>`
---* `6`: Parameter, normally `#`
---* `7`: Superscript, normally `^`
---* `8`: Subscript, normally `_`
---* `9`: Ignored character, normally `<null>`
---* `10`: Space, normally `<space>` and `<tab>`
---* `11`: Letter, normally only contains the letters `a,...,z` and `A,...,Z`. These characters can be used in command names
---* `12`: Other, normally everything else not listed in the other categories
---* `13`: Active character, for example `~`
---* `14`: Comment character, normally `%`
---* `15`: Invalid character, normally `<delete>`
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1438-L1464](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1438-L1464)
---
---@param char_code integer # The character code, for example ASCII or UNICODE character code.
---@param cat_code integer # The category code (`0` stands for the escape character, normally `\`, `1` stands for begin grouping, normally `{` and so on).
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setcatcode(char_code, cat_code) end

---
---Specify a
---category table to use on assignment or on query (default in both cases is the
---current one).
---
---## Category codes:
---
---* `0`: Escape character, normally `\`
---* `1`: Begin grouping, normally `{`
---* `2`: End grouping, normally `}`
---* `3`: Math shift, normally `$`
---* `4`: Alignment tab, normally `&`
---* `5`: End of line, normally `<return>`
---* `6`: Parameter, normally `#`
---* `7`: Superscript, normally `^`
---* `8`: Subscript, normally `_`
---* `9`: Ignored character, normally `<null>`
---* `10`: Space, normally `<space>` and `<tab>`
---* `11`: Letter, normally only contains the letters `a,...,z` and `A,...,Z`. These characters can be used in command names
---* `12`: Other, normally everything else not listed in the other categories
---* `13`: Active character, for example `~`
---* `14`: Comment character, normally `%`
---* `15`: Invalid character, normally `<delete>`
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1438-L1464](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1438-L1464)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param cat_table integer
---@param char_code integer # The character code, for example ASCII or UNICODE character code.
---@param cat_code integer # The category code (`0` stands for the escape character, normally `\`, `1` stands for begin grouping, normally `{` and so on).
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setcatcode(global, cat_table, char_code, cat_code) end

---
---Specify a
---category table to use on assignment or on query (default in both cases is the
---current one).
---
---## Category codes:
---
---* `0`: Escape character, normally `\`
---* `1`: Begin grouping, normally `{`
---* `2`: End grouping, normally `}`
---* `3`: Math shift, normally `$`
---* `4`: Alignment tab, normally `&`
---* `5`: End of line, normally `<return>`
---* `6`: Parameter, normally `#`
---* `7`: Superscript, normally `^`
---* `8`: Subscript, normally `_`
---* `9`: Ignored character, normally `<null>`
---* `10`: Space, normally `<space>` and `<tab>`
---* `11`: Letter, normally only contains the letters `a,...,z` and `A,...,Z`. These characters can be used in command names
---* `12`: Other, normally everything else not listed in the other categories
---* `13`: Active character, for example `~`
---* `14`: Comment character, normally `%`
---* `15`: Invalid character, normally `<delete>`
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1438-L1464](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1438-L1464)
---
---@param cat_table integer
---@param char_code integer # The character code, for example ASCII or UNICODE character code.
---@param cat_code integer # The category code (`0` stands for the escape character, normally `\`, `1` stands for begin grouping, normally `{` and so on).
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setcatcode(global, cat_table, char_code, cat_code) end

---
---The function call interface for `catcode` (category code) also allows you to specify a
---category table to use on assignment or on query (default in both cases is the
---current one):
---
---## Category codes:
---
---* `0`: Escape character, normally `\`
---* `1`: Begin grouping, normally `{`
---* `2`: End grouping, normally `}`
---* `3`: Math shift, normally `$`
---* `4`: Alignment tab, normally `&`
---* `5`: End of line, normally `<return>`
---* `6`: Parameter, normally `#`
---* `7`: Superscript, normally `^`
---* `8`: Subscript, normally `_`
---* `9`: Ignored character, normally `<null>`
---* `10`: Space, normally `<space>` and `<tab>`
---* `11`: Letter, normally only contains the letters `a,...,z` and `A,...,Z`. These characters can be used in command names
---* `12`: Other, normally everything else not listed in the other categories
---* `13`: Active character, for example `~`
---* `14`: Comment character, normally `%`
---* `15`: Invalid character, normally `<delete>`
---
---@param cat_table integer
---@param char_code integer # The character code, for example ASCII or UNICODE character code.
---
---@return integer cat_code # The category code (`0` stands for the escape character, normally `\`, `1` stands for begin grouping, normally `{` and so on).
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getcatcode(cat_table, char_code) end

---
---The *TeX*'s character code table `mathcode` can be accessed and written to using
---a virtual subtable of the `tex` table.
---
---```lua
---for i = 1, 128 do
---  local mathcode = tex.mathcode[i]
---  print(i, utf8.char(i), mathcode[1], mathcode[2], mathcode[3])
---end
---```
---
---__Output:__
---
---```
---1		3	2	35
---2		0	1	11
---3		0	1	12
---4		2	2	94
---5		0	2	58
---6		3	2	50
---7		0	1	25
---8		0	1	21
---9		0	1	13
---10		0	1	14
---11		3	2	34
---12		2	2	6
---1		2	2	8
---14		0	2	49
---15		0	1	64
---16		3	2	26
---17		3	2	27
---18		2	2	92
---19		2	2	91
---20		0	2	56
---21		0	2	57
---22		2	2	10
---23		3	2	36
---24		3	2	32
---25		3	2	33
---26	�	8	0	0
---27			2	5
---28		3	2	20
---29		3	2	21
---30		3	2	17
---31		2	2	95
---32	 	8	0	0
---33	!	5	0	33
---34	"	0	0	34
---35	#	0	0	35
---36	$	0	0	36
---37	%	0	0	37
---38	&	0	0	38
---39	'	8	0	0
---40	(	4	0	40
---41	)	5	0	41
---42	*	2	2	3
---43	+	2	0	43
---44	,	6	1	59
---45	-	2	2	0
---46	.	0	1	58
---47	/	0	1	61
---48	0	7	0	48
---49	1	7	0	49
---50	2	7	0	50
---51	3	7	0	51
---52	4	7	0	52
---53	5	7	0	53
---54	6	7	0	54
---55	7	7	0	55
---56	8	7	0	56
---57	9	7	0	57
---58	:	3	0	58
---59	;	6	0	59
---60	<	3	1	60
---61	=	3	0	61
---62	>	3	1	62
---63	?	5	0	63
---64	@	0	0	64
---65	A	7	1	65
---66	B	7	1	66
---67	C	7	1	67
---68	D	7	1	68
---69	E	7	1	69
---70	F	7	1	70
---71	G	7	1	71
---72	H	7	1	72
---73	I	7	1	73
---74	J	7	1	74
---75	K	7	1	75
---76	L	7	1	76
---77	M	7	1	77
---78	N	7	1	78
---79	O	7	1	79
---80	P	7	1	80
---81	Q	7	1	81
---82	R	7	1	82
---83	S	7	1	83
---84	T	7	1	84
---85	U	7	1	85
---86	V	7	1	86
---87	W	7	1	87
---88	X	7	1	88
---89	Y	7	1	89
---90	Z	7	1	90
---91	[	4	0	91
---92	\	0	2	110
---93	]	5	0	93
---94	^	0	0	94
---95	_	8	0	0
---96	`	0	0	96
---97	a	7	1	97
---98	b	7	1	98
---99	c	7	1	99
---100	d	7	1	100
---101	e	7	1	101
---102	f	7	1	102
---103	g	7	1	103
---104	h	7	1	104
---105	i	7	1	105
---106	j	7	1	106
---107	k	7	1	107
---108	l	7	1	108
---109	m	7	1	109
---110	n	7	1	110
---111	o	7	1	111
---112	p	7	1	112
---113	q	7	1	113
---114	r	7	1	114
---115	s	7	1	115
---116	t	7	1	116
---117	u	7	1	117
---118	v	7	1	118
---119	w	7	1	119
---120	x	7	1	120
---121	y	7	1	121
---122	z	7	1	122
---123	{	4	2	102
---124	|	0	2	106
---125	}	5	2	103
---126	~	0	0	126
---127		1	2	115
---128		0	0	128
---```
---
---In math mode, the math atoms require more structure. Each symbol
---originates from a different font and receives different spacing
---based on its class (operator, binary infix, relation, etc.).
---Following the typical style of the 1970s, these properties are
---compactly packed into bit fields within a single integer called a
---mathcode. The mathcode is usually expressed in hexadecimal so
---that the fields can easily be pulled apart. The mathcode of `+` in
---plain TeX is set as:
---
---```tex
---\mathcode`\+="202B
---```
---
---This means it is class 2 (binary infix) and fam0 (the Roman font).
---The character hex is `2B`, which is the decimal `43`, the character
---code for `+` in the Roman font encoding.
---
---The `mathcode` section in the
---[plain.tex](https://mirrors.ctan.org/macros/plain/base/plain.tex)
---format file:
---
---```tex
---\mathcode`\^^?="1273 % \smallint
---% INITEX sets up \mathcode x=x, for x=0..255, except that
---% \mathcode x=x+"7100, for x = `A to `Z and `a to `z;
---% \mathcode x=x+"7000, for x = `0 to `9.
---% The following changes define internal codes as recommended
---% in Appendix C of The TeXbook:
---\mathcode`\^^@="2201 % \cdot
---\mathcode`\^^A="3223 % \downarrow
---\mathcode`\^^B="010B % \alpha
---\mathcode`\^^C="010C % \beta
---\mathcode`\^^D="225E % \land
---\mathcode`\^^E="023A % \lnot
---\mathcode`\^^F="3232 % \in
---\mathcode`\^^G="0119 % \pi
---\mathcode`\^^H="0115 % \lambda
---\mathcode`\^^I="010D % \gamma
---\mathcode`\^^J="010E % \delta
---\mathcode`\^^K="3222 % \uparrow
---\mathcode`\^^L="2206 % \pm
---\mathcode`\^^M="2208 % \oplus
---\mathcode`\^^N="0231 % \infty
---\mathcode`\^^O="0140 % \partial
---\mathcode`\^^P="321A % \subset
---\mathcode`\^^Q="321B % \supset
---\mathcode`\^^R="225C % \cap
---\mathcode`\^^S="225B % \cup
---\mathcode`\^^T="0238 % \forall
---\mathcode`\^^U="0239 % \exists
---\mathcode`\^^V="220A % \otimes
---\mathcode`\^^W="3224 % \leftrightarrow
---\mathcode`\^^X="3220 % \leftarrow
---\mathcode`\^^Y="3221 % \rightarrow
---\mathcode`\^^Z="8000 % \ne
---\mathcode`\^^[="2205 % \diamond
---\mathcode`\^^\="3214 % \le
---\mathcode`\^^]="3215 % \ge
---\mathcode`\^^^="3211 % \equiv
---\mathcode`\^^_="225F % \lor
---\mathcode`\ ="8000 % \space
---\mathcode`\!="5021
---\mathcode`\'="8000 % ^\prime
---\mathcode`\(="4028
---\mathcode`\)="5029
---\mathcode`\*="2203 % \ast
---\mathcode`\+="202B
---\mathcode`\,="613B
---\mathcode`\-="2200
---\mathcode`\.="013A
---\mathcode`\/="013D
---\mathcode`\:="303A
---\mathcode`\;="603B
---\mathcode`\<="313C
---\mathcode`\=="303D
---\mathcode`\>="313E
---\mathcode`\?="503F
---\mathcode`\[="405B
---\mathcode`\\="026E % \backslash
---\mathcode`\]="505D
---\mathcode`\_="8000 % \_
---\mathcode`\{="4266
---\mathcode`\|="026A
---\mathcode`\}="5267
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3711](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3711)
---* Corresponding plain TeX control sequence: [\mathcode](https://www.tug.org/utilities/plain/cseq.html#mathcode-rp)
---* [tex.stackexchange.com](https://tex.stackexchange.com/a/109440)
---
---@type table<integer, MathCode>
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.mathcode = {}

---
---The table for `mathcode` is an array of 3 integers, like this:
---
---```lua
---{
---    -- class
---    -- family
---    -- character
---}
---```
---
---@alias MathCode integer[]

---
---__class__:
---
---| Class | Meaning          | Example |
---|-------|------------------|---------|
---| 0     | Ordinary         | `/`     |
---| 1     | Large operator   | `\sum`  |
---| 2     | Binary operation | `+`     |
---| 3     | Relation         | `=`     |
---| 4     | Opening          | `(`     |
---| 5     | Closing          | `)`     |
---| 6     | Punctuation      | `,`     |
---| 7     | Variable family  | `x`     |
---
---__family__:
---
---```tex
---% Family 0 (Roman)
---\font\tenrm=mdbchr7t at10pt
---\font\sevenrm=mdbchr7t at7pt
---\font\fiverm=mdbchr7t at5pt
---\textfont0=\tenrm
---\scriptfont0=\sevenrm
---27
---\scriptscriptfont0=\fiverm
---\def\rm{\fam=0 \tenrm}
---%
---% Family 1 (Math italic)
---\font\teni=mdbchri7m at10pt
---\font\seveni=mdbchri7m at7pt
---\font\fivei=mdbchri7m at5pt
---\textfont1=\teni
---\scriptfont1=\seveni
---\scriptscriptfont1=\fivei
---\def\mit{\fam=1}
---%
---% Family 2 (Math symbols)
---\font\tensy=md-chr7y at10pt
---\font\sevensy=md-chr7y at7pt
---\font\fivesy=md-chr7y at5pt
---\textfont2=\tensy
---\scriptfont2=\
---% Family 3 (Math extension)
---\font\tenex=mdbchr7v at10pt
---\font\sevenex=mdbchr7v at7pt
---\font\fiveex=mdbchr7v at5pt
---\textfont3=\tenex
---\scriptfont3=\sevenex
---\scriptscriptfont3=\fiveex
---%
---% Family 4 (Italic text)
---\font\tenit=mdbchri7t at10pt
---\font\sevenit=mdbchri7t at7pt
---\font\fiveit=mdbchri7t at5pt
---\textfont\itfam=\tenit
---\scriptfont\itfam=\sevenit
---\scriptscriptfont\itfam=\fiveit
---\def\it{\fam=\itfam \tenit}
---%
---% Family 5 (Slanted text)
---\font\tensl=mdbchro7t at10pt
---\font\sevensl=mdbchro7t at7pt
---\font\fivesl=mdbchro7t at5pt
---\textfont\slfam=\tensl
---\scriptfont\slfam=\sevensl
---\scriptscriptfont\slfam=\fivesl
---\def\sl{\fam=\slfam \tensl}
---%
---% Family 6 (Bold text)
---\font\tenbf=mdbchb7t at10pt
---\font\sevenbf=mdbchb7t at7pt
---\font\fivebf=mdbchb7t at5pt
---\textfont\bffam=\tenbf
---\scriptfont\bffam=\sevenbf
---\scriptscriptfont\bffam=\fivebf
---\def\bf{\fam=\bffam \tenbf}
---%
---% Family 7 (Typewriter)
---\font\tentt=cmtt10---%
---\rm
---% Sets normal roman font
---\font\seventt=cmtt10 at7pt
---\font\fivett=cmtt10 at5pt
---\textfont\ttfam=\tentt
---\scriptfont\ttfam=\seventt
---\scriptscriptfont\ttfam=\fivett
---\def\tt{\fam=\ttfam \tentt}
---```sevensy
---\scriptscriptfont2=\fivesy
---\def\cal{\fam=2}
---%
---% Family 3 (Math extension)
---\font\tenex=mdbchr7v at10pt
---\font\sevenex=mdbchr7v at7pt
---\font\fiveex=mdbchr7v at5pt
---\textfont3=\tenex
---\scriptfont3=\sevenex
---\scriptscriptfont3=\fiveex
---%
---% Family 4 (Italic text)
---\font\tenit=mdbchri7t at10pt
---\font\sevenit=mdbchri7t at7pt
---\font\fiveit=mdbchri7t at5pt
---\textfont\itfam=\tenit
---\scriptfont\itfam=\sevenit
---\scriptscriptfont\itfam=\fiveit
---\def\it{\fam=\itfam \tenit}
---%
---% Family 5 (Slanted text)
---\font\tensl=mdbchro7t at10pt
---\font\sevensl=mdbchro7t at7pt
---\font\fivesl=mdbchro7t at5pt
---\textfont\slfam=\tensl
---\scriptfont\slfam=\sevensl
---\scriptscriptfont\slfam=\fivesl
---\def\sl{\fam=\slfam \tensl}
---%
---% Family 6 (Bold text)
---\font\tenbf=mdbchb7t at10pt
---\font\sevenbf=mdbchb7t at7pt
---\font\fivebf=mdbchb7t at5pt
---\textfont\bffam=\tenbf
---\scriptfont\bffam=\sevenbf
---\scriptscriptfont\bffam=\fivebf
---\def\bf{\fam=\bffam \tenbf}
---%
---% Family 7 (Typewriter)
---\font\tentt=cmtt10---%
---\rm
---% Sets normal roman font
---\font\seventt=cmtt10 at7pt
---\font\fivett=cmtt10 at5pt
---\textfont\ttfam=\tentt
---\scriptfont\ttfam=\seventt
---\scriptscriptfont\ttfam=\fivett
---\def\tt{\fam=\ttfam \tentt}
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1524-1561](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1524-1561)
---* Corresponding plain TeX control sequence: [\mathcode](https://www.tug.org/utilities/plain/cseq.html#mathcode-rp)
---* Donald E. Knuth: The TeXbook, Page 154
---* TUGboat, Volume 31 (2010), No. 1, Mathematical typefaces in TEX documents, Amit Raj Dhawan
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param char_code integer  # The ASCII or UNICODE charcater code point.
---@param class integer # The class to which a math character belongs (`0`: Ordinary, `1`: Large operator, `2`: Binary operation, `3`: Relation, `4`: Opening, `5`: Closing, `6`: Punctuation, `7`: Variable family).
---@param family integer # TeX uses fonts from one or more of the sixteen font families to typeset mathematical characters. Each font family consists of three fonts — textfont, scriptfont, and scriptscriptfont. (`0`: Roman, `1`: Math italic, `2`: Math symbol, `3`: Math extension, `4`: Italic text, `5`: Slanted text, `6`: Bold text, `7`: Typewriter)
---@param character integer # The character position
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setmathcode(global, char_code, class, family, character) end

---
---__class__:
---
---| Class | Meaning          | Example |
---|-------|------------------|---------|
---| 0     | Ordinary         | `/`     |
---| 1     | Large operator   | `\sum`  |
---| 2     | Binary operation | `+`     |
---| 3     | Relation         | `=`     |
---| 4     | Opening          | `(`     |
---| 5     | Closing          | `)`     |
---| 6     | Punctuation      | `,`     |
---| 7     | Variable family  | `x`     |
---
---__family__:
---
---```tex
---% Family 0 (Roman)
---\font\tenrm=mdbchr7t at10pt
---\font\sevenrm=mdbchr7t at7pt
---\font\fiverm=mdbchr7t at5pt
---\textfont0=\tenrm
---\scriptfont0=\sevenrm
---27
---\scriptscriptfont0=\fiverm
---\def\rm{\fam=0 \tenrm}
---%
---% Family 1 (Math italic)
---\font\teni=mdbchri7m at10pt
---\font\seveni=mdbchri7m at7pt
---\font\fivei=mdbchri7m at5pt
---\textfont1=\teni
---\scriptfont1=\seveni
---\scriptscriptfont1=\fivei
---\def\mit{\fam=1}
---%
---% Family 2 (Math symbols)
---\font\tensy=md-chr7y at10pt
---\font\sevensy=md-chr7y at7pt
---\font\fivesy=md-chr7y at5pt
---\textfont2=\tensy
---\scriptfont2=\
---% Family 3 (Math extension)
---\font\tenex=mdbchr7v at10pt
---\font\sevenex=mdbchr7v at7pt
---\font\fiveex=mdbchr7v at5pt
---\textfont3=\tenex
---\scriptfont3=\sevenex
---\scriptscriptfont3=\fiveex
---%
---% Family 4 (Italic text)
---\font\tenit=mdbchri7t at10pt
---\font\sevenit=mdbchri7t at7pt
---\font\fiveit=mdbchri7t at5pt
---\textfont\itfam=\tenit
---\scriptfont\itfam=\sevenit
---\scriptscriptfont\itfam=\fiveit
---\def\it{\fam=\itfam \tenit}
---%
---% Family 5 (Slanted text)
---\font\tensl=mdbchro7t at10pt
---\font\sevensl=mdbchro7t at7pt
---\font\fivesl=mdbchro7t at5pt
---\textfont\slfam=\tensl
---\scriptfont\slfam=\sevensl
---\scriptscriptfont\slfam=\fivesl
---\def\sl{\fam=\slfam \tensl}
---%
---% Family 6 (Bold text)
---\font\tenbf=mdbchb7t at10pt
---\font\sevenbf=mdbchb7t at7pt
---\font\fivebf=mdbchb7t at5pt
---\textfont\bffam=\tenbf
---\scriptfont\bffam=\sevenbf
---\scriptscriptfont\bffam=\fivebf
---\def\bf{\fam=\bffam \tenbf}
---%
---% Family 7 (Typewriter)
---\font\tentt=cmtt10---%
---\rm
---% Sets normal roman font
---\font\seventt=cmtt10 at7pt
---\font\fivett=cmtt10 at5pt
---\textfont\ttfam=\tentt
---\scriptfont\ttfam=\seventt
---\scriptscriptfont\ttfam=\fivett
---\def\tt{\fam=\ttfam \tentt}
---```sevensy
---\scriptscriptfont2=\fivesy
---\def\cal{\fam=2}
---%
---% Family 3 (Math extension)
---\font\tenex=mdbchr7v at10pt
---\font\sevenex=mdbchr7v at7pt
---\font\fiveex=mdbchr7v at5pt
---\textfont3=\tenex
---\scriptfont3=\sevenex
---\scriptscriptfont3=\fiveex
---%
---% Family 4 (Italic text)
---\font\tenit=mdbchri7t at10pt
---\font\sevenit=mdbchri7t at7pt
---\font\fiveit=mdbchri7t at5pt
---\textfont\itfam=\tenit
---\scriptfont\itfam=\sevenit
---\scriptscriptfont\itfam=\fiveit
---\def\it{\fam=\itfam \tenit}
---%
---% Family 5 (Slanted text)
---\font\tensl=mdbchro7t at10pt
---\font\sevensl=mdbchro7t at7pt
---\font\fivesl=mdbchro7t at5pt
---\textfont\slfam=\tensl
---\scriptfont\slfam=\sevensl
---\scriptscriptfont\slfam=\fivesl
---\def\sl{\fam=\slfam \tensl}
---%
---% Family 6 (Bold text)
---\font\tenbf=mdbchb7t at10pt
---\font\sevenbf=mdbchb7t at7pt
---\font\fivebf=mdbchb7t at5pt
---\textfont\bffam=\tenbf
---\scriptfont\bffam=\sevenbf
---\scriptscriptfont\bffam=\fivebf
---\def\bf{\fam=\bffam \tenbf}
---%
---% Family 7 (Typewriter)
---\font\tentt=cmtt10---%
---\rm
---% Sets normal roman font
---\font\seventt=cmtt10 at7pt
---\font\fivett=cmtt10 at5pt
---\textfont\ttfam=\tentt
---\scriptfont\ttfam=\seventt
---\scriptscriptfont\ttfam=\fivett
---\def\tt{\fam=\ttfam \tentt}
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1524-L1561](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1524-L1561)
---* Corresponding plain TeX control sequence: [\mathcode](https://www.tug.org/utilities/plain/cseq.html#mathcode-rp)
---* Donald E. Knuth: The TeXbook, Page 154
---* TUGboat, Volume 31 (2010), No. 1, Mathematical typefaces in TEX documents, Amit Raj Dhawan
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param char_code integer # The ASCII or UNICODE charcater code point.
---@param class integer # The class to which a math character belongs (`0`: Ordinary, `1`: Large operator, `2`: Binary operation, `3`: Relation, `4`: Opening, `5`: Closing, `6`: Punctuation, `7`: Variable family).
---@param family integer # TeX uses fonts from one or more of the sixteen font families to typeset mathematical characters. Each font family consists of three fonts — textfont, scriptfont, and scriptscriptfont. (`0`: Roman, `1`: Math italic, `2`: Math symbol, `3`: Math extension, `4`: Italic text, `5`: Slanted text, `6`: Bold text, `7`: Typewriter)
---@param character integer # The character position
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setmathcode(global, char_code, class, family, character) end

---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1524-L1561](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1524-L1561)
---* Corresponding plain TeX control sequence: [\mathcode](https://www.tug.org/utilities/plain/cseq.html#mathcode-rp)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param char_code integer # The ASCII or UNICODE charcater code point.
---@param math_code MathCode # A table with three integers (class, family, character).
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setmathcode(global, char_code, math_code) end

---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1524-L1561](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1524-L1561)
---* Corresponding plain TeX control sequence: [\mathcode](https://www.tug.org/utilities/plain/cseq.html#mathcode-rp)
---
---@param char_code integer  # The ASCII or UNICODE charcater code point.
---@param math_code MathCode # A table with three integers (class, family, character).
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setmathcode(char_code, math_code) end

---
---Retrieve the math code of a character as a table with three integers (class, family, character).
---
---__Examples from The TeXBook, page 154__:
---
---`\mathcode‘<="313C`
---
---* ASCII `<`: 60 (decimal) 3C (hexadecimal)
---* class: 3
---* family: 1
---* character: 60
---
---```lua
---local mathcode = tex.getmathcode(60)
---print(mathcode[1], mathcode[2], mathcode[3]) -- 3 1 60
---```
---
---`\mathcode‘*="2203`
---
---* ASCII `*`: 42 (decimal) 2A (hexadecimal)
---* class: 2
---* family: 2
---* character: 3
---
---```lua
---local mathcode = tex.getmathcode(42)
---print(mathcode[1], mathcode[2], mathcode[3]) -- 2 2 3
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1563-L1577](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1563-L1577)
---* Corresponding plain TeX control sequence: [\mathcode](https://www.tug.org/utilities/plain/cseq.html#mathcode-rp)
---
---@param char_code integer
---
---@return MathCode math_code # A table with three integers (class, family, character).
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getmathcode(char_code) end

---
---Retrieve the math code of a character as three integers (class, family, character).
---
---__Examples from The TeXBook, page 154__:
---
---`\mathcode‘<="313C`
---
---* ASCII `<`: 60 (decimal) 3C (hexadecimal)
---* class: 3
---* family: 1
---* character: 60
---
---```lua
---print(tex.getmathcodes(60)) -- 3 1 60
---```
---
---`\mathcode‘*="2203`
---
---* ASCII `*`: 42 (decimal) 2A (hexadecimal)
---* class: 2
---* family: 2
---* character: 3
---
---```lua
---print(tex.getmathcodes(42)) -- 2 2 3
---```
---
---__class__:
---
---| Class | Meaning          | Example |
---|-------|------------------|---------|
---| 0     | Ordinary         | `/`     |
---| 1     | Large operator   | `\sum`  |
---| 2     | Binary operation | `+`     |
---| 3     | Relation         | `=`     |
---| 4     | Opening          | `(`     |
---| 5     | Closing          | `)`     |
---| 6     | Punctuation      | `,`     |
---| 7     | Variable family  | `x`     |
---
---__family__:
---
---```tex
---% Family 0 (Roman)
---\font\tenrm=mdbchr7t at10pt
---\font\sevenrm=mdbchr7t at7pt
---\font\fiverm=mdbchr7t at5pt
---\textfont0=\tenrm
---\scriptfont0=\sevenrm
---27
---\scriptscriptfont0=\fiverm
---\def\rm{\fam=0 \tenrm}
---%
---% Family 1 (Math italic)
---\font\teni=mdbchri7m at10pt
---\font\seveni=mdbchri7m at7pt
---\font\fivei=mdbchri7m at5pt
---\textfont1=\teni
---\scriptfont1=\seveni
---\scriptscriptfont1=\fivei
---\def\mit{\fam=1}
---%
---% Family 2 (Math symbols)
---\font\tensy=md-chr7y at10pt
---\font\sevensy=md-chr7y at7pt
---\font\fivesy=md-chr7y at5pt
---\textfont2=\tensy
---\scriptfont2=\
---% Family 3 (Math extension)
---\font\tenex=mdbchr7v at10pt
---\font\sevenex=mdbchr7v at7pt
---\font\fiveex=mdbchr7v at5pt
---\textfont3=\tenex
---\scriptfont3=\sevenex
---\scriptscriptfont3=\fiveex
---%
---% Family 4 (Italic text)
---\font\tenit=mdbchri7t at10pt
---\font\sevenit=mdbchri7t at7pt
---\font\fiveit=mdbchri7t at5pt
---\textfont\itfam=\tenit
---\scriptfont\itfam=\sevenit
---\scriptscriptfont\itfam=\fiveit
---\def\it{\fam=\itfam \tenit}
---%
---% Family 5 (Slanted text)
---\font\tensl=mdbchro7t at10pt
---\font\sevensl=mdbchro7t at7pt
---\font\fivesl=mdbchro7t at5pt
---\textfont\slfam=\tensl
---\scriptfont\slfam=\sevensl
---\scriptscriptfont\slfam=\fivesl
---\def\sl{\fam=\slfam \tensl}
---%
---% Family 6 (Bold text)
---\font\tenbf=mdbchb7t at10pt
---\font\sevenbf=mdbchb7t at7pt
---\font\fivebf=mdbchb7t at5pt
---\textfont\bffam=\tenbf
---\scriptfont\bffam=\sevenbf
---\scriptscriptfont\bffam=\fivebf
---\def\bf{\fam=\bffam \tenbf}
---%
---% Family 7 (Typewriter)
---\font\tentt=cmtt10---%
---\rm
---% Sets normal roman font
---\font\seventt=cmtt10 at7pt
---\font\fivett=cmtt10 at5pt
---\textfont\ttfam=\tentt
---\scriptfont\ttfam=\seventt
---\scriptscriptfont\ttfam=\fivett
---\def\tt{\fam=\ttfam \tentt}
---```sevensy
---\scriptscriptfont2=\fivesy
---\def\cal{\fam=2}
---%
---% Family 3 (Math extension)
---\font\tenex=mdbchr7v at10pt
---\font\sevenex=mdbchr7v at7pt
---\font\fiveex=mdbchr7v at5pt
---\textfont3=\tenex
---\scriptfont3=\sevenex
---\scriptscriptfont3=\fiveex
---%
---% Family 4 (Italic text)
---\font\tenit=mdbchri7t at10pt
---\font\sevenit=mdbchri7t at7pt
---\font\fiveit=mdbchri7t at5pt
---\textfont\itfam=\tenit
---\scriptfont\itfam=\sevenit
---\scriptscriptfont\itfam=\fiveit
---\def\it{\fam=\itfam \tenit}
---%
---% Family 5 (Slanted text)
---\font\tensl=mdbchro7t at10pt
---\font\sevensl=mdbchro7t at7pt
---\font\fivesl=mdbchro7t at5pt
---\textfont\slfam=\tensl
---\scriptfont\slfam=\sevensl
---\scriptscriptfont\slfam=\fivesl
---\def\sl{\fam=\slfam \tensl}
---%
---% Family 6 (Bold text)
---\font\tenbf=mdbchb7t at10pt
---\font\sevenbf=mdbchb7t at7pt
---\font\fivebf=mdbchb7t at5pt
---\textfont\bffam=\tenbf
---\scriptfont\bffam=\sevenbf
---\scriptscriptfont\bffam=\fivebf
---\def\bf{\fam=\bffam \tenbf}
---%
---% Family 7 (Typewriter)
---\font\tentt=cmtt10---%
---\rm
---% Sets normal roman font
---\font\seventt=cmtt10 at7pt
---\font\fivett=cmtt10 at5pt
---\textfont\ttfam=\tentt
---\scriptfont\ttfam=\seventt
---\scriptscriptfont\ttfam=\fivett
---\def\tt{\fam=\ttfam \tentt}
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1579-L1589](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1579-L1589)
---* Corresponding plain TeX control sequence: [\mathcode](https://www.tug.org/utilities/plain/cseq.html#mathcode-rp)
---* Donald E. Knuth: The TeXbook, Page 154
---* TUGboat, Volume 31 (2010), No. 1, Mathematical typefaces in TEX documents, Amit Raj Dhawan
---
---@param char_code integer
---
---@return integer class # The class to which a math character belongs (`0`: Ordinary, `1`: Large operator, `2`: Binary operation, `3`: Relation, `4`: Opening, `5`: Closing, `6`: Punctuation, `7`: Variable family).
---@return integer family # TeX uses fonts from one or more of the sixteen font families to typeset mathematical characters. Each font family consists of three fonts — textfont, scriptfont, and scriptscriptfont. (`0`: Roman, `1`: Math italic, `2`: Math symbol, `3`: Math extension, `4`: Italic text, `5`: Slanted text, `6`: Bold text, `7`: Typewriter)
---@return integer character # The character position
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getmathcodes(char_code) end

---
---The *TeX*'s character code tables `delcode` (delimiter code) can be accessed and written to using
---a virtual subtable of the `tex` table.
---

---
---```lua
---for i=1,128 do
---  local delcode = tex.delcode[i]
---  print(i, utf8.char(i), delcode[1], delcode[2], delcode[3], delcode[4])
---end
---```
---
---Output:
---
---```
---1		-1	0	0	0
---...
---40	(	0	40	3	0
---41	)	0	41	3	1
---...
---47	/	0	47	3	14
---...
---60	<	2	104	3	10
---61	=	-1	0	0	0
---62	>	2	105	3	11
---...
---91	[	0	91	3	2
---92	\	2	110	3	15
---93	]	0	93	3	3
---...
---124	|	2	106	3	12
---...
---```
---
---The `delcode` section in the
---[plain.tex](https://mirrors.ctan.org/macros/plain/base/plain.tex)
---format file:
---
---```tex
---% Finally, INITEX sets all \delcode values to -1, except \delcode`.=0
---\delcode`\(="028300
---\delcode`\)="029301
---\delcode`\[="05B302
---\delcode`\]="05D303
---\delcode`\<="26830A
---\delcode`\>="26930B
---\delcode`\/="02F30E
---\delcode`\|="26A30C
---\delcode`\\="26E30F
---% N.B. { and } should NOT get delcodes; otherwise parameter grouping fails!
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3712](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3712)
---* Corresponding plain TeX control sequence: [\delcode](https://www.tug.org/utilities/plain/cseq.html#delcode-rp)
---
---@type table<integer, DelCode>
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.delcode = {}

---
---The table for `delcode` (delimiter code) is an array with 4 numbers, like this:
---
---```lua
---{
---    -- small_family
---    -- small_character
---    -- large_family
---    -- large_character
---}
---```
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias DelCode integer[]

---
----__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1640-L1681](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1640-L1681)
---* Corresponding plain TeX control sequence: [\delcode](https://www.tug.org/utilities/plain/cseq.html#delcode-rp)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param char_code integer
---@param small_family integer
---@param small_character integer
---@param large_family integer
---@param large_character integer
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setdelcode(global, char_code, small_family, small_character, large_family, large_character) end

---
----__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1640-L1681](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1640-L1681)
---* Corresponding plain TeX control sequence: [\delcode](https://www.tug.org/utilities/plain/cseq.html#delcode-rp)
---
---@param char_code integer
---@param small_family integer
---@param small_character integer
---@param large_family integer
---@param large_character integer
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setdelcode(char_code, small_family, small_character, large_family, large_character) end

---
---* Corresponding C source code: [ltexlib.c#L1640-L1681](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1640-L1681)
---* Corresponding plain TeX control sequence: [\delcode](https://www.tug.org/utilities/plain/cseq.html#delcode-rp)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param char_code integer
---@param del_code DelCode
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setdelcode(global, char_code, del_code) end

---
----__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1640-L1681](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1640-L1681)
---* Corresponding plain TeX control sequence: [\delcode](https://www.tug.org/utilities/plain/cseq.html#delcode-rp)
---
---@param char_code integer
---@param del_code DelCode
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setdelcode(char_code, del_code) end

---
---* Corresponding C source code: [ltexlib.c#L1701-L1712](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1701-L1712)
---* Corresponding plain TeX control sequence: [\delcode](https://www.tug.org/utilities/plain/cseq.html#delcode-rp)
---
---@param char_code integer
---
---@return integer small_family
---@return integer small_character
---@return integer large_family
---@return integer large_character
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getdelcodes(char_code) end

---
----__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1683-L1699](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1683-L1699)
---* Corresponding plain TeX control sequence: [\delcode](https://www.tug.org/utilities/plain/cseq.html#delcode-rp)
---
---@param char_code integer
---
---@return DelCode
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getdelcode(char_code) end

---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3706](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3706)
---* Corresponding plain TeX control sequence: [\box](https://www.tug.org/utilities/plain/cseq.html#box-rp)
---
---@type table<integer, Node>
tex.box = {}

---
---Set a box, coming for instance from `hbox`, `vbox` or `vtop`.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1352-L1362](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1352-L1362)
---* Corresponding plain TeX control sequence: [\setbox](https://www.tug.org/utilities/plain/cseq.html#setbox-rp)
---
---@param global 'global' # It is possible to define values globally by using the string `global` as the first function argument.
---@param register integer # A box register number (0 to 65535).
---@param head Node # A `hlist` or `vlist` node.
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setbox(global, register, head) end

---
---Set a box, coming for instance from `hbox`, `vbox` or `vtop`.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1352-L1362](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1352-L1362)
---* Corresponding plain TeX control sequence: [\setbox](https://www.tug.org/utilities/plain/cseq.html#setbox-rp)
---
---@param register integer # A box register number (0 to 65535).
---@param head Node # A `hlist` or `vlist` node.
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setbox(register, head) end

---
---Query an actual box, coming for instance from `hbox`, `vbox` or `vtop`.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1272-L1280](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1272-L1280)
---* Corresponding plain TeX control sequence: [\box](https://www.tug.org/utilities/plain/cseq.html#box-rp)
---
---@param register integer|string # A box register number (0 to 65535) or a cs name (for example `\newbox\MyBox`: `MyBox`)
---
---@return Node|nil head
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getbox(register) end

---
---Check if the given integer is a valid box register number.
---
---__Example:__
---
---```lua
---print(tex.isbox(65535)) -- true
---print(tex.isbox(65536)) -- false
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L1310-L1315](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1310-L1315)
---
---@param register integer|string # A box register number (0 to 65535) or a cs name (for example `\newbox\MyBox`: `MyBox`)
---
---@return boolean
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.isbox(register) end

---
---Register a box for reuse (this is modelled after so
---called xforms in *PDF*). You can (re)use the box with `useboxresource` or
---by creating a rule node with subtype 2.
---
---* Corresponding C source code: [ltexlib.c#L3217-L3278](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3217-L3278)
---
---@see tex.useboxresource
---
---@param n Node|integer # Instead of a box number one can also pass a `[h|v]list` node.
---@param attributes string
---@param resources string
---@param immediate boolean
---@param type integer # When set to non-zero the `/Type` entry is omitted. A value of 1 or 3 still writes a `/BBox`, while 2 or 3 will write a `/Matrix`.
---@param margin integer # The (virtual) margin that extends beyond the effective boundingbox as seen by *TeX*.
---
---@return integer index
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.saveboxresource(n, attributes, resources, immediate, type, margin) end

---
---* Corresponding Lua source code: [ltexlib.c#L3346-L3355](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3346-L3355)
---
---@param n integer
---
---@return Node
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getboxresourcebox(n) end

---
---Generate the reference (a rule type).
---
---The dimensions are optional and the final ones are returned as extra values.
---
---* Corresponding C source code: [ltexlib.c#L3280-L3325](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3280-L3325)
---
---@see tex.saveboxresource
---
---@param n integer
---@param width? integer
---@param height? integer
---@param depth? integer
---
---@return Node|nil margin
---@return integer|nil width
---@return integer|nil height
---@return integer|nil depth
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.useboxresource(n, width, height, depth) end

---
---Return the width, height, depth and margin of the resource.
---
---No dimensions returned means that the resource is
---unknown.
---
---* Corresponding C source code: [ltexlib.c#L3327-L3344](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3327-L3344)
---
---@param n Node
---
---@return integer|nil width
---@return integer|nil height
---@return integer|nil depth
---@return integer|nil margin
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getboxresourcedimensions(n) end

---
---Call the internal
---function that build a page, given that there is something to build.
---
---You should not expect to much from the `triggerbuildpage` helpers because
---often *TeX* doesn't do much if it thinks nothing has to be done, but it might be
---useful for some applications.
---
---* Corresponding C source code: [ltexlib.c#L3357-L3361](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3357-L3361)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.triggerbuildpage() end

---
---Split a box.
---
---The remainder is kept in the original box and a packaged vlist is returned. This
---operation is comparable to the `vsplit` operation. The mode can be `additional` or `exactly` and concerns the split off box.
---
---* Corresponding C source code: [ltexlib.c#L1282-L1308](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1282-L1308)
---
---@param n Node
---@param height integer
---@param mode 'additional' | 'exactly'
---
---@return Node|nil vlist
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.splitbox(n, height, mode) end

---
---Set the internal math parameters.
---
---* Corresponding C source code: [ltexlib.c#L1986-L2015](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1986-L2015)
---
---@param math_param_name MathParamName
---@param math_style_name MathStyleName
---@param value integer
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setmath(math_param_name, math_style_name, value) end

---
---Set the internal math parameters.
---
---* Corresponding C source code: [ltexlib.c#L1986-L2015](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L1986-L2015)
---
---@param global 'global' # Indicate a global assignment.
---@param math_param_name MathParamName
---@param math_style_name MathStyleName
---@param value integer|Node # The value is either a number (representing a dimension or number) or a glue spec node representing a muskip for `ordordspacing` and similar spacing parameters.
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setmath(global, math_param_name, math_style_name, value) end

---
---Query the internal math parameters.
---
---* Corresponding C source code: [ltexlib.c#L2017-L2039](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2017-L2039)
---
---@param math_param_name MathParamName
---@param math_style_name MathStyleName
---
---@return integer|Node value # The value is either a number (representing a dimension or number) or a glue spec node representing a muskip for `ordordspacing` and similar spacing parameters.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getmath(math_param_name, math_style_name) end

---
---The style name minus the trailing “style”.
---
---* Corresponding C source code: [texmath.c#L510-L516](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texmath.c#L510-L516)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias MathStyleName
---|'display'
---|'crampeddisplay'
---|'text'
---|'crampedtext'
---|'script'
---|'crampedscript'
---|'scriptscript'
---|'crampedscriptscript'

---
---The parameter name minus the leading “Umath”.
---
---* Corresponding C source code: [texmath.c#L518-L558](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texmath.c#L518-L558)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias MathParamName
---|'quad'
---|'axis'
---|'operatorsize'
---|'overbarkern'
---|'overbarrule'
---|'overbarvgap'
---|'underbarkern'
---|'underbarrule'
---|'underbarvgap'
---|'radicalkern'
---|'radicalrule'
---|'radicalvgap'
---|'radicaldegreebefore'
---|'radicaldegreeafter'
---|'radicaldegreeraise'
---|'stackvgap'
---|'stacknumup'
---|'stackdenomdown'
---|'fractionrule'
---|'fractionnumvgap'
---|'fractionnumup'
---|'fractiondenomvgap'
---|'fractiondenomdown'
---|'fractiondelsize'
---|'skewedfractionhgap'
---|'skewedfractionvgap'
---|'limitabovevgap'
---|'limitabovebgap'
---|'limitabovekern'
---|'limitbelowvgap'
---|'limitbelowbgap'
---|'limitbelowkern'
---|'nolimitsubfactor'
---|'nolimitsupfactor'
---|'underdelimitervgap'
---|'underdelimiterbgap'
---|'overdelimitervgap'
---|'overdelimiterbgap'
---|'subshiftdrop'
---|'supshiftdrop'
---|'subshiftdown'
---|'subsupshiftdown'
---|'subtopmax'
---|'supshiftup'
---|'supbottommin'
---|'supsubbottommax'
---|'subsupvgap'
---|'spaceafterscript'
---|'connectoroverlapmin'
---|'ordordspacing'
---|'ordopspacing'
---|'ordbinspacing'
---|'ordrelspacing'
---|'ordopenspacing'
---|'ordclosespacing'
---|'ordpunctspacing'
---|'ordinnerspacing'
---|'opordspacing'
---|'opopspacing'
---|'opbinspacing'
---|'oprelspacing'
---|'opopenspacing'
---|'opclosespacing'
---|'oppunctspacing'
---|'opinnerspacing'
---|'binordspacing'
---|'binopspacing'
---|'binbinspacing'
---|'binrelspacing'
---|'binopenspacing'
---|'binclosespacing'
---|'binpunctspacing'
---|'bininnerspacing'
---|'relordspacing'
---|'relopspacing'
---|'relbinspacing'
---|'relrelspacing'
---|'relopenspacing'
---|'relclosespacing'
---|'relpunctspacing'
---|'relinnerspacing'
---|'openordspacing'
---|'openopspacing'
---|'openbinspacing'
---|'openrelspacing'
---|'openopenspacing'
---|'openclosespacing'
---|'openpunctspacing'
---|'openinnerspacing'
---|'closeordspacing'
---|'closeopspacing'
---|'closebinspacing'
---|'closerelspacing'
---|'closeopenspacing'
---|'closeclosespacing'
---|'closepunctspacing'
---|'closeinnerspacing'
---|'punctordspacing'
---|'punctopspacing'
---|'punctbinspacing'
---|'punctrelspacing'
---|'punctopenspacing'
---|'punctclosespacing'
---|'punctpunctspacing'
---|'punctinnerspacing'
---|'innerordspacing'
---|'inneropspacing'
---|'innerbinspacing'
---|'innerrelspacing'
---|'inneropenspacing'
---|'innerclosespacing'
---|'innerpunctspacing'
---|'innerinnerspacing'

---
---The virtual table `tex.lists` contains the set of internal registers that
---keep track of building page lists.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3713](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3713)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lists = {}

---
---circular list of pending insertions
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lists.page_ins_head = nil

---
---the recent contributions
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lists.contrib_head = nil

---
---the current page content
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lists.page_head = nil

---
---used for held-over items for next page
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lists.hold_head = nil

---
---head of the current `vadjust` list
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lists.adjust_head = nil

---
---head of the current `\vadjust pre` list
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lists.pre_adjust_head = nil

---
---head of the discarded items of a page break
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lists.page_discards_head = nil

---
---head of the discarded items in a vsplit
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.lists.split_discards_head = nil

---
---You have
---to be careful with what you set as *TeX* can have expectations with regards to
---how a list is constructed or in what state it is.
---
---* Corresponding C source code: [ltexlib.c#L2266-L2321](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2266-L2321)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setlist() end

---
---* Corresponding C source code: [ltexlib.c#L2202-L2264](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2202-L2264)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getlist() end

---
---The virtual table `nest` contains the currently active semantic nesting
---state. It has two main parts: a zero-based array of userdata for the semantic
---nest itself, and the numerical value `ptr`, which gives the highest
---available index. Neither the array items in `nest[]` nor `ptr` can be
---assigned to (as this would confuse the typesetting engine beyond repair), but you
---can assign to the individual values inside the array items, e.g. `tex.nest[tex.nest.ptr].prevdepth`.
---
---`tex.nest[tex.nest.ptr]` is the current nest state, `nest[0]` the
---outermost (main vertical list) level.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3714](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3714)
---
---@type table<integer, Nest>
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.nest = {}

---
---highest
---available index.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
tex.nest.ptr = 0

---
---The getter function is `getnest`. You
---can pass a number (which gives you a list), nothing or `top`, which returns
---the topmost list, or the string `ptr` which gives you the index of the
---topmost list.
---
---* Corresponding C source code: [ltexlib.c#L2422-L2455](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2422-L2455)
---
---@return Nest
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getnest() end

---
---* Corresponding C source code: [ltexlib.c#L2457-L2461](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2457-L2461)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.setnest() end

---
---* Corresponding C source code: [ltexlib.c#L2325-L2361](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2325-L2361)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class Nest
---@field mode number # the meaning of these numbers depends on the engine and sometimes even the version; you can use `tex.getmodevalues()` to get the mapping: positive values signal vertical, horizontal and math mode, while negative values indicate inner and inline variants (all modes).
---@field modeline integer # source input line where this mode was entered in, negative inside the output routine (all modes).
---@field head Node # the head of the current list (all modes).
---@field tail Node # the tail of the current list (all modes).
---@field prevgraf integer # number of lines in the previous paragraph (vertical mode).
---@field prevdepth integer # depth of the previous paragraph (vertical mode).
---@field spacefactor integer # the current space factor (horizontal mode).
---@field dirs Node # used for temporary storage by the line break algorithm (horizontal mode).
---@field noad Node #  used for temporary storage of a pending fraction numerator, for `over` etc. (math mode).
---@field delimptr Node # used for temporary storage of the previous math delimiter, for `middle` (math mode).
---@field mathdir boolean # true when during math processing the `mathdir` is not the same as the surrounding `textdir` (math mode).
---@field mathstyle integer # mmode  the current `mathstyle`
---

---
---`tex.getmodevalues()` to get the mapping: positive values signal vertical, horizontal and math mode, while negative values indicate inner and inline variants (all modes).
---
---* Corresponding C source code: [ltexlib.c#L3549-L3565](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3549-L3565)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getmodevalues() end

---
---@alias PrintableInput string|number|integer

---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L1222-L1224](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L1222-L1224)
---* Corresponding C source code: [ltexlib.c#L161-164](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L161-164)
---
---@param input_line PrintableInput # Each string argument is treated by *TeX* as a separate input line.
---@param ... PrintableInput # The very last string of the very last `tex.print` command in a `directlua` will not have the `endlinechar` appended, all others do.
---
---@see tex.sprint
---@see tex.tprint
---@see tex.cprint
---@see tex.write
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.print(input_line, ...) end

---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L1222-L1224](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L1222-L1224)
---* Corresponding C source code: [ltexlib.c#L161-164](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L161-164)
---
---@param catcodetable_no integer The parameter can be used to print the strings using the catcode regime defined by `\catcodetable` `catcodetable_no`. If `catcodetable_no` is `-1`, the currently active catcode regime is used. If `catcodetable_no` is `-2`, the resulting catcodes are the result of `\the` `\toks`: all category codes are `12` (`other`) except for the space character, that has category code `10` (`space`). Otherwise, if `catcodetable_no` is not a valid catcode table, then it is ignored, and the currently active catcode regime is used instead.
---@param input_line PrintableInput # Each string argument is treated by *TeX* as a separate input line.
---@param ... PrintableInput # The very last string of the very last `tex.print` command in a `directlua` will not have the `endlinechar` appended, all others do.
---
---@see tex.print
---@see tex.sprint
---@see tex.tprint
---@see tex.cprint
---@see tex.write
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.print(catcodetable_no, input_line, ...) end

---
---If there is a
---table argument instead of a list of strings, this has to be a consecutive array
---of strings to print (the first non-string value will stop the printing process).
---
---see `LuaTeX` manual: 10.3.14.1 `print`
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L1222-L1224](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L1222-L1224)
---* Corresponding C source code: [ltexlib.c#L161-164](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L161-164)
---
---@param input_lines PrintableInput[] # Each string (or number) element of the input table is treated by *TeX* as a separate input line.
---
---@see tex.sprint
---@see tex.tprint
---@see tex.cprint
---@see tex.write
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.print(input_lines) end

---
---see `LuaTeX` manual: 10.3.14.1 `print`
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L1222-L1224](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L1222-L1224)
---* Corresponding C source code: [ltexlib.c#L161-164](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L161-164)
---
---@param catcodetable_no integer The parameter can be used to print the strings using the catcode regime defined by `\catcodetable` `catcodetable_no`. If `catcodetable_no` is `-1`, the currently active catcode regime is used. If `catcodetable_no` is `-2`, the resulting catcodes are the result of `\the` `\toks`: all category codes are `12` (`other`) except for the space character, that has category code `10` (`space`). Otherwise, if `catcodetable_no` is not a valid catcode table, then it is ignored, and the currently active catcode regime is used instead.
---@param input_lines PrintableInput[] # Each string element of the input table is treated by *TeX* as a separate input line.
---
---@see tex.sprint
---@see tex.tprint
---@see tex.cprint
---@see tex.write
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.print(catcodetable_no, input_lines) end

---
---combines all of its arguments (or the elements of array argument) into a single input line for TeX. So any leading spaces will be treated as if they were typed, etc. (`s(ingle-input-line)print`).
---
---Each string argument is treated by *TeX* as a special kind of input line that
---makes it suitable for use as a partial line input mechanism:
---
---* *TeX* does not switch to the “new line” state, so that leading
---  spaces are not ignored.
---* No `endlinechar` is inserted.
---* Trailing spaces are not removed. Note that this does not prevent
---  *TeX* itself from eating spaces as result of interpreting the line.
---  For example, in
---
---```tex
---before\directlua{tex.sprint("\\relax")tex.sprint(" inbetween")}after
---```
---
---the space before `in between` will be gobbled as a result of the
---“normal” scanning of `\relax`.
---
---If there is a table argument instead of a list of strings, this has to be a
---consecutive array of strings to print (the first non-string value will stop the
---printing process).
---
---The optional argument sets the catcode regime, as with `tex.print`. This
---influences the string arguments (or numbers turned into strings).
---
---Although this needs to be used with care, you can also pass token or node
---userdata objects. These get injected into the stream. Tokens had best be valid
---tokens, while nodes need to be around when they get injected. Therefore it is
---important to realize the following:
---
---* When you inject a token, you need to pass a valid token userdata object. This
---  object will be collected by *Lua* when it no longer is referenced. When it gets
---  printed to *TeX* the token itself gets copied so there is no interference with the
---  *Lua* garbage collection. You manage the object yourself. Because tokens are
---  actually just numbers, there is no real extra overhead at the *TeX* end.
---* When you inject a node, you need to pass a valid node userdata object. The
---  node related to the object will not be collected by *Lua* when it no longer
---  is referenced. It lives on at the *TeX* end in its own memory space. When it
---  gets printed to *TeX* the node reference is used assuming that node stays
---  around. There is no *Lua* garbage collection involved. Again, you manage the
---  object yourself. The node itself is freed when *TeX* is done with it.
---
---If you consider the last remark you might realize that we have a problem when a
---printed mix of strings, tokens and nodes is reused. Inside *TeX* the sequence
---becomes a linked list of input buffers. So, `"123"` or `"\foo{123`"}
---gets read and parsed on the fly, while `<token userdata>` already is
---tokenized and effectively is a token list now. A `<node userdata>` is also
---tokenized into a token list but it has a reference to a real node. Normally this
---goes fine. But now assume that you store the whole lot in a macro: in that case
---the tokenized node can be flushed many times. But, after the first such flush the
---node is used and its memory freed. You can prevent this by using copies which is
---controlled by setting `luacopyinputnodes` to a non-zero value. This is one
---of these fuzzy areas you have to live with if you really mess with these low
---level issues.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L168-171](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L168-171)
---
---@param input PrintableInput
---@param ... PrintableInput
---
---@see tex.print
---@see tex.tprint
---@see tex.cprint
---@see tex.write
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.sprint(input, ...) end

---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L168-171](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L168-171)
---
---@param catcodetable_no integer The parameter can be used to print the strings using the catcode regime defined by `\catcodetable` `catcodetable_no`. If `catcodetable_no` is `-1`, the currently active catcode regime is used. If `catcodetable_no` is `-2`, the resulting catcodes are the result of `\the` `\toks`: all category codes are `12` (`other`) except for the space character, that has category code `10` (`space`). Otherwise, if `catcodetable_no` is not a valid catcode table, then it is ignored, and the currently active catcode regime is used instead.
---@param input PrintableInput
---@param ... PrintableInput
---
---@see tex.print
---@see tex.tprint
---@see tex.cprint
---@see tex.write
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.sprint(catcodetable_no, input, ...) end

---
---(`s(ingle-input-line)print`).
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L168-171](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L168-171)
---
---@param input PrintableInput[] # Each string element of the input table is treated by *TeX* as a separate input argument.
---
---@see tex.print
---@see tex.tprint
---@see tex.cprint
---@see tex.write
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.sprint(input) end

---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L168-171](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L168-171)
---
---@param catcodetable_no integer The parameter can be used to print the strings using the catcode regime defined by `\catcodetable` `catcodetable_no`. If `catcodetable_no` is `-1`, the currently active catcode regime is used. If `catcodetable_no` is `-2`, the resulting catcodes are the result of `\the` `\toks`: all category codes are `12` (`other`) except for the space character, that has category code `10` (`space`). Otherwise, if `catcodetable_no` is not a valid catcode table, then it is ignored, and the currently active catcode regime is used instead.---@param t table
---@param input PrintableInput[] # Each string element of the input table is treated by *TeX* as a separate input argument.
---
---@see tex.print
---@see tex.tprint
---@see tex.cprint
---@see tex.write
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.sprint(catcodetable_no, input) end

---
---Print tables of strings to the input stream (`t(able)print`).
---
---This function is basically a shortcut for repeated calls to
---`tex.sprint(catcodetable_no,  input, ...)`, once for each of the
---supplied argument tables.
---
---```tex
---\begingroup
---\catcode`\_=12
---\savecatcodetable 2
---\endgroup
---\tt\directlua{tex.tprint({ 2, '_' })}
---```
---
---```lua
---tex.tprint({ 1, 1, 2, 3 }) -- output: 123
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L207-242](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L207-242)
---
---@param input PrintableInput[] # Each string element of the input table is treated by *TeX* as a separate input argument.
---@param ... PrintableInput[]
---
---@see tex.print
---@see tex.sprint
---@see tex.cprint
---@see tex.write
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.tprint(input, ...) end

---
---Print to the input stream and specify a catcode (`c(atcode)print`).
---
---```lua
---tex.cprint( 1, " 1: `&{\\foo}") tex.print("\\par") -- a lot of \bgroup s
---tex.cprint( 2, " 2: `&{\\foo}") tex.print("\\par") -- matching \egroup s
---tex.cprint( 9, " 9: `&{\\foo}") tex.print("\\par") -- all get ignored
---tex.cprint(10, "10: `&{\\foo}") tex.print("\\par") -- all become spaces
---tex.cprint(11, "11: `&{\\foo}") tex.print("\\par") -- letters
---tex.cprint(12, "12: `&{\\foo}") tex.print("\\par") -- other characters
---tex.cprint(14, "12: $&{\\foo}") tex.print("\\par") -- comment triggers
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L175-203](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L175-203)
---
---@param catcode integer # The category code (`0` stands for the escape character, normally `\`, `1` stands for begin grouping, normally `{` and so on).
---@param input PrintableInput
---@param ... PrintableInput
---
---@see tex.print
---@see tex.sprint
---@see tex.tprint
---@see tex.write
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.cprint(catcode, input, ...) end

---
---Print to the input stream and specify a catcode (`c(atcode)print`).
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L175-203](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L175-203)
---
---@param catcode integer # The category code (`0` stands for the escape character, normally `\`, `1` stands for begin grouping, normally `{` and so on).
---@param input PrintableInput[] # Each string element of the input table is treated by *TeX* as a separate input argument.
---
---@see tex.print
---@see tex.sprint
---@see tex.tprint
---@see tex.write
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.cprint(catcode, input) end

---
---Print to the input stream using category code `12` (`other`) except
---for the space character, that has category code `10` (`space`).
---
---Each string argument is treated by *TeX* as a special kind of input line that
---makes it suitable for use as a quick way to dump information:
---
---* All catcodes on that line are either “space” (for ` `) or “character” (for all others).
---* There is no `endlinechar` appended.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L175-203](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L154-157)
---
---@param input PrintableInput
---@param ... PrintableInput
---
---@see tex.print
---@see tex.sprint
---@see tex.tprint
---@see tex.cprint
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.write(input, ...) end

---
---Print to the input stream using category code `12` (`other`) except
---for the space character, that has category code `10` (`space`).
---
---Each string argument is treated by *TeX* as a special kind of input line that
---makes it suitable for use as a quick way to dump information:
---
---* All catcodes on that line are either “space” (for ` `) or “character” (for all others).
---* There is no `endlinechar` appended.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L175-203](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L154-157)
---
---@param input PrintableInput[] # Each string element of the input table is treated by *TeX* as a separate input argument.
---
---@see tex.print
---@see tex.sprint
---@see tex.tprint
---@see tex.cprint
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.write(input) end

---
---Rounds *Lua* number `o`, and returns a number that is in the range of a
---valid *TeX* register value. If the number starts out of range, it generates a
---“number too big” error as well.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L2474-2482](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2474-2482)
---
---@param o number
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.round(o) end

---
---Multiplies the *Lua* numbers `o` and `delta`, and returns a rounded
---number that is in the range of a valid *TeX* register value. In the table
---version, it creates a copy of the table with all numeric top-level values scaled
---in that manner. If the multiplied number(s) are of range, it generates
---“number too big” error(s) as well.
---
---Note: the precision of the output of this function will depend on your computer's
---architecture and operating system, so use with care! An interface to *LuaTeX*'s
---internal, 100% portable scale function will be added at a later date.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L2484-2513](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2484-2513)
---
---@param o number
---@param delta number
---
---@return number
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.scale(o, delta) end

---
---@param o table
---@param delta number
---
---@return table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.scale(o, delta) end

---
---Companion to the primitive `\number`.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L2484-2513](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2484-2513)
---* https://www.tug.org/utilities/plain/cseq.html#number-rp
---
---@param n integer
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.number(n) end

---
---Companion to the primitive `\romannumeral`.
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L2061-2064](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2061-2064)
---
---@param n integer # for example `2025`
---
---@return string # for example `mmxxv`
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.romannumeral(n) end

---
---returns the `csname` string that matches a
---font id number (if there is one).
---
---__Reference:__
---
---* Corresponding C source code: [L2047-L2050](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2047-L2050)
---
---@param font_id integer
---
---@return string # for example `\tenrm`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.fontidentifier(font_id) end

---
---The first one returns the name only, the second one reports the size too.
---
---__Reference:__
---
---* Corresponding C source code: [L2042-L2045](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2042-L2045)
---
---@param font_id integer
---
---@return string # for example `cmr10`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.fontname(font_id) end

---
---Converts the number `o` that represents an explicit
---dimension into an integer number of scaled points.
---
---see `LuaTeX` manual: 10.3.15.5 `sp`
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L1395-L1396](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L1395-L1396)
---
---@param o integer
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.sp(o) end

---
---Convert a string `s` that represents an explicit
---dimension into an integer number of scaled points.
---
---For parsing the string, the same scanning and conversion rules are used that
---*LuaTeX* would use if it was scanning a dimension specifier in its *TeX*-like
---input language (this includes generating errors for bad values), expect for the
---following:
---
---* only explicit values are allowed, control sequences are not handled
---* infinite dimension units (`fil...`) are forbidden
---* `mu` units do not generate an error (but may not be useful either)
---
---__Example__:
---
---```lua
---local scaled_points = tex.sp('1cm')
---print(scaled_points) -- 1864679
---```
---
---__Reference__:
---
---* `LuaTeX` manual: 10.3.15.5 `sp` page 204
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L1386-L1413](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L1386-L1413)
---
---@param s string # A string to convert into scaled points.
---
---@return integer # The dimension in the scaled points format.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.sp(s) end

-- Removed?

---
---Create an error like the combination of `\errhelp` and
---`\errmessage`.
---
---During this error, deletions are disabled.
---
---see `LuaTeX` manual: 10.3.15.5 `sp`
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L1438-L1457](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L1438-L1457)
---* Corresponding C source code: [ltexlib.c#L654-692](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L654-692)
---
---@param message string # An error message like `\errmessage`.
---@param help? table # The array part of the `help` table has to contain strings, one for each line of error help.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.error(message, help) end

---
---Show the current
---(expansion) context in case of an error.
---
----__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L3210-L3215](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3210-L3215)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.show_context() end

---
---Start the interpretation.
---
---A run normally
---boils down to *TeX* entering the so called main loop. A token is fetched and
---depending on it current meaning some actions takes place. Sometimes that actions
---comes immediately, sometimes more scanning is needed. Quite often tokens get
---pushed back into the input. This all means that the *TeX* scanner is constantly
---pushing and popping input states, but in the end after all the action is done
---returns to the main loop.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L1459-L1470](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L1459-L1470)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.run() end

---
---Force the end of an interpretation.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L1464](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L1464)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.finish() end

---
---Because of the fact that *TeX* is in a complex dance of expanding, dealing with
---fonts, typesetting paragraphs, messing around with boxes, building pages, and so
---on, you cannot easily run a nested *TeX* run (read nested main loop). However,
---there is an option to force a local run with `runtoks`. The content of the
---given token list register gets expanded locally after which we return to where we
---triggered this expansion, at the *Lua* end. Instead a function can get passed
---that does some work. You have to make sure that at the end *TeX* is in a sane
---state and this is not always trivial. A more complex mechanism would complicate
---*TeX* itself (and probably also harm performance) so this simple local expansion
---loop has to do.
---
---When the `tracingnesting` parameter is set to a value larger than 2 some
---information is reported about the state of the local loop.
---
---Inside for instance an `\edef` the `runtoks` function behaves (at
---least tries to) like it were an `\the`. This prevents unwanted side
---effects: normally in such an definition tokens remain tokens and (for instance)
---characters don't become nodes.
---
---__Example:__
---
---```lua
---\toks0{\setbox0\hbox{one}}
---\toks1{\setbox0\hbox{first}}
---\directlua {
---  tex.forcehmode(true)
---  tex.runtoks(0)
---  print(tex.getbox(0).width)
---  tex.runtoks(1)
---  print(tex.getbox(0).width)
---}
---```
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-tex.tex#L1472-L1508](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-tex.tex#L1472-L1508)
---* Corresponding C source code: [ltexlib.c#L3464-L3536](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3464-L3536)
---
---@param token_register integer
---@param force? boolean # force the local main loop
---@param grouped? boolean # adds a level of grouping.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.runtoks(token_register, force, grouped) end

---
---@param func function
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.runtoks(func) end

---
---You can quit the local loop with `\endlocalcontrol` or from the *Lua* end
---with `tex.quittoks`. In that case you end one level up! Of course in the
---end that can mean that you arrive at the main level in which case an extra end
---will trigger a redundancy warning (not an abort!).
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.quittoks() end

---
---An example of a (possible error triggering) complication is that *TeX* expects to
---be in some state, say horizontal mode, and you have to make sure it is when you
---start feeding back something from *Lua* into *TeX*. Normally a user will not run
---into issues but when you start writing tokens or nodes or have a nested run there
---can be situations that you need to run `forcehmode`. There is no recipe for
---this and intercepting possible cases would weaken *LuaTeX*'s flexibility.
---
---@param indented? boolean
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.forcehmode(indented) end

---
---Return a list of names.
---
---This can be useful for debugging, but note that this
---also reports control sequences that may be unreachable at this moment due to
---local redefinitions: it is strictly a dump of the hash table. You can use `token.create` to inspect properties, for instance when the `command` key
---in a created table equals `123`, you have the `cmdname` value `undefined_cs`.
---
---```lua
---for i,v in pairs(tex.hashtokens()) do ... end
---```
---
---__Reference:__
---
---* Corresponding C source code: [ltexlib.c#L2662-L2690](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2662-L2690)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.hashtokens() end

---
---Associates `csname` with the internal font number `fontid`. The
---definition is global if (and only if) `global` is specified and true (the
---setting of `globaldefs` is not taken into account).
---
---@param csname string
---@param fontid integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.definefont(csname, fontid) end

---
---Associates `csname` with the internal font number `fontid`. The
---definition is global if (and only if) `global` is specified and true (the
---setting of `globaldefs` is not taken into account).
---
---@param global boolean
---@param csname string
---@param fontid integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.definefont(global, csname, fontid) end

---
---This function accepts a prefix string and an array of primitive names. For each
---combination of “prefix” and “name”, the `tex.enableprimitives` first verifies that “name” is an actual primitive
---(it must be returned by one of the `tex.extraprimitives` calls explained
---below, or part of *TeX*82, or `directlua`). If it is not, `tex.enableprimitives` does nothing and skips to the next pair.
---
---But if it is, then it will construct a csname variable by concatenating the
---“prefix” and “name”, unless the “prefix” is already the
---actual prefix of “name”. In the latter case, it will discard the “prefix”, and just use “name”.
---
---Then it will check for the existence of the constructed csname. If the csname is
---currently undefined (note: that is not the same as `relax`), it will
---globally define the csname to have the meaning: run code belonging to the
---primitive “name”. If for some reason the csname is already defined, it
---does nothing and tries the next pair.
---
---An example:
---
---```lua
---tex.enableprimitives('LuaTeX', {'formatname'})
---```
---
---will define `\LuaTeXformatname` with the same intrinsic meaning as the
---documented primitive `formatname`, provided that the control sequences `\LuaTeXformatname` is currently undefined.
---
---When *LuaTeX* is run with `--ini` only the *TeX*82 primitives and `directlua` are available, so no extra primitives {\bf at all}.
---
---If you want to have all the new functionality available using their default
---names, as it is now, you will have to add
---
---```tex
---\ifx\directlua\undefined \else
---    \directlua {tex.enableprimitives('',tex.extraprimitives ())}
---\fi
---```
---
---near the beginning of your format generation file. Or you can choose different
---prefixes for different subsets, as you see fit.
---
---Calling some form of `tex.enableprimitives` is highly important though,
---because if you do not, you will end up with a *TeX*82-lookalike that can run *Lua*
---code but not do much else. The defined csnames are (of course) saved in the
---format and will be available at runtime.
---
---*
---
---* Corresponding C source code: [ltexlib.c#L2752-L2809](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2752-L2809)
---
---@param prefix string
---@param primitive_names string[]
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.enableprimitives(prefix, primitive_names) end

---
---@alias TexEngine
---|'etex'
---|'tex'
---|'core'
---|'luatex'

---
---Return a list of the primitives that originate from the engine(s)
---given by the requested string value(s).
---
---The possible values and their (current)
---return values are given in the following table. In addition the somewhat special
---primitives “\tex{ ”}, “\tex {/”} and “`-`” are defined.
---
---Note that `luatex` does not contain `directlua`, as that is
---considered to be a core primitive, along with all the *TeX*82 primitives, so it is
---part of the list that is returned from `core`.
---
---Running `tex.extraprimitives()` will give you the complete list of
---primitives `-ini` startup. It is exactly equivalent to `tex.extraprimitives("etex","luatex")`.
---
---* Corresponding C source code: [ltexlib.c#L2710-L2750](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2710-L2750)
---
---@param ... TexEngine
---
---@return string[]
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.extraprimitives(...) end

---
---Return a list of all primitives that *LuaTeX* knows about.
---
---* Corresponding C source code: [ltexlib.c#L2691-L2708](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2691-L2708)
---
---@return string[]
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.primitives() end

---
---Calculate the badness.
---
---This helper function is useful during linebreak calculations. The function returns the badness for when total `total`
---is supposed to be made from amounts that sum to `sum`. The returned number is
---a reasonable approximation of `100(total/sum)^3`.
---
---* Corresponding C source code: [ltexlib.c#L3069-L3075](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3069-L3075)
---* Corresponding C source code: [arithmetic.c#L275-L300](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/arithmetic.c#L275-L300)
---
---@param total integer # scaled values
---@param sum integer # scaled values
---
---@return integer badness
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.badness(total, sum) end

---
---Reset the parameters that *TeX* normally resets when a new paragraph
---is seen.
---
---* Corresponding C source code: [ltexlib.c#L3054-L3059](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3054-L3059)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.resetparagraph() end

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class LinebreakParameters
---@field pardir string
---@field pretolerance integer
---@field tracingparagraphs integer
---@field tolerance integer
---@field looseness integer
---@field hyphenpenalty integer
---@field exhyphenpenalty integer
---@field pdfadjustspacing integer
---@field adjdemerits integer
---@field pdfprotrudechars integer
---@field linepenalty integer
---@field lastlinefit integer
---@field doublehyphendemerits integer
---@field finalhyphendemerits integer
---@field hangafter integer
---@field interlinepenalty integer|table # or table  if a table, then it is an array like `interlinepenalties`
---@field clubpenalty integer|table # or table  if a table, then it is an array like `clubpenalties`
---@field widowpenalty integer|table # or table  if a table, then it is an array like `widowpenalties`
---@field brokenpenalty integer
---@field emergencystretch integer # in scaled points
---@field hangindent integer # in scaled points
---@field hsize integer # in scaled points
---@field leftskip GlueSpecNode
---@field rightskip GlueSpecNode
---@field parshape table
---

---
---Note that there is no interface for `displaywidowpenalties`, you have to
---pass the right choice for `widowpenalties` yourself.
---
---It is your own job to make sure that `listhead` is a proper paragraph list:
---this function does not add any nodes to it. To be exact, if you want to replace
---the core line breaking, you may have to do the following (when you are not
---actually working in the `pre_linebreak_filter` or `linebreak_filter`
---callbacks, or when the original list starting at listhead was generated in
---horizontal mode):
---
---* add an “indent box” and perhaps a `local_par` node at the start
---    (only if you need them)
---
---* replace any found final glue by an infinite penalty (or add such a penalty,
---    if the last node is not a glue)
---
---* add a glue node for the `parfillskip` after that penalty node
---
---* make sure all the `prev` pointers are OK
---
---The result is a node list, it still needs to be vpacked if you want to assign it
---to a `vbox`. The returned `info` table contains four values that are
---all numbers:
---
--- name       explanation
---
--- prevdepth  depth of the last line in the broken paragraph
--- prevgraf   number of lines in the broken paragraph
--- looseness  the actual looseness value in the broken paragraph
--- demerits   the total demerits of the chosen solution
---
---Note there are a few things you cannot interface using this function: You cannot
---influence font expansion other than via `pdfadjustspacing`, because the
---settings for that take place elsewhere. The same is true for hbadness and hfuzz
---etc. All these are in the `hpack` routine, and that fetches its own
---variables via globals.
---
---@param listhead Node
---@param parameters LinebreakParameters
---
---@return Node
---@return table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.linebreak(listhead, parameters) end

---
---Ships out box number `n` to the output file, and clears the box register.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.shipout() end

---
---This helper reports the current page state: `empty`, `box_there` or
---`inserts_only` as integer value.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getpagestate() end

---
---This integer reports the current level of the local loop. It's only useful for
---debugging and the (relative state) numbers can change with the implementation.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.getlocallevel() end

-- { "uniform_rand",tex_unif_rand },
-- { "normal_rand", tex_norm_rand },
-- { "lua_math_randomseed", tex_init_rand }, /* syntactic sugar  */

---
---Initialize a new seed.
---
---`lua_math_randomseed` is
---equivalent to this function.
---
---* Corresponding C source code: [ltexlib.c#L3132-L3142](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3132-L3142)
---
---@see tex.lua_math_randomseed
---
---@param seed integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.init_rand(seed) end

---
---Initialize a new seed.
---
---`init_rand` is
---equivalent to this function.
---
---* Corresponding C source code: [ltexlib.c#L3132-L3142](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3132-L3142)
---
---@see tex.init_rand
---
---@param seed integer
---
---You can
---initialize with a new seed with `lua_math_randomseed` (`init_rand` is
---equivalent to this one.)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.lua_math_randomseed(seed) end

---
---For practical reasons *LuaTeX* has its own random number generator. This is the original
---*Lua* random function.
---
---* Corresponding C source code: [ltexlib.c#L3164-L3192](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3164-L3192)
---
---@param lower number
---@param upper number
---
---@return number
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.lua_math_random(lower, upper) end

---
---For practical reasons *LuaTeX* has its own random number generator. This is the original
---*Lua* random function.
---
---* Corresponding C source code: [ltexlib.c#L3164-L3192](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3164-L3192)
---
---@param upper? number
---
---@return number
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.lua_math_random(upper) end

---
---no argument is used
---
---* Corresponding C source code: [ltexlib.c#L3156-L3160](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3156-L3160)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.normal_rand() end

---
---takes a number that will get rounded before being used
---
---* Corresponding C source code: [ltexlib.c#L3144-L3154](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3144-L3154)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.uniform_rand() end

---
--- which behaves like the primitive and expects a scaled integer, so
---
---```lua
---tex.print(tex.uniformdeviate(65536)/65536)
---```
---
---will give a random number between zero and one.
---
---* Corresponding C source code: [ltexlib.c#L2051-L2054](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L2051-L2054)
---
---@param n integer a scaled integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.uniformdeviate(n) end

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias SyntexMode
---|0 # is the default and used normal synctex logic
---|1 # uses the values set by the next helpers
---|2 # sets for glyph nodes
---|3 # sets glyph and glue nodes
---|4 # sets only glyph modes

---
---* Corresponding C source code: [ltexlib.c#L3377-L3382](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3377-L3382)
---
---@param mode SyntexMode
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.set_synctex_mode(mode) end

---
---Return the current mode.
---
---* Corresponding C source code: [ltexlib.c#L3383-L3387](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3383-L3387)
---
---@return SyntexMode mode
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.get_synctex_mode() end

---
---Set the current tag (file) value (obeys save stack).
---
---* Corresponding C source code: [ltexlib.c#L3389-L3394](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3389-L3394)
---
---@param tag integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.set_synctex_tag(tag) end

---
---Get the currently set value of a tag (file).
---
---* Corresponding C source code: [ltexlib.c#L3396-L3400](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3396-L3400)
---
---@return integer tag
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.get_synctex_tag() end

---
---Overload the tag (file) value  (`0` resets).
---
---* Corresponding C source code: [ltexlib.c#L3402-L3407](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3402-L3407)
---
---@param tag integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.force_synctex_tag(tag) end

---
---Set the current line value (obeys save stack).
---
---* Corresponding C source code: [ltexlib.c#L3416-L3421](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3416-L3421)
---
---@param line integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.set_synctex_line(line) end

---
---Get the currently set value of a line.
---
---* Corresponding C source code: [ltexlib.c#L3423-L3427](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3423-L3427)
---
---@return integer line
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.get_synctex_line() end

---
---Overload the line value (`0` resets)
---
---* Corresponding C source code: [L3409-L3414](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3409-L3414)
---
---@param line integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.force_synctex_line(line) end

---
---Disable synctex file logging.
---
---* Corresponding C source code: [ltexlib.c#L3429-L3434](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexlib.c#L3429-L3434)
---
---@param flag integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/tex.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function tex.set_synctex_no_files(flag) end

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
lua = {}

---
---Version information: This library contains one read-only item:
---
---```lua
---<string> s = lua.version
---```
---
---This returns the *Lua* version identifier string.
---
---__Reference:__
---
---* Corresponding C source code: [llualib.c#L416-L417](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llualib.c#L416-L417)
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
lua.version = ""

---
---Use the `bytecode` table to store *Lua* code chunks. The accepted values for
---assignments are functions and `nil`. Likewise, the retrieved value is
---either a function or `nil`.
---
---The contents of the `lua.bytecode` array is stored inside the format file
---as actual *Lua* bytecode, so it can also be used to preload *Lua* code. The
---function must not contain any upvalues.
---@type table<integer, function|nil>
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
lua.bytecode = {}

---
---Save a function in a bytecode register.
---
---__Example:__
---
---```lua
---lua.setbytecode(13, function () print('A message') end)
---local print_message = lua.getbytecode(13)
---print_message() -- 'A message'
---```
---
---__Reference:__
---
---* Corresponding C source code: [llualib.c#L249-L315](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llualib.c#L249-L315)
---
---@param n integer
---@param f function|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lua.setbytecode(n, f) end

---
---Return a previously stored function from a bytecode register.
---
---__Example:__
---
---```lua
---lua.setbytecode(13, function () print('A message') end)
---local print_message = lua.getbytecode(13)
---print_message() -- 'A message'
---```
---
---__Reference:__
---
---* Corresponding C source code: [llualib.c#L187-L212](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llualib.c#L187-L212)
---
---@param n integer
---
---@return function|nil f
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lua.getbytecode(n) end

---
---Return two numbers, one for the command handler and one for the graphical user interface (on Microsoft Windows).
---
---__Reference:__
---
---* Corresponding C source code: [llualib.c#L382-L392](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llualib.c#L382-L392)
---
---@return integer # command handler
---@return integer # graphical user interface
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lua.getcodepage() end

---
---There is an array of 65536 (0-65535) potential chunk names for use with the
---`directlua` and `latelua` primitives.
---
---```
---lua.name[<number> n] = <string> s
---<string> s = lua.name[<number> n]
---```
---@see lua.getluaname
---@see lua.setluaname
---
---@type table<integer, string>
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
lua.name = {}

---
---Set a Lua chunk name.
---
---When a chunk name starts with
---a `@` it will be displayed as a file name. This is a side effect of the way Lua implements error
---handling.
---
---__Reference:__
---
---* Corresponding C source code: [llualib.c#L318-L339](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llualib.c#L318-L339)-
---
---@param chunk_name string|nil # If you want to unset a *Lua* name, you can assign `nil` to it.
---@param index integer
---
---@see lua.name
---@see lua.getluaname
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lua.setluaname(chunk_name, index) end

---
---Return a Lua chunk name.
---
---__Reference:__
---
---* Corresponding C source code: [llualib.c#L341-L354](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llualib.c#L341-L354)
---
---@param index number
---
---@return string|nil
---
---@see lua.name
---@see lua.setluaname
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lua.getluaname(index) end

---
---Create a new empty table and push it onto the stack.
---
---Parameter `index` is a hint for how many elements the table will have as a sequence; parameter `hash` is a hint for how many other elements the table will have. Lua may use these hints to preallocate memory for the new table. This preallocation is useful for performance when you know in advance how many elements the table will have.
---
---__Reference:__
---
---* Corresponding C source code: [llualib.c#L362-L368](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llualib.c#L362-L368)
---* [lua_createtable](https://pgl.yoyo.org/luai/i/lua_createtable)
---
---@param index integer
---@param hash integer
---
---@return table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lua.newtable(index, hash) end

---
---Return a number indicating
---how much nesting is going on.
---
---It is only of use as a breakpoint when
---checking some mechanism going haywire.
---
---__Reference:__
---
---* `LuaTeX` manual: 10.1.4 Introspection
---* Corresponding C source code: [llualib.c#L370-L374](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llualib.c#L370-L374)
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lua.getstacktop() end

---
---Return a number indicating
---how much nesting is going on.
---
---It is only of use as a breakpoint when
---checking some mechanism going haywire.
---
---__Reference:__
---
---* Corresponding C source code: [llualib.c#L376-L380](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llualib.c#L376-L380)
---* `LuaTeX` manual: 10.1.4 Introspection
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lua.getcalllevel() end

---
---The `\directlua` commands involves tokenization of its argument (after
---picking up an optional name or number specification). The tokenlist is then
---converted into a string and given to *Lua* to turn into a function that is
---called. The overhead is rather small but when you have millions of calls it can
---have some impact. For this reason there is a variant call available: `\luafunction`. This command is used as follows:
---
---```tex
---\directlua {
---    local t = lua.get_functions_table()
---    t[1] = function() tex.print("!") end
---    t[2] = function() tex.print("?") end
---}
---
---\luafunction1
---\luafunction2
---```
---
---Of course the functions can also be defined in a separate file. There is no limit
---on the number of functions apart from normal *Lua* limitations. Of course there
---is the limitation of no arguments but that would involve parsing and thereby give
---no gain. The function, when called in fact gets one argument, being the index, so
---in the following example the number `8` gets typeset.
---
---```tex
---\directlua {
---    local t = lua.get_functions_table()
---    t[8] = function(slot) tex.print(slot) end
---}
---```
--- ---
---
---```lua
---token.set_lua("mycode", id)
---token.set_lua("mycode", id, "global", "protected")
---```
---
---This creates a token that refers to a *Lua* function with an entry in the table
---that you can access with `lua.get_functions_table`. It is the companion
---to `luadef`.
---
---__Reference:__
---
---* `LuaTeX` manual: 2.4.4 `\luafunction`, `\luafunctioncall` and `\luadef`
---* `LuaTeX` manual: 10.6.4 Macros
---* Corresponding C source code: [llualib.c#L356-L360](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llualib.c#L356-L360)
---
---@return {[integer]: fun(slot: integer)}
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lua.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lua.get_functions_table() end
---
---The `pdfscanner` library allows interpretation of *PDF* content streams and
---`/ToUnicode` (cmap) streams. You can get those streams from the `pdfe` library.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdfscanner.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
pdfscanner = {}

---
---The first argument should be a *Lua* string or a stream or array object coming
---from the `pdfe` library. The second argument, `operatortable`, should
---be a *Lua* table where the keys are *PDF* operator name strings and the values
---are *Lua* functions (defined by you) that are used to process those operators.
---The functions are called whenever the scanner finds one of these *PDF* operators
---in the content stream(s). The functions are called with two arguments: the `scanner` object itself, and the `info` table that was passed are the third
---argument to `pdfscanner.scan`.
---
---Internally, `pdfscanner.scan` loops over the *PDF* operators in the
---stream(s), collecting operands on an internal stack until it finds a *PDF*
---operator. If that *PDF* operator's name exists in `operatortable`, then the
---associated function is executed. After the function has run (or when there is no
---function to execute) the internal operand stack is cleared in preparation for the
---next operator, and processing continues.
---
---The `scanner` argument to the processing functions is needed because it
---offers various methods to get the actual operands from the internal operand
---stack.
---
---```lua
---local operatortable = { }
---
---operatortable.Do = function(scanner,info)
---    local resources = info.resources
---    if resources then
---        local val     = scanner:pop()
---        local name    = val[2]
---        local xobject = resources.XObject
---        print(info.space .. "Uses XObject " .. name)
---        local resources = xobject.Resources
---        if resources then local newinfo =  { space     = info.space .. "  ", resources = resources, } pdfscanner.scan(entry, operatortable, newinfo)
---        end
---    end
---end
---
---local function Analyze(filename)
---    local doc = pdfe.open(filename)
---    if doc then
---        local pages = doc.Pages
---        for i=1,#pages do local page = pages[i] local info = { space     = "  " , resources = page.Resources, } print("Page " .. i)
---         -- pdfscanner.scan(page.Contents,operatortable,info) pdfscanner.scan(page.Contents(),operatortable,info)
---        end
---    end
---end
---
---Analyze("foo.pdf")
---```
---
---__Reference:__
---
---* Corresponding C source code: [lpdfscannerlib.c#L680-L828](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdfscannerlib.c#L680-L828)
---
---@param pdf string|PdfeStream|PdfeArray
---@param operatortable Operatorable
---@param info table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdfscanner.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdfscanner.scan(pdf, operatortable, info) end

---
---@class Scanner
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdfscanner.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
local Scanner = {}

---
---__Reference:__
---
---* Corresponding C source code: [lpdfscannerlib.c#L1063-L1069](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdfscannerlib.c#L1063-L1069)
---
---@return any
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdfscanner.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Scanner:pop() end

---
---Return only the value of a real or integer
---
---__Reference:__
---
---* Corresponding C source code: [lpdfscannerlib.c#L1013-L1021](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdfscannerlib.c#L1013-L1021)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdfscanner.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Scanner:popnumber() end

---
---Return only the value of a name
---
---__Reference:__
---
---* Corresponding C source code: [lpdfscannerlib.c#L1039-L1045](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdfscannerlib.c#L1039-L1045)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdfscanner.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Scanner:popname() end

---
---Return only the value of a string
---
---__Reference:__
---
---* Corresponding C source code: [lpdfscannerlib.c#L1031-L1037](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdfscannerlib.c#L1031-L1037)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdfscanner.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Scanner:popstring() end

---
---Return only the value of a array
---
---__Reference:__
---
---* Corresponding C source code: [lpdfscannerlib.c#L1047-L1053](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdfscannerlib.c#L1047-L1053)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdfscanner.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Scanner:poparray() end

---
---Return only the value of a dict
---
---__Reference:__
---
---* Corresponding C source code: [lpdfscannerlib.c#L1055-L1061](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdfscannerlib.c#L1055-L1061)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdfscanner.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Scanner:popdictionary() end

---
---Return only the value of a boolean
---
---__Reference:__
---
---* Corresponding C source code: [lpdfscannerlib.c#L1023-L1029](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdfscannerlib.c#L1023-L1029)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdfscanner.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Scanner:popboolean() end

---
---Abort further processing of this scan() call
---
---__Reference:__
---
---* Corresponding C source code: [lpdfscannerlib.c#L830-L836](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdfscannerlib.c#L830-L836)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdfscanner.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Scanner:done() end

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdfscanner.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias Operatorable table<string, fun(scanner: Scanner, info: table)>
---
---This library contains a number of run-time configuration items that you may find useful in message
---reporting, as well as an iterator function that gets all of the names and values as a `table`.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/status.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
status = {}

---
---__Reference:__
---
---* Corresponding C source code: [lstatslib.c#L289-L389](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstatslib.c#L289-L389)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/status.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class StatusInfo
---@field output_active boolean # `true` if the output routine is active, for example `false`.
---@field best_page_break Node # The current best break (a node)
---
---@field filename string # The name of the current input file , for example `./list.tex`.
---@field inputid integer # The numeric id of the current input, for example `2118770`.
---@field linenumber integer # The location in the current input file, for example `12`.
---
---@field lasterrorstring string # The last *TeX* error string.
---@field lastluaerrorstring string # The last *Lua* error string.
---@field lastwarningtag string # The last warning string.
---@field lastwarningstring string # The last warning tag, normally an indication of in what part.
---@field lasterrorcontext string # The last error context string (with newlines).
---
---@field pdf_gone integer # Written *PDF* bytes, for example `0`.
---@field pdf_ptr integer # Not yet written *PDF* bytes, for example `0`.
---@field dvi_gone integer # Written *DVI* bytes, for example `1631423304`.
---@field dvi_ptr integer # Not yet written *DVI* bytes, for example `-1858237624`.
---
---@field total_pages integer # The total pages of written pages, for example `0`.
---@field output_file_name string # The name of the *PDF* or *DVI* file.
---@field log_name string # The name of the log file, for example `list.log`.
---@field banner string # The terminal display banner, for example `This is LuaHBTeX, Version 1.15.0 (TeX Live 2022)`.
---@field luatex_version integer # The *LuaTeX* version number, for example `115`.
---@field luatex_revision integer # The *LuaTeX* revision string.
---@field development_id integer # for example `7509`.
---@field luatex_hashtype string # The hash method used (in *Lua*JITTEX), for example `lua`.
---@field luatex_hashchars integer # The length to which *Lua* hashes strings (`2^n`), for example `6`.
---@field luatex_engine string # The *LuaTeX* engine identifier, for example `luahbtex`.
---
---@field ini_version boolean # `true` if this is an *iniTeX* run, for example `false`.
---
---@field shell_escape integer # `0` means disabled, `1` means anything is permitted, and `2` is restricted, for example `2`.
---@field safer_option integer # `1` means safer is enforced, for example `0`.
---@field kpse_used integer # `1` means that kpse is used, for example `1`.
---
---@field var_used integer # The variable (one-word) memory in use, for example `309`.
---@field dyn_used integer # The token (multi-word) memory in use, for example `383564`.
---
---@field str_ptr integer # The string pointer, for example `2119239`.
---@field init_str_ptr integer # The *iniTeX* string pointer, for example `2118768`.
---@field max_strings integer # The maximum allowed strings, for example `500000`.
---@field pool_ptr string # The pool index, for example `393692`.
---@field init_pool_ptr integer # The *iniTeX* string pool index, for example `0`.
---@field pool_size integer # The current size allocated for string characters, for example `20000`.
---@field var_mem_max integer # Maximum variable memory of allocated words for nodes, for example `100000`.
---@field node_mem_usage string # A string giving insight into currently used nodes, for example `2 hlist, 1 rule, 1 dir, 3 kern, 1 glyph, 3 attribute, 48 glue_spec, 3 attribute_list, 1 write`.
---@field fix_mem_max integer # The maximum number of allocated words for tokens, for example `1977958`.
---@field fix_mem_min integer # The minimum number of allocated words for tokens, for example `0`.
---@field fix_mem_end integer # maximum number of used tokens, for example `1869998`.
---@field cs_count integer # The count of control sequences, for example `20658`.
---@field hash_size integer # The size of the hash, for example `65536`.
---@field hash_extra integer # The extra allowed hash, for example `600000`.
---@field font_ptr integer # The active fonts pointer, for example `15`.
---@field max_in_stack integer # The maximum of the used input stack entries, for example `33`.
---@field max_nest_stack integer # The maximum of the used nesting stack entries, for example `0`.
---@field max_param_stack integer # The maximum of the used parameter stack entries, for example `37`.
---@field max_buf_stack integer # The maximum used buffer position, for example `126`.
---@field max_save_stack integer # The maximum of the used save stack entries, for example `50`.
---@field stack_size integer # The input stack size, for example `10000`.
---@field nest_size integer # The nesting stack size, for example `1000`.
---@field param_size integer # The parameter stack size, for example `20000`.
---@field buf_size integer # The current allocated size of the line buffer, for example `200000`.
---@field save_size integer # The save stack size, for example `200000`.
---@field input_ptr integer # The level of input we're at, for example `1`.
---@field obj_ptr integer # The *PDF* object pointer.
---@field obj_tab_size integer # The *PDF* object table size, for example `1000`.
---@field pdf_os_cntr integer # The *PDF* object stream pointer, for example `0`.
---@field pdf_os_objidx integer # The *PDF* object stream index, for example `0`.
---@field pdf_dest_names_ptr integer # max *PDF* destination pointer, for example `0`.
---@field dest_names_size integer # The *PDF* destination table size, for example `1000`.
---@field pdf_mem_ptr integer # The *PDF* memory used, for example `1`.
---@field pdf_mem_size integer # The *PDF* memory size, for example `10000`.
---@field largest_used_mark integer # The largest referenced marks class, for example `0`.
---@field luabytecodes integer # The active *Lua* bytecode registers, for example `1`.
---@field luabytecode_bytes integer # The used bytes in the *Lua* bytecode registers, for example `32`.
---@field luastate_bytes integer # The bytes in use by *Lua* interpreters, for example `54667449`.
---
---@field callbacks integer # The total number of executed callbacks so far, for example `9`.
---@field indirect_callbacks integer # The total number of those callbacks that were themselves a result of other callbacks (e.g. file readers), for example `0`.
---
---@field saved_callbacks integer # alias for `indirect_callbacks` for example `0`.
---@field late_callbacks integer # for example `0`.
---@field direct_callbacks integer for example `10`.
---@field function_callbacks integer # of active fonts, for example `238`.
---
---@field lc_collate string # the value of `LC_COLLATE` at startup time (becomes `C` at startup), for example `en_US.UTF-8`.
---@field lc_ctype string # the value of `LC_CTYPE` at startup time (becomes `C` at startup), for example `en_US.UTF-8`.
---@field lc_numeric string # the value of `LC_NUMERIC` at startup time, for example `de_DE.UTF-8`.

---
---The keys in the table are the known items, the value is the current value. Almost
---all of the values in `status` are fetched through a metatable at run-time
---whenever they are accessed, so you cannot use `pairs` on `status`,
---but you can use `pairs` on `info`, of course. If you do not
---need the full list, you can also ask for a single item by using its name as an
---index into `status`.
---
---__Example:__
---
---```tex
---\catcode`_=12
---\directlua{
---  info = status.list()
---  for key, value in pairs(info) do
---    tex.print('"' .. key .. '": '.. tostring(value), '\string\\par')
---  end
---}
---```
---
---__Reference:__
---
---* Corresponding C source code: [lstatslib.c#L476-L488](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstatslib.c#L476-L488)
---
---@return StatusInfo
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/status.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function status.list() end

---
---Wipe error and warning messages.
---
---__Reference:__
---
---* Corresponding C source code: [lstatslib.c#L490-L501](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstatslib.c#L490-L501)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/status.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function status.resetmessages() end

---
---Set the exit code.
---
---__Reference:__
---
---* Corresponding C source code: [lstatslib.c#L503-L506](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstatslib.c#L503-L506)
---
---@param code integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/status.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function status.setexitcode(code) end

---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/gzip.lua
gzip = {}

---
---Open a file name using `gzopen`.
---
---Behaviour is identical to the description
---given in the `zlib` library.
---
---The `mode` parameter is as in fopen (`rb` or `wb`) but can also include a compression level (`wb9`) or a strategy: 'f' for filtered data as in `wb6f`, `h` for Huffman-only compression as in `wb1h`, `R` for run-length encoding as in `wb1R`, or `F` for fixed code compression as in `wb9F`. (See the description of deflateInit2 for more information about the strategy parameter.) `T` will request transparent writing or appending with no compression and not using the gzip format.
---
---`a` can be used instead of `w` to request that the gzip stream that will be written be appended to the file. `+` will result in an error, since reading and writing to the same gzip file is not supported. The addition of `x` when writing will create the file exclusively, which fails if the file already exists. On systems that support it, the addition of `e` when reading or writing will set the flag to close the file on an `execve()` call.
---
---It returns a new file handle, or, in case of errors, `nil` plus an error
---message
---
---__Example:__
---
---```lua
---local g_file = gzip.open('test.gz', 'wb9')
---assert(g_file)
---for i = 1, 10 do
---  g_file:write('This is line ' .. i, '\n')
---end
---g_file:close()
---```
---
---__Reference:__
---
---* Corresponding C source code: [lgzip.c#L132-L138](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lgzip.c#L132-L138)
---* zlib manual [manual.html](https://zlib.net/manual.html)
---
---@param filename string
---@param mode? string
---
---@return GFile|nil
---@return nil|string err
function gzip.open(filename, mode) end

---
---Same behaviour as io.lines in the `io` standard library provided by lua
---with the aditional feature of working with gzip files. If a normal text
---file is read it will read it normaly (normal gzopen behaviour).
---
---__Example:__
---
---```lua
---for line in gzip.lines('test.gz') do
---  print(line)
---end
---```
---
---__Reference:__
---
---* Corresponding C source code: [lgzip.c#L159-L166](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lgzip.c#L159-L166)
---
---@see io.lines
---
---@param filename string
---
---@return fun(): string
function gzip.lines(filename) end

---
---Close the file.
---
---__Example:__
---
---```lua
---local g_file = gzip.open('test.gz', 'rb')
---assert(g_file)
---local success = gzip.close(g_file)
---assert(success == true)
----- gzip.close(g_file) -> attempt to use a closed file
---```
---
---__Reference:__
---
---* Corresponding C source code: [lgzip.c#L107-L109](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lgzip.c#L107-L109)
---
---@see GFile.close
---
---@param file GFile
---
---@return true success
function gzip.close(file) end

---
---@class GFile
local GFile = {}

---
---This function takes no parameters and flushes all output to working file.
---The same as calling 'gzflush(file, Z_FINISH)' so writing to the file will
---most likely not work as expected. This is subject to change in the future
---if there is a strong reason for it to happen.
---
---__Reference:__
---
---* Corresponding C source code: [lgzip.c#L327-L329](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lgzip.c#L327-L329)
function GFile:flush() end

---
---Reads the file file, according to the given formats, which specify what
---to read. For each format, the function returns a string with the characters
---read, or nil if it cannot read data with the specified format. When called
---without formats, it uses a default format that reads the entire next line
---(see below).
---
---The available formats are
---
---`*a`   reads the whole file, starting at the current position. On end of
---file, it returns the empty string.
---`*l`   reads the next line (skipping the end of line), returning nil on
---end of file. This is the default format.
---number reads a string with up to that number of characters, returning
---nil on end of file. If number is zero, it reads nothing and
---returns an empty string, or nil on end of file.
---
---Unlink io.read, the `*n` format will not be available.
---
---__Reference:__
---
---* Corresponding C source code: [lgzip.c#L263-L265](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lgzip.c#L263-L265)
---
function GFile:read(format1, ...) end

---
---Returns an iterator function that, each time it is called, returns a new
---line from the file. Therefore, the construction
---for line in file:lines() do ... end
---will iterate over all lines of the file. (Unlike gzip.lines, this function
---does not close the file when the loop ends.)
---
---__Reference:__
---
---* Corresponding C source code: [lgzip.c#L152-L156](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lgzip.c#L152-L156)
---
function GFile:lines() end

---
---Sets and gets the file position, measured from the beginning of the file,
---to the position given by offset plus a base specified by the string whence,
---as follows:
---
---`set` base is position 0 (beginning of the file);
---`cur` base is current position;
---
---In case of success, function seek returns the final file position, measured in bytes from the beginning of ---the file. If this function fails, it returns nil, plus a string describing the error.
---The default value for whence is `cur`, and for offset is 0. Therefore, the call file:seek() returns the ---current file position, without changing it; the call file:seek("set") sets the position to the beginning of ---the file (and returns 0); and the call file:seek("end") sets the position to the end of the file, and ---returns its size.
---
---This function is subject to limitations imposed by `gzseek` function from
---`zlib` library, such as the inability to use `end` as the base for seeking
---and the inability to seek backwards when writing.
---
---__Reference:__
---
---* Corresponding C source code: [lgzip.c#L310-L324](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lgzip.c#L310-L324)
---
function GFile:seek(whence, offset) end

---
---Write the value of each of its arguments to the filehandle file.
---
---The
---arguments must be strings or numbers. To write other values, use tostring
---or string.format before write
---
---__Example:__
---
---```lua
---local g_file = gzip.open('test.gz', 'wb9')
---assert(g_file)
---for i = 1, 10 do
---  g_file:write('This is line ' .. i, '\n')
--end
---g_file:close()
---```
---
---__Reference:__
---
---* Corresponding C source code: [lgzip.c#L305-L307](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lgzip.c#L305-L307)
function GFile:write(value1, ...) end

---
---Close the file.
---
---__Example:__
---
---```lua
---local g_file = gzip.open('test.gz', 'wb9')
---assert(g_file)
---for i = 1, 10 do
---  g_file:write('This is line ' .. i, '\n')
---end
---g_file:close()
---```
function GFile:close() end

---
---https://github.com/TeX-Live/luatex/tree/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luamd5
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/517487384d3b0b4b003fb3180ea415f52eeb5f5f/source/texk/web2c/luatexdir/lua/luatex-core.lua#L220-L241
---Changes to upstream:
---* local md5 table
---* additional function md5.sumHEXA()

---
---The definitions are developed in this repository: https://github.com/LuaCATS/md5

---
---`md5`, by Roberto Ierusalimschy http://www.inf.puc-rio.br/~roberto/md5/md5-5/md5.html.
---
---This library offers basic cryptographic facilities for Lua 5.0: a hash (digest) function, an a pair crypt/decrypt.
---
---All functions are registered inside a table `md5`.
---
---😱 [Types](https://github.com/LuaCATS/md5/blob/main/library/md5.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/md5/pulls)
md5 = {}

---
---Compute the MD5 message-digest of the string `message`.
---
---This function takes as input a message of arbitrary length and content
---and returns as output a 128-bit "fingerprint" (or "message digest")
---of the input.
---
---The output is formated as a binary string with 16 characters.
---It is conjectured that it is computationally infeasible to produce
---two messages having the same message digest, or to produce any
---message having a given pre-specified target message digest.
---(see
---RFC 1321)
---
---__Example:__
---
---```lua
---local function convert_to_hex(str)
---  return (str:gsub('.', function (c)
---      return string.format('%02X', string.byte(c))
---  end))
---end
---local hash = convert_to_hex(md5.sum('test'))
---assert(hash == '098F6BCD4621D373CADE4E832627B4F6')
---```
---
---@param message string
---
---@return string
---
---😱 [Types](https://github.com/LuaCATS/md5/blob/main/library/md5.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/md5/pulls)
function md5.sum(message) end

---
---Compute the MD5 lower case hexadecimal message-digest of the string `message`.
---
---Similar to `md5.sum()`,
---but returns its value as a string of 32 hexadecimal digits (lower case letters).
---
---__Example:__
---
---```lua
---local hash = md5.sumhexa('test')
---assert(hash == '098f6bcd4621d373cade4e832627b4f6')
---```
---
---@param message string
---
---@return string # for example `098f6bcd4621d373cade4e832627b4f6`
---
---😱 [Types](https://github.com/LuaCATS/md5/blob/main/library/md5.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/md5/pulls)
function md5.sumhexa(message) end

---
---Compute the MD5 upper case hexadecimal message-digest of the string `message`.
---
---Similar to `md5.sum()`
---but returns its value as a string of 32 hexadecimal digits (upper case letters).
---
---__Example:__
---
---```lua
---local hash = md5.sumHEXA('test')
---assert(hash == '098F6BCD4621D373CADE4E832627B4F6')
---```
---
---@param message string
---
---@return string # for example `098F6BCD4621D373CADE4E832627B4F6`
function md5.sumHEXA(message) end

---
---Encrypt a string, using MD5 in CFB (Cipher-feedback mode).
---
---`message` is an arbitrary binary string to be encrypted.
---`key` is an arbitrary binary string to be used as a key.
---`seed` is an arbitrary binary string to be used as a seed;
---Returns the cyphertext (as a binary string).
---
---If no seed is provided,
---the function uses the result of `os.time()` as a seed.
---It is recommended that you use different seeds for each message;
---the seed itself is not private, and should contain no private data,
---because it goes plain in the beginning of the encrypted message.
---
---The length of the cyphertext is the length of the message plus the
---length of the seed plus one.
---
---__Example:__
---
---```lua
---local function convert_to_hex(str)
---  return (str:gsub('.', function(c)
---      return string.format('%02X', string.byte(c))
---  end))
---end
---
---local result = convert_to_hex(md5.crypt('secret', '123', 'seed'))
---assert(result == '0473656564974C4C1B3848')
---```
---
---@param message string # An arbitrary binary string to be encrypted.
---@param key string # An arbitrary binary string to be used as a key.
---@param seed? string # An optional arbitrary binary string to be used as a seed. If no seed is provided, the function uses the result of `time()` as a seed.
---
---@return string cyphertext # The cyphertext (as a binary string).
---
---😱 [Types](https://github.com/LuaCATS/md5/blob/main/library/md5.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/md5/pulls)
function md5.crypt(message, key, seed) end

---
---Decrypt a string.
---
---The input `message` must be the result of a previous call
---to `crypt`.
---For any `message`, `key`,
---and `seed`, we have that
---
---```lua
---local encrypted = md5.crypt('message', '123', 'seed')
---local message = md5.decrypt(encrypted, '123')
---assert(message == 'message')
---```
---
---@param message string # message to be decrypted (this must be the result of a previous call to `crypt`).
---@param key string # arbitrary binary string to be used as a key.
---
---@return string plaintext # The plaintext.
---
---😱 [Types](https://github.com/LuaCATS/md5/blob/main/library/md5.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/md5/pulls)
function md5.decrypt(message, key) end

---
---Do a bit-a-bit exclusive `or` of strings `s1` and `s2`.
---
---Both strings must have the same length,
---which will be also the length of the resulting string.
---
---__Example:__
---
---```lua
---assert(md5.exor('', '') == '')
---assert(md5.exor('alo alo', '\0\0\0\0\0\0\0') == 'alo alo')
---```
---
---@param s1 string # arbitrary binary string.
---@param s2 string # arbitrary binary string with same length as `s1`.
---
---@return string # a binary string with same length as `s1` and `s2`, where each bit is the exclusive `or` of the corresponding bits in `s1`-`s2`.
---
---😱 [Types](https://github.com/LuaCATS/md5/blob/main/library/md5.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/md5/pulls)
function md5.exor(s1, s2) end
---
---The `string` library has a few extra functions.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/string.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
string = {}

---
---Break a string into pieces.
---
---This functions splits a stringa into sub-strings based
---on the value of the string argument `separator`.
---The second argument is a string that is either empty (this splits the string into
---characters), a single character (this splits on each occurrence of that
---character, possibly introducing empty strings), or a single character followed by
---the plus sign `+` (this special version does not create empty sub-strings).
---The default value for `separator` is ` +` (multiple spaces). Note:
---`separator` is not hidden by surrounding braces as it would be if this function was
---written in *TeX* macros.
---
---__Example:__
---
---```lua
---for _, word in ipairs(string.explode("one  two three")) do
---  print(word)
---end
----- one
----- two
----- three
---
---for _, word in ipairs(string.explode("one,,two,three", ',')) do
---  print(word)
---end
---
----- one
-----
----- two
----- three
---
---for _, word in ipairs(string.explode("one,,two,three", ',+')) do
---  print(word)
---end
----- one
----- two
----- three
---```
---
---__Reference:__
---
---* Corresponding C source code: [lstrlibext.c#L247-309](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/4f2b914d365bab8a2747afe6e8c86d0f1c8475f7/source/texk/web2c/luatexdir/lua/lstrlibext.c#L247-309)
---* https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/4f2b914d365bab8a2747afe6e8c86d0f1c8475f7/manual/luatex-lua.tex#L399-409
---
---@param text string # A text that is to be divided into several substrings.
---@param separator? string # A separator that is used to split the string (default ` +`).
---
---@return string[]
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/string.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function string.explode(text, separator) end

---
---Return the Unicode codepoints of the characters in the given string.
---
---__Example:__
---
---```lua
---local a = string.utfvalue("abc")
---print(a) -- 97
---
---local a, b, c = string.utfvalue("abc")
---print(a, b, c) -- 97 98 99
---
---print(string.utfvalue("abc")) -- 97 98 99
---```
---
---__Reference:__
---
---* Corresponding C source code: [lstrlibext.c#L421-L461](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstrlibext.c#L421-L461)
---* Source file of the `LuaTeX` manual: [luatex-lua.tex#L418](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/4f2b914d365bab8a2747afe6e8c86d0f1c8475f7/manual/luatex-lua.tex#L418)
---
---@see string.utfvalues
---
---@param text string # The input string.
---
---@return integer, integer ...  # The Unicode codepoints of the characters in the given string.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/string.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function string.utfvalue(text) end

---
---Provide an iterator function that iterates over each character of the
---string by returning an integer value in the Unicode range.
---
---__Example:__
---
---```lua
---for code_point in string.utfvalues("abc") do
---  print(code_point)
---end
----- 97
----- 98
----- 99
---```
---__Reference:__
---
---* Corresponding C source code: [lstrlibext.c#L183-L189](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstrlibext.c#L183-L189)
---* Source file of the `LuaTeX` manual: [luatex-lua.tex#L418](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/4f2b914d365bab8a2747afe6e8c86d0f1c8475f7/manual/luatex-lua.tex#L418)
---
---@see string.utfvalue
---
---@param text string # The input string.
---
---@return fun(): integer code_point   # an integer value in the *Unicode* range
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/string.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function string.utfvalues(text) end

---
---Convert multiple unicode code points into a string.
---
---__Example:__
---
---```
---print(string.utfcharacter(97, 98, 99)) -- abc
---```
---
---__Reference:__
---
---* Corresponding C source code: [lstrlibext.c#L402-L412](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstrlibext.c#L402-L412)
---* Source file of the `LuaTeX` manual: [luatex-lua.tex#L421](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/4f2b914d365bab8a2747afe6e8c86d0f1c8475f7/manual/luatex-lua.tex#L421)
---
---@param code_point integer # A Unicode code point
---@param ... integer # For each character a integer argument
---
---@return string # A string with the characters of the given code points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/string.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function string.utfcharacter(code_point, ...) end

---
---Provide an iterator function that iterates over each character of the
---string by returning a string with a single UTF-8 token in it.
---
---__Example:__
---
---```lua
---for character in string.utfcharacters("\u{61}\u{62}\u{63}") do
---  print(character)
---end
----- a
----- b
----- c
---```
---
---__Reference:__
---
---* Corresponding C source code: [lstrlibext.c#L128-L134](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstrlibext.c#L128-L134)
---* Source file of the `LuaTeX` manual: [luatex-lua.tex#L469-470](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/4f2b914d365bab8a2747afe6e8c86d0f1c8475f7/manual/luatex-lua.tex#L469-470)
---
---@param text string # The input string.
---
---@return fun(): string character # a string with a single *UTF-8* token in it
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/string.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function string.utfcharacters(text) end

---
---Return the length of the given string.
---
---__Example:__
---
---```lua
---print(string.len("äöü"))
---print(string.utflength("äöü"))
---```
---
---__Reference:__
---
---* Corresponding C source code: [lstrlibext.c#L465-L488](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstrlibext.c#L465-L488)
---* Source file of the `LuaTeX` manual: [luatex-lua.tex#L473](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/4f2b914d365bab8a2747afe6e8c86d0f1c8475f7/manual/luatex-lua.tex#L473)
---
---@param text string # The input string.
---
---@return integer # The length of the given string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/string.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function string.utflength(text) end

---
---Provide an iterator function that iterates over each character of the
---string by returning a string containing __one byte__.
---
---__Example:__
---
---```lua
---for character in string.characters('abc') do
---  print(character)
---end
----- a
----- b
----- c
---
---for character in string.characters('äöü') do
---  print(character)
---end
----- �
----- �
----- �
----- �
----- �
----- �
---```
---
---__Reference:__
---
---* Corresponding C source code: [lstrlibext.c#L239-L245](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstrlibext.c#L239-L245)
---* Source file of the `LuaTeX` manual: [luatex-lua.tex#L424](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/4f2b914d365bab8a2747afe6e8c86d0f1c8475f7/manual/luatex-lua.tex#L424)
---
---@see string.bytes
---
---@param text string # The input string.
---
---@return fun(): string # A string containing __one byte__.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/string.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function string.characters(text) end

---
---Provide an iterator function that iterates over each character of the
---string by returning two strings.
---
---Each of these returned strings contains __one byte__ or an __empty__
---second string if the input string length was odd.
---
---__Example:__
---
---```lua
---for c1, c2 in string.characterpairs('äöü') do
---  print(c1, c2)
---  print(c1 .. c2)
---end
----- �	�
----- ä
----- �	�
----- ö
----- �	�
----- ü
---
---for c1, c2 in string.characterpairs('a') do
---  print("'" .. c1 .. "'", "'" .. c2 .. "'")
---end
----- 'a'	''
---```
---
---__Reference:__
---
---* Corresponding C source code: [lstrlibext.c#L216-L222](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstrlibext.c#L216-L222)
---* Source file of the `LuaTeX` manual: [luatex-lua.tex#L427-428](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/4f2b914d365bab8a2747afe6e8c86d0f1c8475f7/manual/luatex-lua.tex#L427-428)
---
---@see string.bytepairs
---
---@param text string # The input string.
---
---@return fun(): string, string # Two strings of __one byte__ each, or an __empty__ second string if the string length was odd.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/string.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function string.characterpairs(text) end

---
---Provide an iterator function that iterates over each character of the
---string by returning a single byte value.
---
---__Example:__
---
---```lua
---for byte in string.bytes('abc') do
---  print(byte)
---end
----- 97
----- 98
----- 99
---
---for byte in string.bytes('äöü') do
---  print(byte)
---end
----- 195
----- 164
----- 195
----- 182
----- 195
----- 188
---```
---
---__Reference:__
---
---* Corresponding C source code: [lstrlibext.c#L85-L91](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstrlibext.c#L85-L91)
---* Source file of the `LuaTeX` manual: [luatex-lua.tex#L431](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/4f2b914d365bab8a2747afe6e8c86d0f1c8475f7/manual/luatex-lua.tex#L431)
---
---@see string.characters
---
---@param text string # The input string.
---
---@return fun(): integer # A single byte value.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/string.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function string.bytes(text) end

---
---Provide an iterator function that iterates over each character of the
---string by returning two byte values or `nil`.
---
---If the input string has an odd length, `nil` is returned.
---
---__Example:__
---
---```lua
---for b1, b2 in string.bytepairs('abc') do
---  print(b1, b2)
---end
----- 97	98
----- 99	nil
---
---for b1, b2 in string.bytepairs('äöü') do
---  print(b1, b2)
---end
----- 195	164
----- 195	182
----- 195	188
---```
---
---__Reference:__
---
---* Corresponding C source code: [lstrlibext.c#L62-L68](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lstrlibext.c#L62-L68)
---* Source file of the `LuaTeX` manual: [luatex-lua.tex#L434-435](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/4f2b914d365bab8a2747afe6e8c86d0f1c8475f7/manual/luatex-lua.tex#L434-435)
---
---@see string.characterpairs
---
---@param text string # The input string.
---
---@return fun(): integer, integer|nil # Two byte values or `nil` as the second return value if the input string length was odd.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/string.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function string.bytepairs(text) end
---
---# Binary input from strings with `sio`
---
---A similar set of function as in the `fio` library is available in the `sio` library: `sio.readcardinal1`, `sio.readcardinal2`,
---`sio.readcardinal3`, `sio.readcardinal4`, `sio.readcardinaltable`, `sio.readinteger1`, `sio.readinteger2`,
---`sio.readinteger3`, `sio.readinteger4`, `sio.readintegertable`, `sio.readfixed2`, `sio.readfixed4`,
---`sio.read2dot14`, `sio.setposition`, `sio.getposition`, `sio.skipposition`, `sio.readbytes` and
---`sio.readbytetable`. Here the first argument is a string instead of a
---file handle. More details can be found in the previous section.
---
---|                            |                                     |
---|----------------------------|-------------------------------------|
---| `readcardinal1(s)`         | a 1 byte unsigned integer           |
---| `readcardinal2(s)`         | a 2 byte unsigned integer           |
---| `readcardinal3(s)`         | a 3 byte unsigned integer           |
---| `readcardinal4(s)`         | a 4 byte unsigned integer           |
---| `readcardinaltable(s,n,b)` | `n` cardinals of `b` bytes          |
---| `readinteger1(s)`          | a 1 byte signed integer             |
---| `readinteger2(s)`          | a 2 byte signed integer             |
---| `readinteger3(s)`          | a 3 byte signed integer             |
---| `readinteger4(s)`          | a 4 byte signed integer             |
---| `readintegertable(s,n,b)`  | `n` integers of `b` bytes           |
---| `readfixed2(s)`            | a 2 byte float (used in font files) |
---| `readfixed4(s)`            | a 4 byte float (used in font files) |
---| `read2dot14(s)`            | a 2 byte float (used in font files) |
---| `setposition(s,p)`         | goto position `p`                   |
---| `getposition(s)`           | get the current position            |
---| `skipposition(s,n)`        | skip `n` positions                  |
---| `readbytes(s,n)`           | `n` bytes                           |
---| `readbytetable(f,n)`       | `n` bytes                           |
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
sio = {}

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L124-L135](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L124-L135)
---
---@param s string
---
---@return integer # a 1 byte unsigned integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readcardinal1(s) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L160-L172](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L160-L172)
---
---@param s string
---
---@return integer # a 2 byte unsigned integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readcardinal2(s) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L212-L225](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L212-L225)
---
---@param s string
---
---@return integer # a 3 byte unsigned integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readcardinal3(s) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L268-L282](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L268-L282)
---
---@param s string
---
---@return integer # a 4 byte unsigned integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readcardinal4(s) end

---
---@param s string
---@param n integer
---@param b integer
---
---@return table<integer, integer>
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readcardinaltable(s, n, b) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L124-L135](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L124-L135)
---
---little endian variant
---
---@param s string
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readcardinal1le(s) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L173-L185](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L173-L185)
---
---little endian variant
---
---@param s string
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readcardinal2le(s) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L226-L239](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L226-L239)
---
---little endian variant
---
---@param s string
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readcardinal3le(s) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L283-L297](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L283-L297)
---
---little endian variant
---
---@param s string
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readcardinal4le(s) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L444-L458](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L444-L458)
---
---@param s string
---
---@return integer # a 1 byte signed integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readinteger1(s) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L485-L500](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L485-L500)
---
---@param s string
---
---@return integer # a 2 byte signed integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readinteger2(s) end

---
---@param s string
---
---@return integer # a 3 byte signed integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readinteger3(s) end

---
---@param s string
---
---@return integer # a 4 byte signed integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readinteger4(s) end

---
---@param s string
---@param n integer
---@param b integer
---
---@return table<integer, integer>
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readintegertable(s, n, b) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L444-L458](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L444-L458)
---little endian variant
---
---@param s string
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readinteger1le(s) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L501-L516](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L501-L516)
---
---little endian variant
---
---@param s string
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readinteger2le(s) end

---
---little endian variant
---
---@param s string
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readinteger3le(s) end

---
---little endian variant
---
---@param s string
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readinteger4le(s) end

---
---@param s string
---
---@return number
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readfixed2(s) end

---
---@param s string
---
---@return number
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readfixed4(s) end

---
---@param s string
---
---@return number
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.read2dot14(s) end

---
---@param s string
---@param p integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.setposition(s, p) end

---
---@param s string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.getposition(s) end

---
---@param s string
---@param n integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.skipposition(s, n) end

---
---@param s string
---@param n integer
---
---@return integer ...
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readbytes(s, n) end

---
---@param s string
---@param n integer
---
---@return table<integer, integer>
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readbytetable(s, n) end

---
---@param s string
---
---@return string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/sio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function sio.readline(s) end

---
---This is a table that is created empty. A startup *Lua* script could
---fill this table with a number of settings that are read out by
---the executable after loading and executing the startup file.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig = {}

---
---`false` totally disables \KPATHSEA\ initialisation, and enables
---interpretation of the following numeric key--value pairs. (only ever unset
---this if you implement {\it all\/} file find callbacks!)
---
---`shell_escape`  string  `'f'`  Use `'y'` or `'t'` or `'1'` to enable `\write18` unconditionally, `'p'`
---to enable the commands that are listed in `shell_escape_commands`
---
---shell_escape_commands  string   Comma-separated list of command
---names that may be executed by `\write18` even if `shell_escape`
---is set to `'p'`. Do {\it not\/} use spaces around commas, separate any
---required command arguments by using a space, and use the \ASCII\ double quote
---(`"`) for any needed argument or path quoting
---@type boolean
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.kpse_init = true

---
---Minimum pool space after TeX's own strings; must be at least 25000 less than pool_size, but doesn't need to be nearly that large.
---
---__Reference:__
---
---* [texmf.cnf#L810-L811](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf#L810-L811)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.string_vacancies = 75000

---
---Min pool space left after loading .fmt.
---
---__Reference:__
---
---* [texmf.cnf#L815](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf#L815)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.pool_free = 5000

---
---Maximum number of strings.
---
---__Reference:__
---
---* [texmf.cnf#L813](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf#L813)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.max_strings = 15000

---
---Ensure at least this many strings are free after loading .fmt.
---
---__Reference:__
---
---* [texmf.cnf#L817](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf#L817)
---
---@type integer
texconfig.strings_free = 100

---
---simultaneous semantic levels (e.g., groups)
---
---__Reference:__
---
---* [kpathsea/texmf.cnf](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.nest_size = 50

---
---simultaneous input files and error insertions, also applies to MetaPost
---
---__Reference:__
---
---* [kpathsea/texmf.cnf](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.max_in_open = 15

---
---simultaneous macro parameters, also applies to MP
---
---__Reference:__
---
---* [kpathsea/texmf.cnf](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.param_size = 60

---
---for saving values outside current group
---
---__Reference:__
---
---* [kpathsea/texmf.cnf](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.save_size = 4000

---
---simultaneous input sources
---
---__Reference:__
---
---* [kpathsea/texmf.cnf](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.stack_size = 300

---
---These work best if they are the same as the I/O buffer size, but it doesn't matter much.  Must be a multiple of 8.
---
---__Reference:__
---
---* [kpathsea/texmf.cnf](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.dvi_buf_size = 16384

---
---It's probably inadvisable to change these. At any rate, we must have:
---45 < error_line      < 255;
---30 < half_error_line < error_line - 15;
---60 <= max_print_line;
---These apply to TeX, Metafont, and MetaPost.
---"max_print_line" applies to BibTeX family
---
---__Reference:__
---
---* [kpathsea/texmf.cnf](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.error_line = 79

---
---It's probably inadvisable to change these. At any rate, we must have:
---45 < error_line      < 255;
---30 < half_error_line < error_line - 15;
---60 <= max_print_line;
---These apply to TeX, Metafont, and MetaPost.
---"max_print_line" applies to BibTeX family
---
---__Reference:__
---
---* [kpathsea/texmf.cnf](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.half_error_line = 50

---
---It's probably inadvisable to change these. At any rate, we must have:
---45 < error_line      < 255;
---30 < half_error_line < error_line - 15;
---60 <= max_print_line;
---These apply to TeX, Metafont, and MetaPost.
---"max_print_line" applies to BibTeX family
---
---__Reference:__
---
---* [kpathsea/texmf.cnf](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/2f443efc11d9b87ec0592ddd64c90d9ad15ea498/source/texk/kpathsea/texmf.cnf)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.max_print_line = 79

---
---Extra space for the hash table of control sequences.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.hash_extra = 0

---
---For pdftex and luatex: default resolution for bitmaps; commonly set via \pdfpkresolution when needed to be changed.
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.pk_dpi = 72

---
---`false` disables *TeX*'s normal file open-close feedback (the
---assumption is that callbacks will take care of that)
---@type boolean
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.trace_file_names = true

---
---do `file:line` style error messages
---@type boolean
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.file_line_error = false

---
---abort run on the first encountered error
---@type boolean
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.halt_on_error = false

---
---if no format name was given on the command line, this key will be tested first
---instead of simply quitting
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.formatname = ""

---
---if no input file name was given on the command line, this key will be tested
---first instead of simply giving up
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.jobname = ""

---
---character to put in front of traced macros (see next value)
---
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.level_chr = 0

---
---when larger than zero the input nesting level will be shown when `\tracingmacros` is set; levels above this value will be clipped with
---the level shown up front
---
---Note: the numeric values that match web2c parameters are only used if `kpse_init` is explicitly set to `false`. In all other cases, the normal
---values from `texmf.cnf` are used.
---
---You can kick in your own nesting level visualizer, for instance:
---
---```
---callback.register("input_level_string",function(n)
---         if tex.tracingmacros > 0 and tex.count.tracingstacklevels > 0 then
---    if tex.tracingmacros > 1 then
---        return "! " .. string.rep(">",n) .. " "
---    end
---end)
---```
---
---Or, in sync with other engines (not checked):
---
---\newcount\tracingstacklevels
---
---```
---\directlua {
---  callback.register("input_level_string", function(n)
---    if tex.tracingmacros > 0 then
---      local l = tex.count.tracingstacklevels
---      if l > 0 then
---        return string.rep(" ",l) .. string.rep(".",n-l)
---      end
---    end
---  end)
---}
---```
---@type integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texconfig.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texconfig.level_max = 0

---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luapeg/lpeg.c
---Changes to upstream: global lpeg table

---
---The definitions are developed in this repository: https://github.com/LuaCATS/lpeg

---
---This type definition is based on the [HTML documentation](http://www.inf.puc-rio.br/~roberto/lpeg/) of the LPeg library. A different HTML documentation can be found at http://stevedonovan.github.io/lua-stdlibs/modules/lpeg.html.
---
---*LPeg* is a new pattern-matching library for Lua,
---based on
---[Parsing Expression Grammars](https://bford.info/packrat/) (PEGs).
---This text is a reference manual for the library.
---For a more formal treatment of LPeg,
---as well as some discussion about its implementation,
---see
---[A Text Pattern-Matching Tool based on Parsing Expression Grammars](http://www.inf.puc-rio.br/~roberto/docs/peg.pdf).
---(You may also be interested in my
---[talk about LPeg](https://vimeo.com/1485123)
---given at the III Lua Workshop.)
---
---Following the Snobol tradition,
---LPeg defines patterns as first-class objects.
---That is, patterns are regular Lua values
---(represented by userdata).
---The library offers several functions to create
---and compose patterns.
---With the use of metamethods,
---several of these functions are provided as infix or prefix
---operators.
---On the one hand,
---the result is usually much more verbose than the typical
---encoding of patterns using the so called
---*regular expressions*
---(which typically are not regular expressions in the formal sense).
---On the other hand,
---first-class patterns allow much better documentation
---(as it is easy to comment the code,
---to break complex definitions in smaller parts, etc.)
---and are extensible,
---as we can define new functions to create and compose patterns.
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
lpeg = {}

---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
---@class Pattern
---@operator len: Pattern
---@operator unm: Pattern
---@operator add(Pattern): Pattern
---@operator sub(Pattern): Pattern
---@operator mul(Pattern): Pattern
---@operator mul(Capture): Pattern
---@operator div(string): Capture
---@operator div(integer): Capture
---@operator div(table): Capture
---@operator div(function): Capture
---@operator pow(integer): Pattern
---@operator mod(function): Capture
local Pattern = {}

---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
---@alias Capture Pattern
---@operator len: Pattern
---@operator add(Capture): Pattern
---@operator mul(Capture): Pattern
---@operator mul(Pattern): Pattern
---@operator div(string): Capture
---@operator div(integer): Capture
---@operator div(table): Capture
---@operator div(function): Capture
---@operator pow(integer): Pattern

---
---Match the given `pattern` against the `subject` string.
---
---If the match succeeds,
---returns the index in the subject of the first character after the match,
---or the captured values
---(if the pattern captured any value).
---
---An optional numeric argument `init` makes the match
---start at that position in the subject string.
---As usual in Lua libraries,
---a negative value counts from the end.
---
---Unlike typical pattern-matching functions,
---`match` works only in anchored mode;
---that is, it tries to match the pattern with a prefix of
---the given subject string (at position `init`),
---not with an arbitrary substring of the subject.
---So, if we want to find a pattern anywhere in a string,
---we must either write a loop in Lua or write a pattern that
---matches anywhere.
---This second approach is easy and quite efficient;
---
---__Example:__
---
---```lua
---local pattern = lpeg.R('az') ^ 1 * -1
---assert(pattern:match('hello') == 6)
---assert(lpeg.match(pattern, 'hello') == 6)
---assert(pattern:match('1 hello') == nil)
---```
---
---@param pattern Pattern|string|integer|boolean|table|function
---@param subject string
---@param init? integer
---@param ... any
---
---@return any ...
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.match(pattern, subject, init, ...) end

---
---Match the given `pattern` against the `subject` string.
---
---If the match succeeds,
---returns the index in the subject of the first character after the match,
---or the captured values
---(if the pattern captured any value).
---
---An optional numeric argument `init` makes the match
---start at that position in the subject string.
---As usual in Lua libraries,
---a negative value counts from the end.
---
---Unlike typical pattern-matching functions,
---`match` works only in anchored mode;
---that is, it tries to match the pattern with a prefix of
---the given subject string (at position `init`),
---not with an arbitrary substring of the subject.
---So, if we want to find a pattern anywhere in a string,
---we must either write a loop in Lua or write a pattern that
---matches anywhere.
---This second approach is easy and quite efficient;
---
---__Example:__
---
---```lua
---local pattern = lpeg.R('az') ^ 1 * -1
---assert(pattern:match('hello') == 6)
---assert(lpeg.match(pattern, 'hello') == 6)
---assert(pattern:match('1 hello') == nil)
---```
---
---@param subject string
---@param init? integer
---@param ... any
---
---@return any ...
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function Pattern:match(subject, init, ...) end

---
---Return the string `"pattern"` if the given value is a pattern, otherwise `nil`.
---
---@return 'pattern'|nil
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.type(value) end

---
---A string (not a function) with the running version of LPeg.
---
---Note: In earlier versions of LPeg this field was a function.
---
---@type string
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
lpeg.version = ""

---
---Set a limit for the size of the backtrack stack used by LPeg to
---track calls and choices.
---
---The default limit is `400`.
---Most well-written patterns need little backtrack levels and
---therefore you seldom need to change this limit;
---before changing it you should try to rewrite your
---pattern to avoid the need for extra space.
---Nevertheless, a few useful patterns may overflow.
---Also, with recursive grammars,
---subjects with deep recursion may also need larger limits.
---
---@param max integer
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.setmaxstack(max) end

---
---Convert the given value into a proper pattern.
---
---The following rules are applied:
---
---* If the argument is a pattern,
---it is returned unmodified.
---
---* If the argument is a string,
---it is translated to a pattern that matches the string literally.
---
---* If the argument is a non-negative number `n`,
---the result is a pattern that matches exactly `n` characters.
---
---* If the argument is a negative number `-n`,
---the result is a pattern that
---succeeds only if the input string has less than `n` characters left:
---`lpeg.P(-n)`
---is equivalent to `-lpeg.P(n)`
---(see the unary minus operation).
---
---* If the argument is a boolean,
---the result is a pattern that always succeeds or always fails
---(according to the boolean value),
---without consuming any input.
---
---* If the argument is a table,
---it is interpreted as a grammar
---(see Grammars).
---
---* If the argument is a function,
---returns a pattern equivalent to a
---match-time capture over the empty string.
---
---@param value Pattern|string|integer|boolean|table|function
---
---@return Pattern
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.P(value) end

---
---Return a pattern that
---matches only if the input string at the current position
---is preceded by `patt`.
---
---Pattern `patt` must match only strings
---with some fixed length,
---and it cannot contain captures.
---
---Like the `and` predicate,
---this pattern never consumes any input,
---independently of success or failure.
---
---@param pattern Pattern|string|integer|boolean|table
---
---@return Pattern
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.B(pattern) end

---
---Return a pattern that matches any single character
---belonging to one of the given ranges.
---
---Each `range` is a string `xy` of length 2,
---representing all characters with code
---between the codes of `x` and `y`
---(both inclusive).
---
---As an example, the pattern
---`lpeg.R('09')` matches any digit,
---and `lpeg.R('az', 'AZ')` matches any ASCII letter.
---
---__Example:__
---
---```lua
---local pattern = lpeg.R('az') ^ 1 * -1
---assert(pattern:match('hello') == 6)
---```
---
---@param ... string
---
---@return Pattern
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.R(...) end

---
---Return a pattern that matches any single character that
---appears in the given string.
---(The `S` stands for Set.)
---
---As an example, the pattern
---`lpeg.S('+-*/')` matches any arithmetic operator.
---
---Note that, if `s` is a character
---(that is, a string of length 1),
---then `lpeg.P(s)` is equivalent to `lpeg.S(s)`
---which is equivalent to `lpeg.R(s..s)`.
---Note also that both `lpeg.S('')` and `lpeg.R()`
---are patterns that always fail.
---
---@param string string
---
---@return Pattern
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.S(string) end

---
---Create a non-terminal (a variable) for a grammar.
---
---This operation creates a non-terminal (a variable)
---for a grammar.
---The created non-terminal refers to the rule indexed by `v`
---in the enclosing grammar.
---
---__Example:__
---
---```lua
---local b = lpeg.P({'(' * ((1 - lpeg.S '()') + lpeg.V(1)) ^ 0 * ')'})
---assert(b:match('((string))') == 11)
---assert(b:match('(') == nil)
---```
---
---@param v boolean|string|number|function|table|thread|userdata|lightuserdata
---
---@return Pattern
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.V(v) end

---
---@class Locale
---@field alnum userdata
---@field alpha userdata
---@field cntrl userdata
---@field digit userdata
---@field graph userdata
---@field lower userdata
---@field print userdata
---@field punct userdata
---@field space userdata
---@field upper userdata
---@field xdigit userdata

---
---Return a table with patterns for matching some character classes
---according to the current locale.
---
---The table has fields named
---`alnum`,
---`alpha`,
---`cntrl`,
---`digit`,
---`graph`,
---`lower`,
---`print`,
---`punct`,
---`space`,
---`upper`, and
---`xdigit`,
---each one containing a correspondent pattern.
---Each pattern matches any single character that belongs to its class.
--
---If called with an argument `table`,
---then it creates those fields inside the given table and
---returns that table.
---
---__Example:__
---
---```lua
---lpeg.locale(lpeg)
---local space = lpeg.space ^ 0
---local name = lpeg.C(lpeg.alpha ^ 1) * space
---local sep = lpeg.S(',;') * space
---local pair = lpeg.Cg(name * '=' * space * name) * sep ^ -1
---local list = lpeg.Cf(lpeg.Ct('') * pair ^ 0, rawset)
---local t = list:match('a=b, c = hi; next = pi')
---assert(t.a == 'b')
---assert(t.c == 'hi')
---assert(t.next == 'pi')
---
---local locale = lpeg.locale()
---assert(type(locale.digit) == 'userdata')
---```
---
---@param tab? table
---
---@return Locale
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.locale(tab) end

---
---Create a simple capture.
---
---Creates a simple capture,
---which captures the substring of the subject that matches `patt`.
---The captured value is a string.
---If `patt` has other captures,
---their values are returned after this one.
---
---__Example:__
---
---```lua
---local function split (s, sep)
---  sep = lpeg.P(sep)
---  local elem = lpeg.C((1 - sep) ^ 0)
---  local p = elem * (sep * elem) ^ 0
---  return lpeg.match(p, s)
---end
---
---local a, b, c = split('a,b,c', ',')
---assert(a == 'a')
---assert(b == 'b')
---assert(c == 'c')
---```
---
---@param patt Pattern|string|integer|boolean|table|function
---
---@return Capture
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.C(patt) end

---
---Create an argument capture.
---
---This pattern matches the empty string and
---produces the value given as the nth extra
---argument given in the call to `lpeg.match`.
---
---@param n integer
---
---@return Capture
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.Carg(n) end

---
---Create a back capture.
---
---This pattern matches the empty string and
---produces the values produced by the most recent
---group capture named `name`
---(where `name` can be any Lua value).
---
---Most recent means the last
---complete
---outermost
---group capture with the given name.
---A Complete capture means that the entire pattern
---corresponding to the capture has matched.
---An Outermost capture means that the capture is not inside
---another complete capture.
---
---In the same way that LPeg does not specify when it evaluates captures,
---it does not specify whether it reuses
---values previously produced by the group
---or re-evaluates them.
---
---@param name any
---
---@return Capture
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.Cb(name) end

---
---Create a constant capture.
---
---This pattern matches the empty string and
---produces all given values as its captured values.
---
---@param ... any
---
---@return Capture
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.Cc(...) end

---
---Create a fold capture.
---
---If `patt` produces a list of captures
---C1 C2 ... Cn,
---this capture will produce the value
---`func(...func(func(C1, C2), C3)...,Cn)`,
---that is, it will fold
---(or accumulate, or reduce)
---the captures from `patt` using function `func`.
---
---This capture assumes that `patt` should produce
---at least one capture with at least one value (of any type),
---which becomes the initial value of an accumulator.
---(If you need a specific initial value,
---you may prefix a constant capture to `patt`.)
---For each subsequent capture,
---LPeg calls `func`
---with this accumulator as the first argument and all values produced
---by the capture as extra arguments;
---the first result from this call
---becomes the new value for the accumulator.
---The final value of the accumulator becomes the captured value.
---
---__Example:__
---
---```lua
---local number = lpeg.R('09') ^ 1 / tonumber
---local list = number * (',' * number) ^ 0
---local function add(acc, newvalue) return acc + newvalue end
---local sum = lpeg.Cf(list, add)
---assert(sum:match('10,30,43') == 83)
---```
---
---@param patt Pattern|string|integer|boolean|table|function
---@param func fun(acc, newvalue): (acc: any)
---
---@return Capture
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.Cf(patt, func) end

---
---Create a group capture.
---
---It groups all values returned by `patt`
---into a single capture.
---The group may be anonymous (if no name is given)
---or named with the given name
---(which can be any non-nil Lua value).
---
---@param patt Pattern|string|integer|boolean|table|function
---@param name? string
---
---@return Capture
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.Cg(patt, name) end

---
---Create a position capture.
---
---It matches the empty string and
---captures the position in the subject where the match occurs.
---The captured value is a number.
---
---__Example:__
---
---```lua
---local I = lpeg.Cp()
---local function anywhere(p) return lpeg.P({I * p * I + 1 * lpeg.V(1)}) end
---
---local match_start, match_end = anywhere('world'):match('hello world!')
---assert(match_start == 7)
---assert(match_end == 12)
---```
---
---@return Capture
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.Cp() end

---
---Create a substitution capture.
---
---This function creates a substitution capture,
---which captures the substring of the subject that matches `patt`,
---with substitutions.
---For any capture inside `patt` with a value,
---the substring that matched the capture is replaced by the capture value
---(which should be a string).
---The final captured value is the string resulting from
---all replacements.
---
---__Example:__
---
---```lua
---local function gsub (s, patt, repl)
---  patt = lpeg.P(patt)
---  patt = lpeg.Cs((patt / repl + 1) ^ 0)
---  return lpeg.match(patt, s)
---end
---assert(gsub('Hello, xxx!', 'xxx', 'World') == 'Hello, World!')
---```
---
---@param patt Pattern|string|integer|boolean|table|function
---
---@return Capture
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.Cs(patt) end

---
---Create a table capture.
---
---This capture returns a table with all values from all anonymous captures
---made by `patt` inside this table in successive integer keys,
---starting at 1.
---Moreover,
---for each named capture group created by `patt`,
---the first value of the group is put into the table
---with the group name as its key.
---The captured value is only the table.
---
---@param patt Pattern|string|integer|boolean|table|function
---
---@return Capture
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.Ct(patt) end

---
---Create a match-time capture.
---
---Unlike all other captures,
---this one is evaluated immediately when a match occurs
---(even if it is part of a larger pattern that fails later).
---It forces the immediate evaluation of all its nested captures
---and then calls `function`.
---
---The given function gets as arguments the entire subject,
---the current position (after the match of `patt`),
---plus any capture values produced by `patt`.
---
---The first value returned by `fn`
---defines how the match happens.
---If the call returns a number,
---the match succeeds
---and the returned number becomes the new current position.
---(Assuming a subject and current position `i`,
---the returned number must be in the range `[i, len(s) + 1]`.)
---If the call returns true,
---the match succeeds without consuming any input.
---(So, to return true is equivalent to return `i`.)
---If the call returns `false`, `nil`, or no value,
---the match fails.
---
---Any extra values returned by the function become the
---values produced by the capture.
---
---@param patt Pattern|string|integer|boolean|table|function
---@param fn fun(s: string, i: integer, ...: any): (position: boolean|integer, ...: any)
---
---@return Capture
---
---😱 [Types](https://github.com/LuaCATS/lpeg/blob/main/library/lpeg.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lpeg/pulls)
function lpeg.Cmt(patt, fn) end

---
---The node library contains functions that facilitate dealing with (lists of) nodes and their values.
---They allow you to create, alter, copy, delete, and insert LuaTEX node objects, the core objects
---within the typesetter.
---
---LuaTEX nodes are represented in Lua as userdata with the metadata type `luatex.node.` The
---various parts within a node can be accessed using named fields.
---
---Each node has at least the three fields next, id, and subtype:
---
---* The next field returns the userdata object for the next node in a linked list of nodes, or nil,
---  if there is no next node.
---* The id indicates TEX’s ‘node type’. The field id has a numeric value for efficiency reasons,
---  but some of the library functions also accept a string value instead of id.
---* The subtype is another number. It often gives further information about a node of a particular
---  id, but it is most important when dealing with ‘whatsits’, because they are differentiated
---  solely based on their subtype.
---
---The other available fields depend on the id (and for ‘whatsits’, the subtype) of the node.
---
---Support for unset (alignment) nodes is partial: they can be queried and modified from Lua code,
---but not created.
---
---Nodes can be compared to each other, but: you are actually comparing indices into the node
---memory. This means that equality tests can only be trusted under very limited conditions. It will
---not work correctly in any situation where one of the two nodes has been freed and/or reallocated:
---in that case, there will be false positives.
---
---At the moment, memory management of nodes should still be done explicitly by the user. Nodes
---are not ‘seen’ by the Lua garbage collector, so you have to call the node freeing functions yourself
---when you are no longer in need of a node (list). Nodes form linked lists without reference
---counting, so you have to be careful that when control returns back to LuaTEX itself, you have
---not deleted nodes that are still referenced from a next pointer elsewhere, and that you did not
---create nodes that are referenced more than once. Normally the setters and getters handle this
---for you.
---
---There are statistics available with regards to the allocated node memory, which can be handy
---for tracing.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
node = {}

---
---Deep down in TEX a node has a number which is a numeric entry in a memory table. In fact, this
---model, where TEX manages memory is real fast and one of the reasons why plugging in callbacks
---that operate on nodes is quite fast too. Each node gets a number that is in fact an index in the
---memory table and that number often is reported when you print node related information. You
---go from userdata nodes and there numeric references and back with:
---
---```
---<integer> d = node.todirect(<node> n)
---<node> n = node.tonode(<integer> d)
---```
---
---The userdata model is rather robust as it is a virtual interface with some additional checking
---while the more direct access which uses the node numbers directly. However, even with userdata
---you can get into troubles when you free nodes that are no longer allocated or mess up lists. if
---you apply tostring to a node you see its internal (direct) number and id.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
node.direct = {}

---
---*LuaTeX* only understands 4 of the 16 direction specifiers of aleph: `TLT` (latin), `TRT` (arabic), `RTT` (cjk), `LTL` (mongolian). All other direction specifiers generate an error.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias DirectionSpecifier
---|"TLT" # 0 latin
---|"TRT" # 1 arabic
---|"LTL" # 2 mongolian
---|"RTT" # 3 cjk

---
---In addition to a keyword driven model we also provide an integer driven one.
---@alias DirectionSpecifierId
---|0 # "TLT" latin
---|1 # "TRT" arabic
---|2 # "LTL" mongolian
---|3 # "RTT" cjk

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L493-L542](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L493-L542)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias NodeTypeName
---| 'hlist' # 0 `hlist_node`
---| 'vlist' # 1 `vlist_node`
---| 'rule' # 2 `rule_node`
---| 'ins' # 3 `ins_node`
---| 'mark' # 4 `mark_node`
---| 'adjust' # 5 `adjust_node`
---| 'boundary' # 6 `boundary_node`
---| 'disc' # 7 `disc_node`
---| 'whatsit' # 8 `whatsit_node`
---| 'local_par' # 9 `local_par_node`
---| 'dir' # 10 `dir_node`
---| 'math' # 11 `math_node`
---| 'glue' # 12 `glue_node`
---| 'kern' # 13 `kern_node`
---| 'penalty' # 14 `penalty_node`
---| 'unset' # 15 `unset_node`
---| 'style' # 16 `style_node`
---| 'choice' # 17 `choice_node`
---| 'noad' # 18 `simple_noad`
---| 'radical' # 19 `radical_noad`
---| 'fraction' # 20 `fraction_noad`
---| 'accent' # 21 `accent_noad`
---| 'fence' # 22 `fence_noad`
---| 'math_char' # 23 `math_char_node`
---| 'sub_box' # 24 `sub_box_node`
---| 'sub_mlist' # 25 `sub_mlist_node`
---| 'math_text_char' # 26 `math_text_char_node`
---| 'delim' # 27 `delim_node`
---| 'margin_kern' # 28 `margin_kern_node`
---| 'glyph' # 29 `glyph_node`
---| 'align_record' # 30 `align_record_node`
---| 'pseudo_file' # 31 `pseudo_file_node`
---| 'pseudo_line' # 32 `pseudo_line_node`
---| 'page_insert' # 33 `inserting_node`
---| 'split_insert' # 34 `split_up_node`
---| 'expr_stack' # 35 `expr_node`
---| 'nested_list' # 36 `nesting_node`
---| 'span' # 37 `span_node`
---| 'attribute' # 38 `attribute_node`
---| 'glue_spec' # 39 `glue_spec_node`
---| 'attribute_list' # 40 `attribute_list_node`
---| 'temp' # 41 `temp_node`
---| 'align_stack' # 42 `align_stack_node`
---| 'movement_stack' # 43 `movement_node`
---| 'if_stack' # 44 `if_node`
---| 'unhyphenated' # 45 `unhyphenated_node`
---| 'hyphenated' # 46 `hyphenated_node`
---| 'delta' # 47 `delta_node`
---| 'passive' # 48 `passive_node`
---| 'shape' # 49 `shape_node`

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L493-L542](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L493-L542)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias NodeTypeId
---| 0  # hlist `hlist_node`
---| 1  # vlist `vlist_node`
---| 2  # rule `rule_node`
---| 3  # ins `ins_node`
---| 4  # mark `mark_node`
---| 5  # adjust `adjust_node`
---| 6  # boundary `boundary_node`
---| 7  # disc `disc_node`
---| 8  # whatsit `whatsit_node`
---| 9  # local_par `local_par_node`
---| 10 # dir `dir_node`
---| 11 # math `math_node`
---| 12 # glue `glue_node`
---| 13 # kern `kern_node`
---| 14 # penalty `penalty_node`
---| 15 # unset `unset_node`
---| 16 # style `style_node`
---| 17 # choice `choice_node`
---| 18 # noad `simple_noad`
---| 19 # radical `radical_noad`
---| 20 # fraction `fraction_noad`
---| 21 # accent `accent_noad`
---| 22 # fence `fence_noad`
---| 23 # math_char `math_char_node`
---| 24 # sub_box `sub_box_node`
---| 25 # sub_mlist `sub_mlist_node`
---| 26 # math_text_char `math_text_char_node`
---| 27 # delim `delim_node`
---| 28 # margin_kern `margin_kern_node`
---| 29 # glyph `glyph_node`
---| 30 # align_record `align_record_node`
---| 31 # pseudo_file `pseudo_file_node`
---| 32 # pseudo_line `pseudo_line_node`
---| 33 # page_insert `inserting_node`
---| 34 # split_insert `split_up_node`
---| 35 # expr_stack `expr_node`
---| 36 # nested_list `nesting_node`
---| 37 # span `span_node`
---| 38 # attribute `attribute_node`
---| 39 # glue_spec `glue_spec_node`
---| 40 # attribute_list `attribute_list_node`
---| 41 # temp `temp_node`
---| 42 # align_stack `align_stack_node`
---| 43 # movement_stack `movement_node`
---| 44 # if_stack `if_node`
---| 45 # unhyphenated `unhyphenated_node`
---| 46 # hyphenated `hyphenated_node`
---| 47 # delta `delta_node`
---| 48 # passive `passive_node`
---| 49 # shape `shape_node`

---
---A number in the range `[0,4]` indicating the glue order.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias GlueOrder 0|1|2|3|4

---
---The calculated glue ratio.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias GlueSet number

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias GlueSign
---|0 # `normal`,
---|1 # `stretching`,
---|2 # `shrinking`

---
---A node that comprise actual typesetting commands. A few fields are
---present in all nodes regardless of their type, these are:
---
--- __Reference:__
---
---* Source file of the `LuaTeX` manual:  [luatex-nodes.tex#L49-L76](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L49-L76)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class Node
---@field next Node|nil # the next node in a list, or nil
---@field prev Node|nil # That prev field is always present, but only initialized on explicit request: when the function `node.slide()` is called, it will set up the `prev` fields to be a backwards pointer in the argument node list. By now most of *TeX*'s node processing makes sure that the `prev` nodes are valid but there can be exceptions, especially when the internal magic uses a leading `temp` nodes to temporarily store a state.
---@field id integer # the node’s type (id) number
---@field subtype integer # the node subtype identifier. The `subtype` is sometimes just a dummy entry because not all nodes actually use the `subtype`, but this way you can be sure that all nodes accept it as a valid field name, and that is often handy in node list traversal.
---@field head? Node
---@field attr Node # A list of attributes.. almost all nodes also have an `attr` field

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L633-L661](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L633-L661)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias HlistNodeSubtype
---|0 # unknown
---|1 # line
---|2 # box
---|3 # indent
---|4 # alignment
---|5 # cell
---|6 # equation
---|7 # equationnumber
---|8 # math
---|9 # mathchar
---|10 # hextensible
---|11 # vextensible
---|12 # hdelimiter
---|13 # vdelimiter
---|14 # overdelimiter
---|15 # underdelimiter
---|16 # numerator
---|17 # denominator
---|18 # limits
---|19 # fraction
---|20 # nucleus
---|21 # sup
---|22 # sub
---|23 # degree
---|24 # scripts
---|25 # over
---|26 # under
---|27 # accent
---|28 # radical

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L841-L851](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L841-L851)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class ListNode: Node
---@field width integer # the width of the box
---@field height integer # the height of the box
---@field depth integer # the depth of the box
---@field shift integer # a displacement perpendicular to the character progression direction
---@field glue_order GlueOrder
---@field glue_set GlueSet
---@field glue_sign GlueSign
---@field head Node # the first node of the body of this list
---@field list Node # the first node of the body of this list
---@field dir DirectionSpecifier

---
---A warning: never assign a node list to the `head` field unless you are sure
---its internal link structure is correct, otherwise an error may result.
---
---Note: the field name `head` and `list` are both valid. Sometimes it
---makes more sense to refer to a list by `head`, sometimes `list` makes
---more sense.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L78-L108](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L78-L108)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class HlistNode: ListNode
---@field subtype HlistNodeSubtype

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias VlistNodeSubtype
---|0 # unknown
---|4 # alignment
---|5 # cell

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class VlistNode: ListNode
---@field subtype VlistNodeSubtype

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L602-L611](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L602-L611)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias RuleNodeSubtype
---|0 # normal
---|1 # box
---|2 # image
---|3 # empty
---|4 # user
---|5 # over
---|6 # under
---|7 # fraction
---|8 # radical
---|9 # outline

---
---Contrary to traditional *TeX*, *LuaTeX* has more `rule` subtypes because we
---also use rules to store reuseable objects and images. User nodes are invisible
---and can be intercepted by a callback.
---
---The `left` and `right` keys are somewhat special (and experimental).
---When rules are auto adapting to the surrounding box width you can enforce a shift
---to the right by setting `left`. The value is also subtracted from the width
---which can be a value set by the engine itself and is not entirely under user
---control. The `right` is also subtracted from the width. It all happens in
---the backend so these are not affecting the calculations in the frontend (actually
---the auto settings also happen in the backend). For a vertical rule `left`
---affects the height and `right` affects the depth. There is no matching
---interface at the *TeX* end (although we can have more keywords for rules it would
---complicate matters and introduce a speed penalty.) However, you can just
---construct a rule node with *Lua* and write it to the *TeX* input. The `outline` subtype is just a convenient variant and the `transform` field
---specifies the width of the outline.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L119-L157](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L119-L157)
---* Corresponding C source code: [texnodes.c#L912-L920](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L912-L920)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class RuleNode: Node
---@field subtype RuleNodeSubtype
---@field attr Node # A list of attributes.
---@field width integer # the width of the rule where the special value −1073741824 is used for ‘running’ glue dimensions
---@field height integer # the height of the rule (can be negative)
---@field depth integer # the depth of the rule (can be negative)
---@field left integer # shift at the left end (also subtracted from width)
---@field right integer # (subtracted from width)
---@field dir DirectionSpecifier the direction of this rule
---@field index integer # an optional index that can be referred too
---@field transform integer # an private variable (also used to specify outline width)

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L818-L824](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L818-L824)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class InsNode: Node
---@field subtype number # the insertion class
---@field attr Node # A list of attributes.
---@field cost integer # the penalty associated with this insert
---@field height integer # height of the insert
---@field depth integer # depth of the insert
---@field head Node # the first node of the body of this insert
---@field list Node # the first node of the body of this insert

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L868-L871](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L868-L871)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class MarkNode: Node
---@field subtype integer # unused
---@field attr Node # A list of attributes.
---@field class integer # the mark class
---@field mark table # a table representing a token list

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L669-L670](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L669-L670)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias AdjustNodeSubtype
---|0 # normal
---|1 # pre

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L725-L727](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L725-L727)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class AdjustNode: Node
---@field subtype AdjustNodeSubtype
---@field attr Node # A list of attributes.
---@field head Node # adjusted material
---@field list Node # adjusted material

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L620-L625](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L620-L625)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias DiscNodeSubtype
---|0 # discretionary
---|1 # explicit
---|2 # automatic
---|3 # regular
---|4 # first
---|5 # second

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L758-L763](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L758-L763)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class DiscNode: Node
---@field subtype DiscNodeSubtype
---@field attr Node # A list of attributes.
---@field pre Node # pointer to the pre-break text
---@field post Node # pointer to the post-break text
---@field replace Node # pointer to the no-break text
---@field penalty integer # the penalty associated with the break, normally `hyphenpenalty` or `exhyphenpenalty`

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L663-L664](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L663-L664)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias MathNodeSubtype
---|0 # beginmath
---|1 # endmath

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L873-L880](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L873-L880)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class MathNode: Node
---@field subtype MathNodeSubtype
---@field attr Node # A list of attributes.
---@field surround integer # width of the `mathsurround` kern

---
---Skips are about the only type of data objects in traditional *TeX* that are not a
---simple value. They are inserted when *TeX* sees a space in the text flow but also
---by `hskip` and `vskip`. The structure that represents the glue
---components of a skip is called a `glue_spec`.
---
---The effective width of some glue subtypes depends on the stretch or shrink needed
---to make the encapsulating box fit its dimensions. For instance, in a paragraph
---lines normally have glue representing spaces and these stretch or shrink to make
---the content fit in the available space. The `effective_glue` function that
---takes a glue node and a parent (hlist or vlist) returns the effective width of
---that glue item. When you pass `true` as third argument the value will be
---rounded.
---
---A `glue_spec` node is a special kind of node that is used for storing a set
---of glue values in registers. Originally they were also used to store properties
---of glue nodes (using a system of reference counts) but we now keep these
---properties in the glue nodes themselves, which gives a cleaner interface to *Lua*.
---
---The indirect spec approach was in fact an optimization in the original *TeX*
---code. First of all it can save quite some memory because all these spaces that
---become glue now share the same specification (only the reference count is
---incremented), and zero testing is also a bit faster because only the pointer has
---to be checked (this is no longer true for engines that implement for instance
---protrusion where we really need to ensure that zero is zero when we test for
---bounds). Another side effect is that glue specifications are read-only, so in
---the end copies need to be made when they are used from *Lua* (each assignment to
---a field can result in a new copy). So in the end the advantages of sharing are
---not that high (and nowadays memory is less an issue, also given that a glue node
---is only a few memory words larger than a spec).
---
---In addition there are the `width`, `stretch` `stretch_order`,
---`shrink`, and `shrink_order` fields. Note that we use the key `width` in both horizontal and vertical glue. This suits the *TeX* internals well
---so we decided to stick to that naming.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L794-L799](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L794-L799)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class GlueSpecNode: Node
---@field width integer # The horizontal or vertical displacement.
---@field stretch integer # An extra (positive) displacement or stretch amount.
---@field stretch_order integer # Factor applied to stretch amount.
---@field shrink integer # An extra (negative) displacement or shrink amount.
---@field shrink_order integer # Factor applied to shrink amount.

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias GlueNodeSubtype
---|0 # userskip
---|1 # lineskip
---|2 # baselineskip
---|3 # parskip
---|4 # abovedisplayskip
---|5 # belowdisplayskip
---|6 # abovedisplayshortskip
---|7 # belowdisplayshortskip
---|8 # leftskip
---|9 # rightskip
---|10 # topskip
---|11 # splittopskip
---|12 # tabskip
---|13 # spaceskip
---|14 # xspaceskip
---|15 # parfillskip
---|16 # mathskip
---|17 # thinmuskip
---|18 # medmuskip
---|19 # thickmuskip
---|98 # conditionalmathskip
---|99 # muglue
---|100 # leaders
---|101 # cleaders
---|102 # xleaders
---|103 # gleaders

---
---A regular word space also results in a `spaceskip` subtype (this used to be
---a `userskip` with subtype zero).
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L785-L792](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L785-L792)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class GlueNode: Node
---@field subtype GlueNodeSubtype
---@field leader Node # A pointer to a box or rule for leaders.
---@field width integer # The horizontal or vertical displacement.
---@field stretch integer # An extra (positive) displacement or stretch amount.
---@field stretch_order integer # Factor applied to stretch amount.
---@field shrink integer # An extra (negative) displacement or shrink amount.
---@field shrink_order integer # Factor applied to shrink amount.

---
---Return the
---effective width of a glue node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8473-L8503](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8473-L8503)
---
---@param glue Node
---@param parent HlistNode|VlistNode
---@param round? boolean # When you pass true as third argument the value will be rounded.
---
---@return number|integer|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.effective_glue(glue, parent, round) end

---
---Return the
---effective width of a glue node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8505-L8533](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8505-L8533)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param parent HlistNode|VlistNode
---@param round? boolean # When you pass true as third argument the value will be rounded.
---
---@return number|integer|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.effective_glue(d, parent, round) end

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L597-L600](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L597-L600)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias KernNodeSubtype
---|0 # fontkern
---|1 # userkern
---|2 # accentkern
---|3 # italiccorrection

---
---The `kern` command creates such nodes but for instance the font and math
---machinery can also add them.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L836-L839](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L836-L839)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class KernNode: Node
---@field subtype KernNodeSubtype
---@field kern integer # Fixed horizontal or vertical advance (in scaled points)

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L587-L595](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L587-L595)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias PenaltyNodeSubtype
---|0 # userpenalty
---|1 # linebreakpenalty
---|2 # linepenalty
---|3 # wordpenalty
---|4 # finalpenalty
---|5 # noadpenalty
---|6 # beforedisplaypenalty
---|7 # afterdisplaypenalty
---|8 # equationnumberpenalty

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L898-L900](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L898-L900)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PenaltyNode: Node
---@field subtype PenaltyNodeSubtype
---@field attr Node # A list of attributes.
---@field penalty integer # the penalty value

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L613-L618](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L613-L618)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias GlyphNodeSubtype
---|0 # character
---|1 # ligature
---|2 # ghost
---|3 # left
---|4 # right

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L801-L816](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L801-L816)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class GlyphNode: Node
---@field subtype GlyphNodeSubtype # A bit field
---@field attr Node # A list of attributes
---@field char integer # The character index. in the font
---@field font integer # The font identifier
---@field lang integer # The language identifier
---@field left integer # The frozen `\lefthyphenmnin` value
---@field right integer # The frozen `\righthyphenmnin` value
---@field uchyph boolean # The frozen `uchyph` value
---@field components Node # A Pointer to ligature components
---@field xoffset integer # A virtual displacement in horizontal direction
---@field yoffset integer # A virtual displacement in vertical direction
---@field width integer # The (original) width of the character
---@field height integer # The (original) height of the character
---@field depth integer # The (original) depth of the character
---@field expansion_factor integer # The to be applied expansion factor
---@field data integer # A general purpose field for users (we had room for it)

---
---Return `true` when a glyph or disc node references that font.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L7639-L7655](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L7639-L7655)
---
---@param n Node
---@param font integer
---
---@return boolean
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.uses_font(n, font) end

---
---Return `true` when a glyph or disc node references that font.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L7621-L7637](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L7621-L7637)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param font integer
---
---@return boolean
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.uses_font(d, font) end

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L582-L585](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L582-L585)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias BoundaryNodeSubtype
---|0 # cancel
---|1 # user
---|2 # protrusion
---|3 # word

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L735-L737](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L735-L737)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class BoundaryNode: Node
---@field subtype BoundaryNodeSubtype
---@field attr Node # A list of attributes.
---@field value integer # values 0-255 are reserved

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L853-L861](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L853-L861)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class LocalParNode: Node
---@field attr Node # A list of attributes.
---@field pen_inter integer # local interline penalty (from `localinterlinepenalty`)
---@field pen_broken integer # local broken penalty (from `localbrokenpenalty`)
---@field dir string # the direction of this par. see \in [dirnodes]
---@field box_left Node # the `localleftbox`
---@field box_left_width integer # width of the `localleftbox`
---@field box_right Node # the `localrightbox`
---@field box_right_width integer # width of the `localrightbox`

---
---Direction specifiers are three-letter combinations of `T`, `B`,
---`R`, and `L`. These are built up out of three separate items:
---
---* the first is the direction of the “top” of paragraphs
---* the second is the direction of the “start” of lines
---* the third is the direction of the “top” of glyphs
---
---However, only four combinations are accepted: `TLT`, `TRT`, `RTT`, and `LTL`. Inside actual `dir` nodes, the representation of
---`dir` is not a three-letter but a combination of numbers. When printed the
---direction is indicated by a `+` or `-`, indicating whether the value
---is pushed or popped from the direction stack.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L753-L756](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L753-L756)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class DirNode: Node
---@field attr Node # A list of attributes.
---@field dir string # the direction (but see below)
---@field level integer # nesting level of this direction whatsit

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L666-L667](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L666-L667)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias MarginKernNodeSubtype
---|0 # left
---|1 # right

---
---Margin kerns result from protrusion.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L863-L866](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L863-L866)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class MarginKernNode: Node
---@field subtype MarginKernNodeSubtype
---@field attr Node # A list of attributes.
---@field width integer # the advance of the kern
---@field glyph Node # the glyph to be used

---
---The `math_char` is the simplest subnode field, it contains the character
---and family for a single glyph object.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L882-L885](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L882-L885)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class MathCharNode: Node
---@field attr Node # A list of attributes.
---@field char integer # The character index.
---@field fam integer # The family number.

---
---The `math_text_char` is a special
---case that you will not normally encounter, it arises temporarily during math list
---conversion (its sole function is to suppress a following italic correction).
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L887-L890](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L887-L890)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class MathTextCharNode: Node
---@field attr Node # A list of attributes.
---@field char integer # The character index.
---@field fam integer # The family number.

---
---For subsidiary list items
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L933-L935](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L933-L935)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class SubBoxNode: Node
---@field attr Node # A list of attributes.
---@field head Node # A list of nodes. `head` points to a “normal” vbox or hbox.

---
---For subsidiary list items
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L937-L939](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L937-L939)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class SubMlistNode: Node
---@field attr Node # A list of attributes.
---@field head Node # A list of nodes. `head` points to a math list that is yet to be converted.

---
---There is a fifth subnode type that is used exclusively for delimiter fields. As
---before, the `next` and `prev` fields are unused.
---
---The fields `large_char` and `large_fam` can be zero, in that case the
---font that is set for the `small_fam` is expected to provide the large
---version as an extension to the `small_char`.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L746-L751](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L746-L751)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class DelimNode: Node
---@field attr Node # A list of attributes.
---@field small_char integer # character index of base character
---@field small_fam integer # family number of base character
---@field large_char integer # character index of next larger character
---@field large_fam integer # family number of next larger character

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L672-L684](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L672-L684)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias NoadNodeSubtype
---|0 # ord
---|1 # opdisplaylimits
---|2 # oplimits
---|3 # opnolimits
---|4 # bin
---|5 # rel
---|6 # open
---|7 # close
---|8 # punct
---|9 # inner
---|10 # under
---|11 # over
---|12 # vcenter

---
---First, there are the objects (the *TeX* book calls them “atoms”) that are
---associated with the simple math objects: ord, op, bin, rel, open, close, punct,
---inner, over, under, vcent. These all have the same fields, and they are combined
---into a single node type with separate subtypes for differentiation.
---
---Some noads have an option field. The values in this bitset are common:
---
--- | meaning         |  bits           |
--- |-----------------|-----------------|
--- | set             |          `0x08` |
--- | internal        | `0x00` + `0x08` |
--- | internal        | `0x01` + `0x08` |
--- | axis            | `0x02` + `0x08` |
--- | no axis         | `0x04` + `0x08` |
--- | exact           | `0x10` + `0x08` |
--- | left            | `0x11` + `0x08` |
--- | middle          | `0x12` + `0x08` |
--- | right           | `0x14` + `0x08` |
--- | no sub script   | `0x21` + `0x08` |
--- | no super script | `0x22` + `0x08` |
--- | no script       | `0x23` + `0x08` |
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L892-L896](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L892-L896)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class NoadNode: Node
---@field subtype NoadNodeSubtype
---@field attr Node # A list of attributes.
---@field nucleus Node # base
---@field sub Node # subscript
---@field sup Node # superscript
---@field options integer # bitset of rendering options

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L694-L697](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L694-L697)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias AccentNodeSubtype
---|0 # bothflexible
---|1 # fixedtop
---|2 # fixedbottom
---|3 # fixedboth

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L714-L723](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L714-L723)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class AccentNode: Node
---@field attr Node # A list of attributes.
---@field subtype AccentNodeSubtype
---@field nucleus Node # base
---@field sub Node # subscript
---@field sup Node # superscript
---@field accent Node # top accent
---@field bot_accent Node # bottom accent
---@field top_accent Node # top_accent
---@field overlay_accent Node # overlay_accent
---@field fraction integer # larger step criterium (divided by 1000)

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L929-L931](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L929-L931)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class StyleNode: Node
---@field attr Node # A list of attributes.
---@field style string # contains the style

---
---Warning: never assign a node list to the `display`, `text`, `script`, or `scriptscript` field unless you are sure its internal link
---structure is correct, otherwise an error can occur.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L739-L744](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L739-L744)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class ChoiceNode: Node
---@field attr Node # A list of attributes.
---@field display Node # list of display size alternatives
---@field text Node # list of text size alternatives
---@field script Node # list of scriptsize alternatives
---@field scriptscript Node # list of scriptscriptsize alternatives

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L686-L692](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L686-L692)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias RadicalNodeSubtype
---|0 # radical
---|1 # uradical
---|2 # uroot
---|3 # uunderdelimiter
---|4 # uoverdelimiter
---|5 # udelimiterunder
---|6 # udelimiterover

---
----
--Warning: never assign a node list to the `nucleus`, `sub`, `sup`, `left`, or `degree` field unless you are sure its internal
---link structure is correct, otherwise an error can be triggered.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L902-L910](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L902-L910)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class RadicalNode: Node
---@field subtype RadicalNodeSubtype
---@field attr Node # A list of attributes.
---@field nucleus KernNode # base
---@field sub KernNode # subscript
---@field sup KernNode # superscript
---@field left DelimNode
---@field degree KernNode # only set by `Uroot`
---@field width integer # required width
---@field options integer # bitset of rendering options

---
---Warning: never assign a node list to the `num`, or `denom` field
---unless you are sure its internal link structure is correct, otherwise an error
---can result.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L774-L783](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L774-L783)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FractionNode: Node
---@field attr Node # A list of attributes.
---@field width integer # (optional) width of the fraction
---@field num KernNode # numerator
---@field denom KernNode # denominator
---@field left DelimNode # left side symbol
---@field right DelimNode # right side symbol
---@field middle DelimNode # middle symbol
---@field options integer # bitset of rendering options
---

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L627-L631](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L627-L631)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FenceNodeSubtype
---|0 # unset
---|1 # left
---|2 # middle
---|3 # right
---|4 # no

---
---Warning: some of these fields are used by the renderer and might get adapted in
---the process.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L765-L772](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L765-L772)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FenceNode: Node
---@field subtype FenceNodeSubtype
---@field attr Node # list of attributes
---@field delim DelimNode # delimiter specification
---@field italic integer # italic correction
---@field height integer # required height
---@field depth integer # required depth
---@field options integer # bitset of rendering options
---@field class integer # spacing related class

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias WhatsitTypeId
---|0  # open
---|1  # write
---|2  # close
---|3  # special
---|6  # save_pos
---|7  # late_lua
---|8  # user_defined
---|16 # pdf_literal
---|17 # pdf_refobj
---|18 # pdf_annot
---|19 # pdf_start_link
---|20 # pdf_end_link
---|21 # pdf_dest
---|22 # pdf_action
---|23 # pdf_thread
---|24 # pdf_start_thread
---|25 # pdf_end_thread
---|26 # pdf_thread_data
---|27 # pdf_link_data
---|28 # pdf_colorstack
---|29 # pdf_setmatrix
---|30 # pdf_save
---|31 # pdf_restore
---|32 # pdf_link_state

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias WhatsitTypeName
---| 'open' # 0
---| 'write' # 1
---| 'close' # 2
---| 'special' # 3
---| 'save_pos' # 6
---| 'late_lua' # 7
---| 'user_defined' # 8
---| 'pdf_literal' # 16
---| 'pdf_refobj' # 17
---| 'pdf_annot' # 18
---| 'pdf_start_link' # 19
---| 'pdf_end_link' # 20
---| 'pdf_dest' # 21
---| 'pdf_action' # 22
---| 'pdf_thread' # 23
---| 'pdf_start_thread' # 24
---| 'pdf_end_thread' # 25
---| 'pdf_thread_data' # 26
---| 'pdf_link_data' # 27
---| 'pdf_colorstack' # 28
---| 'pdf_setmatrix' # 29
---| 'pdf_save' # 30
---| 'pdf_restore' # 31
---| 'pdf_link_state' # 32

---
---Whatsit nodes come in many subtypes that you can ask for them by running
---`node.whatsits`.
---
---Some of them are generic and independent of the output mode and others are
---specific to the chosen backend: *DVI* or \PDF. Here we discuss the generic
---font-end nodes nodes.
---
---Source: [luatex-nodes.tex#L781-L797](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L781-L797)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class WhatsitNode: Node

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class OpenWhatsitNode: Node
---@field attr Node # A list of attributes.
---@field stream integer # *TeX*'s stream id number
---@field name string # file name
---@field ext string # file extension
---@field area string # file area (this may become obsolete)

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class WriteWhatsitNode: Node
---@field attr Node # A list of attributes.
---@field stream integer # *TeX*'s stream id number
---@field data table # a table representing the token list to be written

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L1018-L1020](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L1018-L1020)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class CloseWhatsitNode: Node
---@field attr Node # A list of attributes.
---@field stream integer # *TeX*'s stream id number

---
---User-defined whatsit nodes can only be created and handled from *Lua* code. In
---effect, they are an extension to the extension mechanism. The *LuaTeX* engine
---will simply step over such whatsits without ever looking at the contents.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L833-L864](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L833-L864)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class UserDefinedWhatsitNode: WhatsitNode
---@field user_id integer # id number
---@field type 97|100|108|110|115|116 # The `type` can have one of six distinct values. The number is the ASCII value if the first character of the type name (so you can use string.byte("l") instead of `108`): 97 “a” list of attributes (a node list), 100 “d” a *Lua* number, 108 “l” a *Lua* value (table, number, boolean, etc), 110 “n” a node list, 115 “s” a *Lua* string, 116 “t” a *Lua* token list in *Lua* table form (a list of triplets).
---@field value number|Node|string|table

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class SavePosWhatsitNode: WhatsitNode
---@field attr Node # A list of attributes.

---
---The difference between `data` and `string` is that on assignment, the
---`data` field is converted to a token list, cf. use as `latelua`. The
---`string` version is treated as a literal string.
---
---When a function is used, it gets called with as first argument the node that triggers
---the call.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L1022-L1027](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L1022-L1027)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class LateLuaWhatsitNode: WhatsitNode
---@field attr Node # A list of attributes.
---@field data string|function # the to be written information stored as *Lua* value
---@field token string # the to be written information stored as token list
---@field name string # the name to use for *Lua* error reporting

---
---There is only one DVI backend whatsit, and it just flushes its content to the
---output file.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class SpecialWhatsitNode: WhatsitNode
---@field attr Node # A list of attributes.
---@field data string # the `special` information

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L1082-L1088](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L1082-L1088)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias PdfLiteralModes
---|0 `origin`
---|1 `page`
---|2 `direct`
---|3 `raw`
---|4 `text`
---|5 `font`
---|6 `special`

---
---The higher the number, the less checking and the more you can run into trouble.
---Especially the `raw` variant can produce bad *PDF* so you can best check
---what you generate.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L1148-L1151](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L1148-L1151)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfLiteralWhatsitNode: WhatsitNode
---@field attr Node # A list of attributes.
---@field mode PdfLiteralModes # the “mode” setting of this literal
---@field data string # the to be written information stored as *Lua* string
---@field token string # the to be written information stored as token list

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfRefobjWhatsitNode: WhatsitNode
---@field attr Node # A list of attributes.
---@field objnum integer # the referenced *PDF* object number

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfAnnotWhatsitNode: WhatsitNode
---@field attr Node # A list of attributes.
---@field width integer # the width (not used in calculations)
---@field height integer # the height (not used in calculations)
---@field depth integer # the depth (not used in calculations)
---@field objnum integer # the referenced *PDF* object number
---@field data string # the annotation data

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfStartLinkWhatsitNode: WhatsitNode
---@field attr Node # A list of attributes.
---@field width integer # the width (not used in calculations)
---@field height integer # the height (not used in calculations)
---@field depth integer # the depth (not used in calculations)
---@field objnum integer # the referenced *PDF* object number
---@field link_attr table # the link attribute token list
---@field action Node # the action to perform

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfEndLinkWhatsitNode: WhatsitNode
---@field attr Node # A list of attributes.

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfDestWhatsitNode: WhatsitNode
---@field attr Node # A list of attributes.
---@field width integer # the width (not used in calculations)
---@field height integer # the height (not used in calculations)
---@field depth integer # the depth (not used in calculations)
---@field named_id integer # is the `dest_id` a string value?
---@field dest_id integer|string # `integer`: the destination id `string`: the destination name
---@field dest_type integer # type of destination
---@field xyz_zoom integer # the zoom factor (times 1000)
---@field objnum integer # the *PDF* object number; for structure references the *PDF* object number of the linked structure element

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L1090-L1093](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L1090-L1093)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias PdfActionTypes
---|0 'page'
---|1 'goto'
---|2 'thread'
---|3 'user'

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L1095-L1097](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L1095-L1097)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias PdfWindowTypes
---|0 'notset'
---|1 'new'
---|2 'nonew'

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L1104-L1111](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L1104-L1111)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfActionWhatsitNode: WhatsitNode
---@field action_type PdfActionTypes # the kind of action involved
---@field named_id integer # are `dest_id` and `struct_id` string values?
---@field action_id integer|string # token list reference or string
---@field file string # the target filename
---@field new_window PdfWindowTypes # the window state of the target
---@field struct_id nil|integer|string # `nil`: the action does not reference a structure; `integer`: id of the referenced structure; `string`: name of the referenced structure destination
---@field data string # the name of the destination

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L1185-L1192](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L1185-L1192)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfThreadWhatsitNode
---@field attr Node # A list of attributes.
---@field width integer # the width (not used in calculations)
---@field height integer # the height (not used in calculations)
---@field depth integer # the depth (not used in calculations)
---@field named_id integer # is `tread_id` a string value?
---@field tread_id integer # the thread id  string  the thread name
---@field thread_attr integer # extra thread information

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L1176-L1183](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L1176-L1183)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfStartThreadWhatsitNode
---@field attr Node # A list of attributes.
---@field width integer # the width (not used in calculations)
---@field height integer # the height (not used in calculations)
---@field depth integer # the depth (not used in calculations)
---@field named_id integer # is `tread_id` a string value?
---@field tread_id integer # the thread id  string  the thread name
---@field thread_attr integer # extra thread information

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L1145-L1146](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L1145-L1146)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfEndThreadWhatsitNode
---@field attr Node # A list of attributes.

---
---From the pdfTeX manual:
---
---`\pdfcolorstack ⟨stack number⟩ ⟨stack action⟩ ⟨general text⟩`
---
---The command operates on the stack of a given number. If ⟨stack action⟩ is `push` keyword, the
---new value provided as ⟨general text⟩ is inserted into the top of the graphic stack and becomes
---the current stack value. If followed by `pop`, the top value is removed from the stack and the
---new top value becomes the current. `set` keyword replaces the current value with ⟨general text⟩
---without changing the stack size. `current` keyword instructs just to use the current stack value
---without modifying the stack at all.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [pdftex-t.tex#L3954-L3980](https://github.com/tex-mirror/pdftex/blob/6fb2352aa70a23ad3830f1434613170be3f3cd74/doc/manual/pdftex-t.tex#L3954-L3980)
---Source: [luatex-nodes.tex#L1097-L1107](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/e1cb50f34dc1451c9c5319dc953305b52a7a96fd/manual/luatex-nodes.tex#L1097-L1107)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfColorstackWhatsitNode: WhatsitNode
---@field stack integer # The colorstack id number.
---@field command integer # The command to execute. ⟨stack action⟩ → set (0) | push (1) | pop (2) | current (3) [texnodes.c#L3523-L3545](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/6472bd794fea67de09f01e1a89e9b12141be7474/source/texk/web2c/luatexdir/tex/texnodes.c#L3523-L3545)
---@field data string # General text that is placed on top of the stack, for example `1 0 0 rg 1 0 0 RG`. `rg` only colors filled outlines while the stroke color is set with `RG`. From the [PDF Reference, fourth edition](https://opensource.adobe.com/dc-acrobat-sdk-docs/pdfstandards/pdfreference1.5_v6.pdf), 4.5.7 Color Operators Page 251: `gray G`: Set the stroking color space to DeviceGray. `gray` is a number between 0.0 (black) and 1.0 (white). `gray g`: Same as `G`, but for nonstroking operations. `r g b RG`: Set the stroking color space to DeviceRGB. Each operand must be a number between 0.0 (minimum intensity) and 1.0 (maximum intensity). `r g b rg`: same as `RG`, but for nonstroking operations. `c m y k K`: Set the stroking color space to DeviceCMYK. Each operand must be a number between 0.0 (zero concentration) and 1.0 (maximum concentration). `c m y k k`: Same as `K`, but for nonstroking operations.

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfSetmatrixWhatsitNode
---@field attr Node # A list of attributes.
---@field data string # data

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfSaveWhatsitNode
---@field attr Node # A list of attributes.

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PdfRestoreWhatsitNode
---@field attr Node # A list of attributes.

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L941-L952](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L941-L952)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class UnsetNode: Node
---@field attr Node
---@field width integer
---@field depth integer
---@field height integer
---@field dir DirectionSpecifier
---@field shrink integer
---@field glue_order integer
---@field glue_sign integer
---@field stretch integer
---@field span unknown
---@field head Node

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class AlignRecordNode: Node

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PseudoFileNode: Node

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PseudoLineNode: Node

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L826-L834](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L826-L834)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PageInsertNode: Node

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L922-L927](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L922-L927)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class SplitInsertNode: Node
---@field height integer
---@field last_ins_ptr integer
---@field best_ins_ptr integer
---@field broken_ptr integer
---@field broken_ins integer

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L474](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L474)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class ExprStackNode: Node

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L475](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L475)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class NestedListNode: Node

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L476](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L476)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class SpanNode: Node

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L729-L731](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L729-L731)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class AttributeNode: Node
---@field number integer
---@field value unknown

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L733](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L733)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class AttributeListNode: Node

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L480](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L480)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class TempNode: Node

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L481](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L481)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class AlignStackNode: Node

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L482](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L482)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class MovementStackNode: Node

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L483](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L483)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class IfStackNode: Node

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L484](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L484)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class UnhyphenatedNode: Node

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L485](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L485)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class HyphenatedNode: Node

---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L486](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L486)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class DeltaNode: Node

---
---Used in the line breaking algorithm.
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L487](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L487)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class PassiveNode: Node

---
---`\parshape`
---
---__Reference:__
---
---* Corresponding C source code: [texnodes.c#L488](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/texnodes.c#L488)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class ShapeNode: Node

---
---Return a number (the internal index of the node) if the argument is a userdata
---object of type `node` and `false` when no node is passed.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L1199-L1211](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1199-L1211)
---* Corresponding C source code: [lnodelib.c#L8295-L8303](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8295-L8303)
---
---@param item any
---
---@return false|integer t
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.is_node(item) end

---
---Return a number (the internal index of the node) if the argument is a userdata
---object of type `node` and `false` when no node is passed.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8326-L8343](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8326-L8343)
---
---@param item any
---
---@return false|integer t
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.is_node(item) end

---
---Return a table that maps node id numbers to node type strings, providing an
---overview of the possible top-level `id` types.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L1218-L1224](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1218-L1224)
---* Corresponding C source code: [lnodelib.c#L3066-L3069](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3066-L3069)
---
---@return table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.types() end

---
---Provides a table of subtype mappings.
---
---TEX’s ‘whatsits’ all have the same id. The various subtypes are defined by their subtype fields.

---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L1226-L1233](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1226-L1233)
---* Corresponding C source code: [lnodelib.c#L3073-L3076](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3073-L3076)
---
---@return table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.whatsits() end

---
---Convert a single type name to its internal numeric representation.
---
---```lua
---node.id('glyph') -- 29
---```
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L1235-L1244](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1235-L1244)
---* Corresponding C source code: [lnodelib.c#L470-L483](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L470-L483)
---
---@param type NodeTypeName
---
---@return NodeTypeId
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.id(type) end

---
---Convert an internal numeric node type representation to an external
---node type string.
---
---If the argument is a number, then the type function converts an
---internal numeric representation to an external string representation.
---Otherwise, it will return the string `node` if the object
---represents a node, and `nil` otherwise.
---
---```lua
---node.type(29) -- glyph
---node.type(node.new("glyph")) -- node
---node.type('xxx') -- nil
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1993-L2009](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1993-L2009)
---
---@param n NodeTypeId # The numeric node type id.
---
---@return NodeTypeName|'node'|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.type(n) end

---
---Convert a single `whatsit` name to its internal numeric representation (subtype).
---
---```lua
---node.subtype('pdf_literal') -- 16
---node.subtype('xxx') -- nil
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1976-L1989](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1976-L1989)
---
---@param whatsit_type_name WhatsitTypeName
---
---@return WhatsitTypeId whatsit_type_id
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.subtype(whatsit_type_name) end

---
---Return an array of valid field names for a particular type of
---node.
---
---If you want to get the valid fields for a “whatsit”, you have to
---supply the second argument also. In other cases, any given second argument will
---be silently ignored.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3080-L3115](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3080-L3115)
---
---@param id NodeTypeId
---@param subtype? number
---
---@return {[number]: string}
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.fields(id, subtype) end

---
---Return a boolean that is only `true` if `n` is actually a node, and it has the field.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2993-L3000](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2993-L3000)
---
---@param n Node
---@param field string
---
---@return boolean t
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.has_field(n, field) end

---
---Return a boolean that is only `true` if `d` is actually a node, and it has the field.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3041-L3049](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3041-L3049)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param field string
---
---@return boolean t
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.has_field(d, field) end

---
---Create a new node.
---
---All its fields are initialized to
---either zero or `nil` except for `id` and `subtype`. Instead of
---numbers you can also use strings (names). If you create a new `whatsit` node
---the second argument is required. As with all node functions, this function
---creates a node at the *TeX* level.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L1299-L1314](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1299-L1314)
---* Corresponding C source code: [lnodelib.c#L2055-L2060](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2055-L2060)
---
---@param id integer|NodeTypeName
---@param subtype? integer|string
---
---@return Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.new(id, subtype) end

---
---Create a new node.
---
---All its fields are initialized to
---either zero or `nil` except for `id` and `subtype`. Instead of
---numbers you can also use strings (names). If you create a new `whatsit` node
---the second argument is required. As with all node functions, this function
---creates a node at the *TeX* level.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L1299-L1314](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1299-L1314)
---* Corresponding C source code: [lnodelib.c#L2064-L2069](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2064-L2069)
---
---@param id integer|NodeTypeName
---@param subtype? integer|string
---
---@return integer d
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.new(id, subtype) end

---
---Free the *TeX* memory allocated for node `n`.
---
---Be careful: no checks are
---done on whether this node is still pointed to from a register or some `next` field: it is up to you to make sure that the internal data structures
---remain correct.
---
---The `free` function returns the next field of the freed node
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2073-L2090](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2073-L2090)
---
---@param n Node
---
---@return Node next
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.free(n) end

---
---Free the *TeX* memory allocated for the specified node.
---
---Be careful: no checks are
---done on whether this node is still pointed to from a register or some `next` field: it is up to you to make sure that the internal data structures
---remain correct.
---
---The `free` function returns the next field of the freed node
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2094-L2109](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2094-L2109)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer next
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.free(d) end

---
---Free the *TeX* memory allocated for the specified node.
---and return nothing.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2113-L2122](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2113-L2122)
---
---@param n Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.flush_node(n) end

---
---Free the *TeX* memory allocated for the specified node.
---and return nothing.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2126-L2133](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2126-L2133)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.flush_node(d) end

---
---Free the *TeX* memory allocated for a list of nodes.
---
---Be
---careful: no checks are done on whether any of these nodes is still pointed to
---from a register or some `next` field: it is up to you to make sure that the
---internal data structures remain correct.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2137-L2146](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2137-L2146)
---
---@param n Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.flush_list(n) end

---
---Free the *TeX* memory allocated for a list of nodes.
---
---Be
---careful: no checks are done on whether any of these nodes is still pointed to
---from a register or some `next` field: it is up to you to make sure that the
---internal data structures remain correct.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2150-L2157](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2150-L2157)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.flush_list(d) end

---
---Create a deep copy of node `n`, including all nested lists as in the case
---of a `hlist` or `vlist` node. Only the `next` field is not copied.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2476-L2485](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2476-L2485)
---
---@param n Node
---
---@return Node m
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.copy(n) end

---
---Create a deep copy of node `n`, including all nested lists as in the case
---of a `hlist` or `vlist` node. Only the `next` field is not copied.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2489-L2500](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2489-L2500)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer e
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.copy(d) end

---
---Create a deep copy of the node list that starts at node `n`.
---
---If
---`m` is also given, the copy stops just before node `m`.
---
---Note that you cannot copy attribute lists this way. However, there is normally no
---need to copy attribute lists as when you do assignments to the `attr` field
---or make changes to specific attributes, the needed copying and freeing takes
---place automatically.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2440-L2452](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2440-L2452)
---
---@param n Node
---@param m? Node
---
---@return Node m
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.copy_list(n, m) end

---
---Create a deep copy of the node list that starts at node `d`.
---
---If
---`e` is also given, the copy stops just before node `e`.
---
---Note that you cannot copy attribute lists this way. However, there is normally no
---need to copy attribute lists as when you do assignments to the `attr` field
---or make changes to specific attributes, the needed copying and freeing takes
---place automatically.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2456-L2472](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2456-L2472)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param e? integer
---
---@return integer e
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.copy_list(d, e) end

---
---Return the node preceding the given node, or `nil` if
---there is no such node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L379-L388](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L379-L388)
---
---@param n Node
---
---@return Node|nil m
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.prev(n) end

---
---Return the node following the given node, or `nil` if
---there is no such node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L390-L399](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L390-L399)
---
---@param n Node
---
---@return Node|nil m
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.next(n) end

---
---Return the currently active list of attributes, if there is one.
---
---The intended usage of `current_attr` is as follows:
---
---```lua
---local x1 = node.new("glyph")
---x1.attr = node.current_attr()
---local x2 = node.new("glyph")
---x2.attr = node.current_attr()
---```
---
---or:
---
---```lua
---local x1 = node.new("glyph")
---local x2 = node.new("glyph")
---local ca = node.current_attr()
---x1.attr = ca
---x2.attr = ca
---```
---
---The attribute lists are ref counted and the assignment takes care of incrementing
---the refcount. You cannot expect the value `ca` to be valid any more when
---you assign attributes (using `tex.setattribute`) or when control has been
---passed back to *TeX*.
---
---Note: this function is somewhat experimental, and it returns the *actual*
---attribute list, not a copy thereof. Therefore, changing any of the attributes in
---the list will change these values for all nodes that have the current attribute
---list assigned to them.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6511-L6532](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6511-L6532)
---
---@return Node|nil m
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.current_attr() end

---
---Return the currently active list of attributes, if there is one.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6511-L6532](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6511-L6532)
---
---@return integer|nil e
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.current_attr() end

---
---Create a new `hlist` by packaging the list that begins at node `n` into a horizontal box.
---
---With only a single argument, this box is created using
---the natural width of its components. In the three argument form, `info`
---must be either `additional` or `exactly`, and `width` is the
---additional (`\hbox spread`) or exact (`\hbox to`) width to be used.
---The second return value is the badness of the generated box.
---
---Caveat: there can be unexpected side-effects to this function, like updating
---some of the `marks` and `\inserts`. Also note that the content of
---`h` is the original node list `n`: if you call `node.free(h)`
---you will also free the node list itself, unless you explicitly set the `list` field to `nil` beforehand. And in a similar way, calling `node.free(n)` will invalidate `h` as well!
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2576-L2619](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2576-L2619)
---
---@param n Node
---@param width? integer
---@param info? string
---@param dir? string
---
---@return Node n
---@return integer badness
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.hpack(n, width, info, dir) end

---
---Create a new `hlist` by packaging the list that begins at node `d` into a horizontal box.
---
---With only a single argument, this box is created using
---the natural width of its components. In the three argument form, `info`
---must be either `additional` or `exactly`, and `width` is the
---additional (`\hbox spread`) or exact (`\hbox to`) width to be used.
---The second return value is the badness of the generated box.
---
---Caveat: there can be unexpected side-effects to this function, like updating
---some of the `marks` and `\inserts`. Also note that the content of
---`h` is the original node list `n`: if you call `node.free(h)`
---you will also free the node list itself, unless you explicitly set the `list` field to `nil` beforehand. And in a similar way, calling `node.free(n)` will invalidate `h` as well!
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2576-L2619](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2576-L2619)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param width? integer
---@param info? string
---@param dir? string
---
---@return integer d
---@return integer badness
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.hpack(d, width, info, dir) end

---
---Create a new `vlist` by packaging the list that begins at node `head` into a vertical box.
---
---With only a single argument, this box is created using
---the natural height of its components.
---
---Caveat: there can be unexpected side-effects to this function, like updating
---some of the `marks` and `\inserts`. Also note that the content of
---`new_head` is the original node list `head`: if you call `node.free(new_head)`
---you will also free the node list itself, unless you explicitly set the `list` field to `nil` beforehand. And in a similar way, calling `node.free(head)` will invalidate `new_head` as well!
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2673-L2716](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2673-L2716)
---
---@param head Node
---@param height? integer # The additional (`\vbox spread`) or exact (`\vbox to`) height to be used.
---@param info? 'additional'|'exactly' # Must be either `additional` (`\vbox spread`) or `exactly` (`\vbox to`).
---@param dir? DirectionSpecifier|DirectionSpecifierId
---
---@return Node new_head
---@return integer badness # The second return value is the badness of the generated box.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.vpack(head, height, info, dir) end

---
---Create a new `vlist` by packaging the list that begins at node `head` into a vertical box.
---
---With only a single argument, this box is created using
---the natural height of its components.
---
---Caveat: there can be unexpected side-effects to this function, like updating
---some of the `marks` and `\inserts`. Also note that the content of
---`new_head` is the original node list `head`: if you call `node.free(new_head)`
---you will also free the node list itself, unless you explicitly set the `list` field to `nil` beforehand. And in a similar way, calling `node.free(head)` will invalidate `new_head` as well!
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2720-L2763](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2720-L2763)
---
---@param head integer # The index number of the node in the memory table for direct access.
---@param height? integer # The additional (`\vbox spread`) or exact (`\vbox to`) height to be used.
---@param info? 'additional'|'exactly' # Must be either `additional` (`\vbox spread`) or `exactly` (`\vbox to`).
---@param dir? DirectionSpecifier|DirectionSpecifierId
---
---@return integer new_head
---@return integer badness # The second return value is the badness of the generated box.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.vpack(head, height, info, dir) end

---
---Add the interlinespace to a line keeping the baselineskip and lineskip into
---account.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8763-L8801](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8763-L8801)
---
---@param n Node # vlist or hlist
---@param prevdepth integer
---
---@return integer new_prevdepth
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.prepend_prevdepth(n, prevdepth) end

---
---Add the interlinespace to a line keeping the baselineskip and lineskip into
---account.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8803-L8840](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8803-L8840)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param prevdepth integer
---
---@return integer new_prevdepth
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.prepend_prevdepth(d, prevdepth) end

---
---Calculate the natural in-line dimensions of the end of the node list starting
---at node `n`.
---
---The return values are scaled points.
---
---You need to keep in mind that this is one of the few places in *TeX* where floats
---are used, which means that you can get small differences in rounding when you
---compare the width reported by `hpack` with `dimensions`.
---
---__Reference:__
---
---[Source: luatex-nodes.tex#L1490-L1546](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1490-L1546)
---* Corresponding C source code: [lnodelib.c#L2767-L2812](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2767-L2812)
---
---@param n Node
---@param dir? DirectionSpecifier
---
---@return integer width # scaled points
---@return integer height # scaled points
---@return integer depth # scaled points
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.dimensions(n, dir) end

---
---Calculate the natural in-line dimensions of the end of the node list starting
---at node `n`.
---
---The return values are scaled points.
---
---You need to keep in mind that this is one of the few places in *TeX* where floats
---are used, which means that you can get small differences in rounding when you
---compare the width reported by `hpack` with `dimensions`.
---
---__Reference:__
---
---[Source: luatex-nodes.tex#L1490-L1546](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1490-L1546)
---* Corresponding C source code: [lnodelib.c#L2838-L2880](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2838-L2880)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param dir? DirectionSpecifier
---
---@return integer width # scaled points
---@return integer height # scaled points
---@return integer depth # scaled points
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.dimensions(d, dir) end

---
---Calculate the natural in-line dimensions of the node list starting
---at node `n` and terminating just before node `t`.
---
---__Reference:__
---
---[Source: luatex-nodes.tex#L1490-L1546](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1490-L1546)
---* Corresponding C source code: [lnodelib.c#L2767-L2812](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2767-L2812)
---
---@param n Node
---@param t Node # terminating node
---@param dir? DirectionSpecifier
---
---@return integer width # scaled points
---@return integer height # scaled points
---@return integer depth # scaled points
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.dimensions(n, t, dir) end

---
---Calculate the natural in-line dimensions of the node list starting
---at node `n` and terminating just before node `t`.
---
---__Reference:__
---
---[Source: luatex-nodes.tex#L1490-L1546](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1490-L1546)
---* Corresponding C source code: [lnodelib.c#L2838-L2880](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2838-L2880)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param t integer # terminating node
---@param dir? DirectionSpecifier
---
---@return integer width # scaled points
---@return integer height # scaled points
---@return integer depth # scaled points
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.dimensions(d, t, dir) end

---
---Calculates the natural in-line dimensions of the end of the node list starting
---at node `n`.
---
---This is an
---alternative format that starts with glue parameters as the first three arguments.
---
---This calling method takes glue settings into account and is especially useful for
---finding the actual width of a sublist of nodes that are already boxed, for
---example in code like this, which prints the width of the space in between the
---`a` and `b` as it would be if `\box0` was used as-is:
---
---```
---\setbox0 = \hbox to 20pt {a b}
---
---\directlua{print (node.dimensions(
---    tex.box[0].glue_set,
---    tex.box[0].glue_sign,
---    tex.box[0].glue_order,
---    tex.box[0].head.next,
---    node.tail(tex.box[0].head)
---)) }
---```
---
---__Reference:__
---
---[Source: luatex-nodes.tex#L1490-L1546](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1490-L1546)
---* Corresponding C source code: [lnodelib.c#L2838-L2880](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2838-L2880)
---
---@param glue_set integer
---@param glue_sign integer
---@param glue_order integer
---@param n Node
---@param dir? DirectionSpecifier
---
---@return integer width # scaled points
---@return integer height # scaled points
---@return integer depth # scaled points
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.dimensions(glue_set, glue_sign, glue_order, n, dir) end

---
---Calculates the natural in-line dimensions of the end of the node list starting
---at node `n`.
---
---This is an
---alternative format that starts with glue parameters as the first three arguments.
---
---This calling method takes glue settings into account and is especially useful for
---finding the actual width of a sublist of nodes that are already boxed, for
---example in code like this, which prints the width of the space in between the
---`a` and `b` as it would be if `\box0` was used as-is:
---
---```
---\setbox0 = \hbox to 20pt {a b}
---
---\directlua{print (node.dimensions(
---    tex.box[0].glue_set,
---    tex.box[0].glue_sign,
---    tex.box[0].glue_order,
---    tex.box[0].head.next,
---    node.tail(tex.box[0].head)
---)) }
---```
---
---__Reference:__
---
---[Source: luatex-nodes.tex#L1490-L1546](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1490-L1546)
---* Corresponding C source code: [lnodelib.c#L2838-L2880](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2838-L2880)
---
---@param glue_set integer
---@param glue_sign integer
---@param glue_order integer
---@param d integer # The index number of the node in the memory table for direct access.
---@param dir? DirectionSpecifier
---
---@return integer width # scaled points
---@return integer height # scaled points
---@return integer depth # scaled points
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.dimensions(glue_set, glue_sign, glue_order, d, dir) end

---
---Calculate the natural in-line dimensions of the node list starting
---at node `n` and terminating just before node `t`.
---
---This is an
---alternative format that starts with glue parameters as the first three arguments.
---
---__Reference:__
---
---[Source: luatex-nodes.tex#L1490-L1546](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1490-L1546)
---* Corresponding C source code: [lnodelib.c#L2767-L2812](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2767-L2812)
---
---@param glue_set integer
---@param glue_sign integer
---@param glue_order integer
---@param d integer # The index number of the node in the memory table for direct access.
---@param t integer # terminating node
---@param dir? DirectionSpecifier
---
---@return integer width # scaled points
---@return integer height # scaled points
---@return integer depth # scaled points
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.dimensions(glue_set, glue_sign, glue_order, d, t, dir) end

---
---Calculate the natural in-line dimensions of the node list starting
---at node `n` and terminating just before node `t`.
---
---This is an
---alternative format that starts with glue parameters as the first three arguments.
---
---__Reference:__
---
---[Source: luatex-nodes.tex#L1490-L1546](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1490-L1546)
---* Corresponding C source code: [lnodelib.c#L2838-L2880](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2838-L2880)
---
---@param glue_set integer
---@param glue_sign integer
---@param glue_order integer
---@param n Node
---@param t Node # terminating node
---@param dir? DirectionSpecifier
---
---@return integer width # scaled points
---@return integer height # scaled points
---@return integer depth # scaled points
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.dimensions(glue_set, glue_sign, glue_order, n, t, dir) end

---
---Calculate the natural in-line dimensions of the node list `parent` starting
---at node `first` and terminating just before node `last`.
---
---This functions saves a few lookups in comparison to `node.dimensions()` and can be more convenient in some
---cases.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2814-L2834](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2814-L2834)
---
---@param parent Node
---@param first Node
---@param last? Node
---
---@return integer width # scaled points
---@return integer height # scaled points
---@return integer depth # scaled points
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.rangedimensions(parent, first, last) end

---
---Calculate the natural in-line dimensions of the node list `parent` starting
---at node `first` and terminating just before node `last`.
---
---This functions saves a few lookups in comparison to `node.dimensions()` and can be more convenient in some
---cases.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2882-L2902](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2882-L2902)
---
---@param parent integer
---@param first integer
---@param last? integer
---
---@return integer width # scaled points
---@return integer height # scaled points
---@return integer depth # scaled points
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.rangedimensions(parent, first, last) end

---
---Run the internal `mlist` to `hlist` conversion, converting the math list in
---`n` into the horizontal list `h`.
---
---The interface is exactly the same
---as for the callback `mlist_to_hlist`.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2906-L2918](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2906-L2918)
---
---@param n Node
---@param display_type string
---@param penalties boolean
---
---@return Node h
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.mlist_to_hlist(n, display_type, penalties) end

---
---see _N._9_9

---
---Return the last node of the node list that starts at `n`.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3262-L3274](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3262-L3274)
---
---@param n Node
---
---@return Node m
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.tail(n) end

---
---Return the last node of the node list that starts at `d`.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3278-L3289](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3278-L3289)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer e
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.tail(d) end

---
---Return the number of nodes contained in the node list that starts at `n`.
---
---If `m` is also supplied it stops at `m` instead of at the end of the
---list. The node `m` is not counted.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L4374-L4386](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L4374-L4386)
---
---@param n Node
---@param m? Node
---
---@return integer i
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.length(n, m) end

---
---Return the number of nodes contained in the node list that starts at `d`.
---
---If `e` is also supplied it stops at `e` instead of at the end of the
---list. The node `d` is not counted.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L4350-L4360](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L4350-L4360)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param e? Node
---
---@return integer i
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.length(d, e) end

---
---Return the number of nodes contained in the node list that starts at `n`
---that have a matching `id` field.
---
---If `m` is also supplied, counting
---stops at `m` instead of at the end of the list. The node `m` is not
---counted. This function also accept string `id`’s.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L4388-L4401](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L4388-L4401)
---
---@param id integer|string
---@param n Node
---@param m? Node
---
---@return integer i
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.count(id, n, m) end

---
---Return the number of nodes contained in the node list that starts at `d`
---that have a matching `id` field.
---
---If `e` is also supplied, counting
---stops at `e` instead of at the end of the list. The node `d` is not
---counted. This function also accept string `id`’s.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L4362-L4369](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L4362-L4369)
---
---@param id integer|string
---@param d integer # The index number of the node in the memory table for direct access.
---@param e? Node
---
---@return integer i
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.count(id, d, e) end

---
---Signal if the glyph is already turned into a character reference
---or not by examining the subtype.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3004-L3024](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3004-L3024)
---
---@param n Node
---@param font? integer
---
---@return boolean|integer|nil
---@return integer|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.is_char(n, font) end

---
---Signal if the glyph is already turned into a character reference
---or not by examining the subtype.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L7572-L7592](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L7572-L7592)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param font? integer
---
---@return boolean|integer|nil
---@return integer|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.is_char(d, font) end

---
---Signal if the glyph is already turned into a character reference
---or not by examining the subtype.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3026-L3037](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3026-L3037)
---
---@param n Node
---
---@return boolean|integer character
---@return integer font
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.is_glyph(n) end

---
---Signal if the glyph is already turned into a character reference
---or not by examining the subtype.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L7594-L7605](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L7594-L7605)
---
---@param n Node
---
---@return boolean|integer character
---@return integer font
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.is_glyph(n) end

---
---Return a *Lua* iterator that loops over the node list that starts at `n`.
---
---__Example:__
---
---```lua
---for n in node.traverse(head) do
---   ...
---end
---```
---
---It should be clear from the definition of the function `f` that even though
---it is possible to add or remove nodes from the node list while traversing, you
---have to take great care to make sure all the `next` (and `prev`)
---pointers remain valid.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L4156-L4168](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L4156-L4168)
---
---@param n Node
---
---@return fun(): t: Node, id: integer, subtype: integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.traverse(n) end

---
---Return a *Lua* iterator that loops over the node list that starts at `d`.
---
---__Example:__
---
---```lua
---for d in node.traverse(head) do
---   ...
---end
---```
---
---It should be clear from the definition of the function `f` that even though
---it is possible to add or remove nodes from the node list while traversing, you
---have to take great care to make sure all the `next` (and `prev`)
---pointers remain valid.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3937-L3953](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3937-L3953)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return fun(): t: integer, id: integer, subtype: integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.traverse(d) end

---
---Return an iterator that loops over all the nodes in the list that starts at
---`n` that have a matching `id` field.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L4196-L4209](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L4196-L4209)
---
---@param id integer
---@param n Node
---
---@return fun(): t: Node, subtype: integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.traverse_id(id, n) end

---
---Return an iterator that loops over all the nodes in the list that starts at
---`d` that have a matching `id` field.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3980-L3995](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3980-L3995)
---
---@param id integer
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return fun(): t: integer, subtype: integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.traverse_id(id, d) end

---
---Loop over the `glyph` nodes in a list.
---
---Only nodes with a subtype less than 256 are seen.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L4237-L4249](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L4237-L4249)
---
---@param n Node
---
---@return fun(): n: Node, font: integer, char: integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.traverse_char(n) end

---
---Loop over the `glyph` nodes in a list.
---
---Only nodes with a subtype less than 256 are seen.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L4022-L4038](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L4022-L4038)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return fun(): d: integer, font: integer, char: integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.traverse_char(d) end

---
---Loop over a list and return the list and
---filter all glyphs.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L4277-L4289](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L4277-L4289)
---
---@param n Node
---
---@return fun(): n: Node, font: integer, char: integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.traverse_glyph(n) end

---
---Loop over a list and return the list and
---filter all glyphs.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L4065-L4081](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L4065-L4081)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return fun(): d: integer, font: integer, char: integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.traverse_glyph(d) end

---
---Loop over the `hlist` and `vlist` nodes in a list.
---
---The four return values can save some time compared to fetching these fields.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L4318-L4330](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L4318-L4330)
---
---@param n Node
---
---@return fun(): n: Node, id: integer, subtype: integer, list: Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.traverse_list(n) end

---
---Loop over the `hlist` and `vlist` nodes in a list.
---
---The four return values can save some time compared to fetching these fields.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L4318-L4330](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L4318-L4330)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return fun(): d: integer, id: integer, subtype: integer, list: Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.traverse_list(d) end

---
---Return the first `glyph` or `disc` node in the given list.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6368-L6382](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6368-L6382)
---
---@param n Node
---
---@return Node|nil n
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.has_glyph(n) end

---
---Return the first `glyph` or `disc` node in the given list.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6368-L6382](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6368-L6382)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil d
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.has_glyph(d) end

---
---Look for and return the next `math` node following the start node `n`.
---
---If
---the given node is a math end node this helper returns that node, else it follows
---the list and returns the next math endnote. If no such node is found `nil` is
---returned.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3293-L3313](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3293-L3313)
---
---@param n Node
---
---@return Node|nil t
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.end_of_math(n) end

---
---Look for and return the next `math` node following the start node `d`.
---
---If
---the given node is a math end node this helper returns that node, else it follows
---the list and returns the next math endnote. If no such node is found `nil` is
---returned.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3317-L3334](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3317-L3334)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil t
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.end_of_math(d) end

---
---Remove the node `current` from the list following `head`.
---
---It is your responsibility to make sure it is really part of that list.
---The return values are the new `head` and `current` nodes. The
---returned `current` is the node following the `current` in the calling
---argument, and is only passed back as a convenience (or `nil`, if there is
---no such node). The returned `head` is more important, because if the
---function is called with `current` equal to `head`, it will be
---changed.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L1775-L1791](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1775-L1791)
---* Corresponding C source code: [lnodelib.c#L2176-L2215](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2176-L2215)
---
---@param head Node
---@param current Node # A node following the list `head`.
---
---@return Node head # The new `head`
---@return Node|nil current # The node following the `current` in the calling
---argument.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.remove(head, current) end

---
---Remove the node `current` from the list following `head`.
---
---It is your responsibility to make sure it is really part of that list.
---The return values are the new `head` and `current` nodes. The
---returned `current` is the node following the `current` in the calling
---argument, and is only passed back as a convenience (or `nil`, if there is
---no such node). The returned `head` is more important, because if the
---function is called with `current` equal to `head`, it will be
---changed.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L1775-L1791](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1775-L1791)
---* Corresponding C source code: [lnodelib.c#L2219-L2267](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2219-L2267)
---
---@param head integer
---@param current integer # A node following the list `head`.
---
---@return integer|nil head # The new `head`
---@return integer|nil current # The node following the `current` in the calling
---argument.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.remove(head, current) end

---
---Insert the node `new` before `current` into the list
---following `head`.
---
---It is your responsibility to make sure that `current` is really part of that list. The return values are the (potentially
---mutated) `head` and the node `new`, set up to be part of the list
---(with correct `next` field). If `head` is initially `nil`, it
---will become `new`.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L1793-L1807](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1793-L1807)
---* Corresponding C source code: [lnodelib.c#L2271-L2315](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2271-L2315)
---
---@param head Node
---@param current Node
---@param new Node
---
---@return Node head
---@return Node new
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.insert_before(head, current, new) end

---
---Insert the node `new` before `current` into the list
---following `head`.
---
---It is your responsibility to make sure that `current` is really part of that list. The return values are the (potentially
---mutated) `head` and the node `new`, set up to be part of the list
---(with correct `next` field). If `head` is initially `nil`, it
---will become `new`.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L1793-L1807](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1793-L1807)
---* Corresponding C source code: [lnodelib.c#L2319-L2357](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2319-L2357)
---
---@param head integer
---@param current integer
---@param new integer
---
---@return integer head
---@return integer new
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.insert_before(head, current, new) end

---
---Insert the node `new` after `current` into the list
---following `head`.
---
---It is your responsibility to make sure that `current` is really part of that list. The return values are the `head` and
---the node `new`, set up to be part of the list (with correct `next`
---field). If `head` is initially `nil`, it will become `new`.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L1809-L1822](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1809-L1822)
---* Corresponding C source code: [lnodelib.c#L2361-L2395](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2361-L2395)
---
---@param head Node
---@param current Node
---@param new Node
---
---@return Node head
---@return Node new
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.insert_after(head, current, new) end

---
---Insert the node `new` after `current` into the list
---following `head`.
---
---It is your responsibility to make sure that `current` is really part of that list. The return values are the `head` and
---the node `new`, set up to be part of the list (with correct `next`
---field). If `head` is initially `nil`, it will become `new`.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L1809-L1822](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1809-L1822)
---* Corresponding C source code: [lnodelib.c#L2399-L2430](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2399-L2430)
---
---@param head integer
---@param current integer
---@param new integer
---
---@return integer head
---@return integer new
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.insert_after(head, current, new) end

---
---Return the first node in the list starting at `n` that is a glyph node
---with a subtype indicating it is a glyph, or `nil`.
---
---If `m` is given,
---processing stops at (but including) that node, otherwise processing stops at the
---end of the list.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6312-L6337](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6312-L6337)
---
---@param n Node
---@param m? Node
---
---@return Node|nil n
---@return boolean success
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.first_glyph(n, m) end

---
---Return the first node in the list starting at `d` that is a glyph node
---with a subtype indicating it is a glyph, or `nil`.
---
---If `e` is given,
---processing stops at (but including) that node, otherwise processing stops at the
---end of the list.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6341-L6362](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6341-L6362)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param e? integer
---
---@return integer|nil d
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.first_glyph(d, e) end

---
---Apply *TeX*-style ligaturing to the specified nodelist.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L5945-L5984](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L5945-L5984)
---
---@param head Node
---@param tail? Node
---
---@return Node head # the new head
---@return Node tail # the new tail (both `head` and `tail` can change into a new ligature)
---@return boolean success
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.ligaturing(head, tail) end

---
---Apply *TeX*-style ligaturing to the specified nodelist.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L5986-L6017](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L5986-L6017)
---
---@param head integer
---@param tail? integer
---
---@return integer head # the new head
---@return integer tail # the new tail (both `head` and `tail` can change into a new ligature)
---@return boolean success
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.ligaturing(head, tail) end

---
---Apply *TeX*-style kerning to the specified node list.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6021-L6060](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6021-L6060)
---
---@param head Node
---@param tail? Node
---
---@return Node head  # the new head (can be an inserted kern node, because special kernings with word boundaries are possible).
---@return Node tail # the new tail (can be an inserted kern node, because special kernings with word boundaries are possible).
---@return boolean success
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.kerning(head, tail) end

---
---Apply *TeX*-style kerning to the specified node list.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6062-L6097](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6062-L6097)
---
---@param head integer
---@param tail? integer
---
---@return integer head  # the new head (can be an inserted kern node, because special kernings with word boundaries are possible).
---@return integer tail # the new tail (can be an inserted kern node, because special kernings with word boundaries are possible).
---@return boolean success
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.kerning(head, tail) end

---
---Convert from `characters` to `glyphs` during node
---processing by subtracting `256` from all glyph node subtypes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6217-L6223](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6217-L6223)
---
---@param n Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.unprotect_glyph(n) end

---
---Convert from `characters` to `glyphs` during node
---processing by subtracting `256` from the glyph node subtype.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6272-L6278](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6272-L6278)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.unprotect_glyph(d) end

---
---Convert from `characters` to `glyphs` during node
---processing by subtracting `256` from the glyph node subtype.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6243-L6259](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6243-L6259)
---
---@param head Node
---@param tail? Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.unprotect_glyphs(head, tail) end

---
---Convert from `characters` to `glyphs` during node
---processing by subtracting `256` from all glyph node subtypes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6295-L6308](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6295-L6308)
---
---@param head integer
---@param tail? integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.unprotect_glyphs(head, tail) end

---
---Add `256` to the `glyph` node subtype
---except that if the value is `1`, add only `255`.
---
---The special handling of `1` means
---that `characters` will become `glyphs` after subtraction of `256`.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6209-L6215](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6209-L6215)
---
---@param n Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.protect_glyph(n) end

---
---Add `256` to the `glyph` node subtype
---except that if the value is `1`, add only `255`.
---
---The special handling of `1` means
---that `characters` will become `glyphs` after subtraction of `256`.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6264-L6270](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6264-L6270)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.protect_glyph(d) end

---
---Add `256` to all `glyph` node subtypes in the node list starting at `head`,
---except that if the value is `1`, add only `255`.
---
---The special handling of `1` means
---that `characters` will become `glyphs` after subtraction of `256`.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6225-L6241](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6225-L6241)
---
---@param head Node
---@param tail? Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.protect_glyphs(head, tail) end

---
---Add `256` to all `glyph` node subtypes in the node list starting at `head`,
---except that if the value is `1`, add only `255`.
---
---The special handling of `1` means
---that `characters` will become `glyphs` after subtraction of `256`.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6280-L6293](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6280-L6293)
---
---@param head integer
---@param tail? integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.protect_glyphs(head, tail) end

---
---Pop the last node from *TeX*'s “current list”.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2556-L2563](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2556-L2563)
---
---@return Node|nil n
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.last_node() end

---
---Pop the last node from *TeX*'s “current list”.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2567-L2572](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2567-L2572)
---
---@return integer|nil n
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.last_node() end

---
---Append a node list to *TeX*'s “current list”.
---
---The
---node list is not deep-copied! There is no error checking either! You mignt need
---to enforce horizontal mode in order for this to work as expected.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L2518-L2521](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L2518-L2521), [luatex-nodes.tex#L1913-L1923](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1913-L1923)
---* Corresponding C source code: [lnodelib.c#L2505-L2525](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2505-L2525)
---
---@param n Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.write(n) end

---
---Append a node list to *TeX*'s “current list”.
---
---The
---node list is not deep-copied! There is no error checking either! You mignt need
---to enforce horizontal mode in order for this to work as expected.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L2518-L2521](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L2518-L2521), [luatex-nodes.tex#L1913-L1923](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L1913-L1923)
---* Corresponding C source code: [lnodelib.c#L2529-L2552](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2529-L2552)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.write(d) end

---
---Return `true` if, for the purpose of line boundary discovery when
---character protrusion is active, this node can be skipped.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6488-L6493](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6488-L6493)
---
---@param n Node
---
---@return boolean skippable
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.protrusion_skippable(n) end

---
---Return `true` if, for the purpose of line boundary discovery when
---character protrusion is active, this node can be skipped.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6497-L6506](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6497-L6506)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return boolean skippable
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.protrusion_skippable(d) end

---
---Set the five properties of a `glue` node in one go.
---
---Non-numeric values are
---equivalent to zero and reset a property.
---
---When you pass values, only arguments that are numbers are assigned so
---
---```
---node.setglue(n,655360,false,65536)
---```
---
---will only adapt the width and shrink.
---
---When a list node is passed, you set the glue, order and sign instead.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3862-L3883](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3862-L3883)
---
---@param n Node
---@param width integer|any
---@param stretch integer|any
---@param shrink integer|any
---@param stretch_order integer|any
---@param shrink_order integer|any
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.setglue(n, width, stretch, shrink, stretch_order, shrink_order) end

---
---Set the five properties of a `glue` node in one go.
---
---Non-numeric values are
---equivalent to zero and reset a property.
---
---When you pass values, only arguments that are numbers are assigned so
---
---```
---node.direct.setglue(n,655360,false,65536)
---```
---
---will only adapt the width and shrink.
---
---When a list node is passed, you set the glue, order and sign instead.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3798-L3818](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3798-L3818)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param width integer|any
---@param stretch integer|any
---@param shrink integer|any
---@param stretch_order integer|any
---@param shrink_order integer|any
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setglue(d, width, stretch, shrink, stretch_order, shrink_order) end

---
---Return 5 values or nothing when no glue is passed.
---
---When the second argument is false, only the width is returned (this is consistent
---with `tex.get`).
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3840-L3860](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3840-L3860)
---
---@param n Node
---
---@return integer|number|nil width_or_glue_set # When a list node (`vlist` or `hlist`) is passed, than the glue set is returned.
---@return integer|nil stretch_or_glue_order # When a list node (`vlist` or `hlist`) is passed, than the glue order is returned.
---@return integer|nil shrink_or_glue_sign # When a list node (`vlist` or `hlist`) is passed, than the glue sign is returned.
---@return integer|nil stretch_order
---@return integer|nil shrink_order
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getglue(n) end

---
---Return 5 values or nothing when no glue is passed.
---
---When the second argument is false, only the width is returned (this is consistent
---with `tex.get`).
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3776-L3796](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3776-L3796)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|number|nil width_or_glue_set # When a list node (`vlist` or `hlist`) is passed, than the glue set is returned.
---@return integer|nil stretch_or_glue_order # When a list node (`vlist` or `hlist`) is passed, than the glue order is returned.
---@return integer|nil shrink_or_glue_sign # When a list node (`vlist` or `hlist`) is passed, than the glue sign is returned.
---@return integer|nil stretch_order
---@return integer|nil shrink_order
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getglue(d) end

---
---Return `true` when the width, stretch and shrink properties
---are zero.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3885-L3899](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3885-L3899)
---
---@param n Node
---
---@return boolean isglue
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.is_zero_glue(n) end

---
---Return `true` when the width, stretch and shrink properties
---are zero.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3820-L3834](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3820-L3834)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return boolean isglue
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.is_zero_glue(d) end

---
---Test if a node has the attribute with number `id` set.
---
---If `value` is
---also supplied, also tests if the value matches `value`. It returns the value,
---or, if no match is found, `nil`.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3339-L3353](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3339-L3353)
---
---@param n Node
---@param id integer
---@param value? integer
---
---@return integer|nil value
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.has_attribute(n, id, value) end

---
---Test if a node has the attribute with number `id` set.
---
---If `value` is
---also supplied, also tests if the value matches `value`. It returns the value,
---or, if no match is found, `nil`.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3357-L3371](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3357-L3371)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param id integer
---@param value? integer
---
---@return integer|nil value
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.has_attribute(d, id, value) end

---
---Test if a node has an attribute with number `id` set.
---
---It returns the
---value, or, if no match is found, `nil`. If no `id` is given then the
---zero attributes is assumed.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3375-L3406](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3375-L3406)
---
---@param n Node
---@param id integer
---
---@return integer|nil value
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.get_attribute(n, id) end

---
---Test if a node has an attribute with number `id` set.
---
---It returns the
---value, or, if no match is found, `nil`. If no `id` is given then the
---zero attributes is assumed.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3450-L3481](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3450-L3481)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param id integer
---
---@return integer|nil value
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.get_attribute(d, id) end

---
---Find the first node that has attribute with number `id` set.
---
---It returns
---the value and the node if there is a match and otherwise nothing.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3408-L3443](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3408-L3443)
---
---@param n Node
---@param id integer
---
---@return integer|nil value
---@return Node|nil n
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.find_attribute(n, id) end

---
---Find the first node that has attribute with number `id` set.
---
---It returns
---the value and the node if there is a match and otherwise nothing.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3503-L3538](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3503-L3538)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param id integer
---
---@return integer|nil value
---@return Node|nil n
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.find_attribute(d, id) end

---
---Set the attribute with number `id` to the value `value`.
---
---Duplicate
---assignments are ignored.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3563-L3578](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3563-L3578)
---
---@param n Node
---@param id integer
---@param value? integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.set_attribute(n, id, value) end

---
---Set the attribute with number `id` to the value `value`.
---
---Duplicate
---assignments are ignored.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3483-L3501](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3483-L3501)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param id integer
---@param value? integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.set_attribute(d, id, value) end

---
---Unset the attribute with the number `id`.
---
---If `value` is also supplied, it
---will only perform this operation if the value matches `value`. Missing
---attributes or attribute-value pairs are ignored.
---
---If the attribute was actually deleted, returns its old value. Otherwise, returns
---`nil`.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3580-L3596](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3580-L3596)
---
---@param n Node
---@param id integer
---@param value? integer
---
---@return integer|nil value
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.unset_attribute(n, id, value) end

---
---Unset the attribute with the number `id`.
---
---If `value` is also supplied, it
---will only perform this operation if the value matches `value`. Missing
---attributes or attribute-value pairs are ignored.
---
---If the attribute was actually deleted, returns its old value. Otherwise, returns
---`nil`.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3540-L3558](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3540-L3558)
---
---@param n Node
---@param id integer
---@param value? integer
---
---@return integer|nil value
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.unset_attribute(n, id, value) end

---
---Return the found
---tail node and make sure that the node lists is double linked.
---
---After some callbacks automatic sliding takes place. This feature can be turned
---off with `node.fix_node_lists(false)` but you better make sure then that
---you don't mess up lists. In most cases *TeX* itself only uses `next`
---pointers but your other callbacks might expect proper `prev` pointers too.
---Future versions of *LuaTeX* can add more checking but this will not influence
---usage.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3226-L3241](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3226-L3241)
---
---@param n Node
---
---@return Node tail
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.slide(n) end

---
---Return the found
---tail node and make sure that the node lists is double linked.
---
---After some callbacks automatic sliding takes place. This feature can be turned
---off with `node.fix_node_lists(false)` but you better make sure then that
---you don't mess up lists. In most cases *TeX* itself only uses `next`
---pointers but your other callbacks might expect proper `prev` pointers too.
---Future versions of *LuaTeX* can add more checking but this will not influence
---usage.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3245-L3258](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3245-L3258)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer tail
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.slide(d) end

---
---Check all `disc` nodes in the node list.
---
---When you fool around with disc nodes you need to be aware of the fact that they
---have a special internal data structure. As long as you reassign the fields when
---you have extended the lists it’s ok because then the tail pointers get updated,
---but when you add to list without reassigning you might end up in trouble when
---the linebreak routine kicks in.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8615-L8627](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8615-L8627)
---
---@param head Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.check_discretionaries(head) end

---
---Check all `disc` nodes in the node list.
---
---When you fool around with disc nodes you need to be aware of the fact that they
---have a special internal data structure. As long as you reassign the fields when
---you have extended the lists it’s ok because then the tail pointers get updated,
---but when you add to list without reassigning you might end up in trouble when
---the linebreak routine kicks in.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8550-L8562](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8550-L8562)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.check_discretionaries(d) end

---
---Check one `disc` node only and also check if the node is a `disc` node.
---
---When you fool around with disc nodes you need to be aware of the fact that they
---have a special internal data structure. As long as you reassign the fields when
---you have extended the lists it’s ok because then the tail pointers get updated,
---but when you add to list without reassigning you might end up in trouble when
---the linebreak routine kicks in.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8629-L8638](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8629-L8638)
---
---@param n Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.check_discretionary(n) end

---
---Check one `disc` node only and also check if the node is a `disc` node.
---
---When you fool around with disc nodes you need to be aware of the fact that they
---have a special internal data structure. As long as you reassign the fields when
---you have extended the lists it’s ok because then the tail pointers get updated,
---but when you add to list without reassigning you might end up in trouble when
---the linebreak routine kicks in.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8564-L8573](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8564-L8573)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.check_discretionary(d) end

---
---Remove the discretionaries in the list and inject the replace
---field when set.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8640-L8679](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8640-L8679)
---
---@param n Node
---
---@return Node head
---@return integer count
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.flatten_discretionaries(n) end

---
---Remove the discretionaries in the list and inject the replace
---field when set.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8575-L8613](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8575-L8613)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return Node head
---@return integer count
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.flatten_discretionaries(d) end

---
---Return the font
---currently associated with the node.
---
---You can normally also access the font with the
---normal font field or getter because it will resolve the family automatically for
---noads.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L2922-L2932](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L2922-L2932)
---
---@param fam integer # family identifier
---
---@return integer id
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.family_font(fam) end

---
---Convert a userdata node into its numeric reference in the memory table.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6552-L6565](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6552-L6565)
---
---@param n Node
---
---@return integer d
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.todirect(n) end

---
---Convert numeric reference in the memory table of a node into a userdata node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6570-L6581](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6570-L6581)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return Node n
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.tonode(d) end

---
---Set the next node of the current node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1767-L1778](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1767-L1778)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param next integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setnext(d, next) end

---
---Return the next node of the current node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1782-L1800](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1782-L1800)
---
---@param n Node
---
---@return Node|nil next
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getnext(n) end

---
---Return the next node of the current node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1756-L1765](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1756-L1765)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil next
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getnext(d) end

---
---Set the previous node of the current node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1816-L1827](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1816-L1827)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param prev integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setprev(d, prev) end

---
---Return the previous node of the current node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1831-L1846](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1831-L1846)
---
---@param n Node
---
---@return Node|nil prev
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getprev(n) end

---
---Return the previous node of a node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1805-L1814](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1805-L1814)
---
---@param n Node
---
---@return Node|nil prev
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getprev(n) end

---
---Set the previous and next pointers at once.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1864-L1880](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1864-L1880)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param prev integer
---@param next integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setboth(d, prev, next) end

---
---Return the previous and next pointer of a node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1884-L1902](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1884-L1902)
---
---@param n Node
---
---@return Node|nil prev
---@return Node|nil next
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getboth(n) end

---
---Return the previous and next pointer of a node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1851-L1862](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1851-L1862)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil next
---@return integer|nil prev
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getboth(d) end

---
---Return the `id` (type) of a node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L487-L496](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L487-L496)
---
---@param n Node
---
---@return integer id
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getid(n) end

---
---Return the `id` (type) of a node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L500-L517](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L500-L517)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer id
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getid(d) end

---
---Set the `subtype` of a node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L533-L540](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L533-L540)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param subtype integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setsubtype(d, subtype) end

---
---Return the `subtype` of a node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L544-L558](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L544-L558)
---
---@param n Node
---
---@return integer subtype
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getsubtype(n) end

---
---Return the `subtype` of a node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L522-L531](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L522-L531)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer subtype
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getsubtype(d) end

---
---Set the font identifier on a `glyph` node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L621-L632](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L621-L632)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@param font integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setfont(d, font) end

---
---Return the font identifier of a `glyph`, `math_char`, `math_text_char` or `delim` node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L636-L654](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L636-L654)
---
---@param n Node
---
---@return integer|nil font
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getfont(n) end

---
---Return the font identifier of a `glyph`, `math_char`, `math_text_char` or `delim` node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L601-L619](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L601-L619)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil font
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getfont(d) end

---
---Set the character index (`char`) in the font on `glyph`, `math_char`, `math_text_char` or `delim` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L702-L717](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L702-L717)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param char integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setchar(d, char) end

---
---Return the character index (`char`) in the font of `glyph`, `math_char`, `math_text_char` or `delim` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L739-L758](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L739-L758)
---
---@param n Node
---
---@return integer|nil char
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getchar(n) end

---
---Return the character index (`char`) in the font of `glyph`, `math_char`, `math_text_char` or `delim` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L659-L678](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L659-L678)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil char
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getchar(d) end

---
---Set the `width`, `height` and `depth` fields of `hlist`, `vlist`, `rule` or `unset` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1307-L1346](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1307-L1346)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param width number # Rounded to an integer
---@param height number # Rounded to an integer
---@param depth number # Rounded to an integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setwhd(d, width, height, depth) end

---
---Return the `width`, `height` and `depth` of a list, rule or (unexpanded) `glyph` as well as `glue` (its spec is looked at) and `unset` node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1350-L1378](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1350-L1378)
---
---@param n Node
---@param get_ex boolean
---
---@return integer width
---@return integer height
---@return integer depth
---@return integer|nil ex # If the node is a `glyph` and `get_ex` is true
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getwhd(n, get_ex) end

---
---Return the `width`, `height` and `depth` of a list, rule or (unexpanded) `glyph` as well as `glue` (its spec is looked at) and `unset` node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1277-L1305](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1277-L1305)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param get_ex boolean
---
---@return integer width
---@return integer height
---@return integer depth
---@return integer|nil ex # If the node is a `glyph` and `get_ex` is true
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getwhd(d, get_ex) end

---
---Set the `pre`, `post`, `replace`, `subtype` and `penalty` on a `disc` node.
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1201-L1232](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1201-L1232)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param pre integer
---@param post? integer
---@param replace? integer
---@param subtype? integer
---@param penalty? integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setdisc(d, pre, post, replace, subtype, penalty) end

---
---Return the `pre`, `post` and `replace` fields and optionally when true is passed also the tail fields.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1236-L1253](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1236-L1253)
---
---@param n Node
---@param get_tail? boolean
---
---@return Node pre
---@return Node post
---@return Node replace
---@return Node|nil pre_tail
---@return Node|nil post_tail
---@return Node|nil replace_tail
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getdisc(n, get_tail) end

---
---Return the `pre`, `post` and `replace` fields and optionally when true is passed also the tail fields.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1182-L1199](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1182-L1199)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param get_tail? boolean
---
---@return integer pre
---@return integer post
---@return integer replace
---@return integer|nil pre_tail
---@return integer|nil post_tail
---@return integer|nil replace_tail
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getdisc(d, get_tail) end

---
---Set child node lists to parent `hlist`, `vlist`, `sub_box`, `sub_mlist`, `ins`, or `adjust` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1404-L1436](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1404-L1436)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param list integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setlist(d, list) end

---
---Get child node lists of parent `hlist`, `vlist`, `sub_box`, `sub_mlist`, `ins`, or `adjust` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1440-L1458](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1440-L1458)
---
---@param n Node
---
---@return Node|nil list
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getlist(n) end

---
---Get child node lists of parent `hlist`, `vlist`, `sub_box`, `sub_mlist`, `ins`, or `adjust` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1382-L1402](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1382-L1402)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil list
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getlist(d) end

---
---Set the leaders to `glue` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1474-L1485](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1474-L1485)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param leader integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setleader(d, leader) end

---
---Get the leaders of `glue` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1489-L1501](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1489-L1501)
---
---@param n Node
---
---@return Node|nil leaders
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getleader(n) end

---
---Get the leaders of `glue` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1463-L1472](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1463-L1472)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil leaders
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getleader(d) end

---
---Set the value of a generic node field.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L7348-L7364](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L7348-L7364)
---
---@param n Node
---@param field string
---@param value any
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.setfield(n, field, value) end

---
---Set the value of a generic node field.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L7660-L8188](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L7660-L8188)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param field string
---@param value any
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setfield(d, field, value) end

---
---Get the value of a generic node field.
---
---Other field names are often shared so a specific getter makes no sense.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L5189-L5207](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L5189-L5207)
---
---@param n Node
---@param field string
---
---@return any|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getfield(n, field) end

---
---Get the value of a generic node field.
---
---Other field names are often shared so a specific getter makes no sense.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L5402-L5891](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L5402-L5891)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param field string
---
---@return any|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getfield(d, field) end

---
---Set the SyncTeX fields, a file number aka tag and a line
---number, for a glue, kern, hlist, vlist, rule and math nodes as well as glyph
---nodes (although this last one is not used in native SyncTeX).
---
---Of course you need to know what you're doing as no checking on sane values takes
---place. Also, the SyncTeX interpreter used in editors is rather peculiar and has
---some assumptions (heuristics).
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8683-L8719](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8683-L8719)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param tag integer
---@param line integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.set_synctex_fields(d, tag, line) end

---
---Query the SyncTeX fields, a file number aka tag and a line
---number, for a glue, kern, hlist, vlist, rule and math nodes as well as glyph
---nodes (although this last one is not used in native SyncTeX).
---
---Of course you need to know what you're doing as no checking on sane values takes
---place. Also, the SyncTeX interpreter used in editors is rather peculiar and has
---some assumptions (heuristics).
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8721-L8759](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8721-L8759)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer tag
---@return integer line
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.get_synctex_fields(d) end

---
---Set the attribute list on a node.
---
---When the second argument of setattributelist is
---`true` the current attribute list is assumed.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L828-L854](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L828-L854)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param attr_list integer|true
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setattributelist(d, attr_list) end

---
---Return the attribute list on a node.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L817-L826](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L817-L826)
---
---@param n Node
---
---@return integer|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getattributelist(n) end

---
---Return the node list of the given box.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8224-L8236](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8224-L8236)
---
---@param box integer
---
---@return integer|nil node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getbox(box) end

---
---Set the field `components` (ligature components) on `glyph` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L774-L785](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L774-L785)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param components integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setcomponents(d, components) end

---
---Return the field `components` (ligature components) of `glyph` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L763-L772](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L763-L772)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getcomponents(d) end

---
---Set the `data` field on `glyph`, `boundary`, `user_defined`, `pdf_literal`, `pdf_late_literal`, `late_lua`, `pdf_setmatrix`, `special`, `late_special` and `write` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1716-L1751](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1716-L1751)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setdata(d) end

---
---Return the `data` field off `glyph`, `boundary`, `user_defined`, `pdf_literal`, `pdf_late_literal`, `late_lua`, `pdf_setmatrix`, `special`, `late_special` and `write` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1678-L1714](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1678-L1714)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return any|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getdata(d) end

---
---Set the direction of `dir`, `hlist`, `vlist`, `rule` and `local_par` nodes as a string.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1093-L1109](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1093-L1109)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param dir DirectionSpecifier
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setdir(d, dir) end

---
---Get the direction  of `dir`, `hlist`, `vlist`, `rule` and `local_par` nodes as a string.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1047-L1067](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1047-L1067)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return DirectionSpecifier
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getdir(d) end

---
---Set the direction of `dir`, `hlist`, `vlist`, `rule` and `local_par` nodes as an integer.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1111-L1134](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1111-L1134)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param dir integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setdirection(d, dir) end

---
---Get the direction of `dir`, `hlist`, `vlist`, `rule` and `local_par` nodes as an integer.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#LL1070-L1092](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#LL1070-L1092)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getdirection(d) end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L580-L596](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L580-L596)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setexpansion(d) end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L563-L578](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L563-L578)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getexpansion(d) end

---
---Set the the family number of `math_char`, `math_text_char`, `delim`, `fraction_noad`, `simple_noad` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L719-L735](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L719-L735)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param fam integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setfam(d, fam) end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L680-L700](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L680-L700)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getfam(d) end

---
---Set the width on `hlist`, `vlist`, `rule`, `glue`, `glue_spec`, `math`, `kern`, `margin_kern`, `ins`, `unset`, `fraction_noad` or `radical_noad` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3641-L3657](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3641-L3657)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param width number # Rounded to an integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setwidth(d, width) end

---
---Get the width of `hlist`, `vlist`, `rule`, `glyph`, `glue`, `glue_spec`, `math`, `kern`, `margin_kern`, `ins` and `unset` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3607-L3639](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3607-L3639)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer width
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getwidth(d) end

---
---Set the height on `hlist`, `vlist`, `rule`, `unset` and `fence_noad` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3681-L3697](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3681-L3697)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setheight(d) end

---
---Return the height off `hlist`, `vlist`, `rule`,  `glyph`, `unset`, `ins` and `fence` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3659-L3679](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3659-L3679)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getheight(d) end

---
---Set the depth on `hlist`, `vlist`, `rule`, `unset` and `fence` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3721-L3737](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3721-L3737)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param depth number
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setdepth(d, depth) end

---
---Return the depth of `hlist`, `vlist`, `rule`, `glyph`, `unset`, `ins` and `fence` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3699-L3719](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3699-L3719)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getdepth(d) end

---
---Set the kerning on `kern` (kern), `margin_kern` (width) and `math` (surround) nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1019-L1042](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1019-L1042)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param kern integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setkern(d, kern) end

---
---Return the kerning of `kern` (kern), `margin_kern` (width) and `math` (surround) nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L995-L1017](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L995-L1017)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@return integer|nil kern
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getkern(d) end

---
---Set the `lang` (the language identifier) field on `glyph` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L801-L812](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L801-L812)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param lang integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setlang(d, lang) end

---
---Return the `lang` (the language identifier) field of `glyph` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L790-L799](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L790-L799)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil lang
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getlang(d) end

---
---Set the `nucleus` (base) field an `simple_noad`, `accent_noad` and `radical_noad` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L916-L930](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L916-L930)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param nucleus integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setnucleus(d, nucleus) end

---
---Return the `nucleus` (base) field of `simple_noad`, `accent_noad` and `radical_noad` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L902-L914](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L902-L914)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil nucleus
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getnucleus(d) end

---
---Set offsets on `glyph` (`xoffset`: a virtual displacement in horizontal direction and
---`yoffset`: a virtual displacement in vertical direction) and `rule` (`left`: shift at the left end (also subtracted from width) and `right`: subtracted from width) nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1156-L1177](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1156-L1177)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param a integer # `xoffset` or `left`
---@param b integer # `yoffset` or `right`
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setoffsets(d, a, b) end

---
---Set the `penalty` field on `glyph` and `rule` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1139-L1154](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1139-L1154)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil a # `xoffset` or `left`
---@return integer|nil b # `yoffset` or `right`
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getoffsets(d) end

---
---Set the `penalty` field on `penalty` and `disc` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L876-L896](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L876-L896)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setpenalty(d) end

---
---Return the `penalty` field of `penalty` and `disc` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L858-L874](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L858-L874)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil penalty
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getpenalty(d) end

---
---Set the `shift` (a displacement perpendicular to the character progression direction) field on `vlist` and `hlist` nodes.
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3756-L3770](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3756-L3770)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param shift integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setshift(d, shift) end

---
---Return the `shift` (a displacement perpendicular to the character progression direction) field off `vlist` and `hlist` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3742-L3754](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3742-L3754)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil shift
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getshift(d) end

---
---Set the `sub` (subscript) field on `simple_noad`, `accent_noad` and `radical_noad` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L946-L960](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L946-L960)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param sub integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setsub(d, sub) end

---
---Return the `sub` (subscript) field of `simple_noad`, `accent_noad` and `radical_noad` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L932-L944](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L932-L944)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil sub
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getsub(d) end

---
---Set the `sup` (superscript) field on `simple_noad`, `accent_noad` and `radical_noad` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L976-L990](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L976-L990)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param sup number
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setsup(d, sup) end

---
---Return the `sup` (superscript) field of `simple_noad`, `accent_noad` and `radical_noad` nodes.
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L962-L974](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L962-L974)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|nil sup
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getsup(d) end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8307-L8322](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8307-L8322)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return integer|false
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.is_direct(d) end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1912-L1957](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1912-L1957)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setlink(d) end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L1959-L1972](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L1959-L1972)
---
---@param l integer # The index number of the node in the memory table for direct access.
---@param r integer # The index number of the node in the memory table for direct access.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setsplit(l, r) end

---
---Assign values to the properties table.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [lnodelib.c#L8397-L8410](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8397-L8410)
---* Corresponding C source code: [lnodelib.c#L8403-L8416](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8403-L8416)
---
---@param n Node
---@param value any
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.setproperty(n, value) end

---
---Assign values to the properties table.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [lnodelib.c#L8397-L8410](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8397-L8410)
---* Corresponding C source code: [lnodelib.c#L8418-L8431](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8418-L8431)
---
---@param d integer # The index number of the node in the memory table for direct access.---@param value any
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.setproperty(d, value) end

---
---Each node also can have a properties table and you can get properties using the `getproperty` function.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L2520-L2523](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L2520-L2523)
---* Corresponding C source code: [lnodelib.c#L8379-L8389](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8379-L8389)
---
---@param node Node
---
---@return any value
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.getproperty(node) end

---
---Each node also can have a properties table and you can get properties using the `getproperty` function.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-nodes.tex#L2520-L2523](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-nodes.tex#L2520-L2523)
---* Corresponding C source code: [lnodelib.c#L8391-L8401](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8391-L8401)
---
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return any value
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.getproperty(d) end

---
---Managing properties in the node (de)allocator functions is disabled by default and is enabled by:
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8351-L8360](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8351-L8360)
---
---@param enable boolean
---@param use_metatable? boolean
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.set_properties_mode(enable, use_metatable) end

---
---Managing properties in the node (de)allocator functions is disabled by default and is enabled by:
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8351-L8360](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8351-L8360)
---
---@param enable boolean
---@param use_metatable? boolean
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.set_properties_mode(enable, use_metatable) end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8364-L8375](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8364-L8375)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.flush_properties_table() end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8364-L8375](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8364-L8375)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.flush_properties_table() end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8439-L8443](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8439-L8443)
---
---@return table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.get_properties_table() end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8433-L8437](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8433-L8437)
---
---@return table
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.get_properties_table() end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L366-L374](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L366-L374)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.fix_node_lists() end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6104-L6122](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6104-L6122)
---
---@param n Node
---@param m? Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.hyphenating(n, m) end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6124-L6142](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6124-L6142)
---
---@param d integer # The index number of the node in the memory table for direct access.
---@param e? integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.hyphenating(d, e) end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L8842-L8868](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L8842-L8868)
---
---@param fnt integer
---@param chr integer
---@param size integer
---@param overlap? integer
---@param horizontal? boolean
---@param attlist? Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.make_extensible(fnt, chr, size, overlap, horizontal, attlist) end

---
---__Example:__
---
---```lua
---node.subtypes('fence') -- {"left", "middle", "right", "no", [0] = "unset"}
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3153-L3222](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3153-L3222)
---
---@param subtype string|integer
---
---@return table<integer, string>
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.subtypes(subtype) end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L5913-L5918](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L5913-L5918)
---
---@param n Node
---@return string # For example `<node    nil <    234 >    nil : glyph 0>`
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.tostring(n) end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L5922-L5931](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L5922-L5931)
---@param d integer # The index number of the node in the memory table for direct access.
---
---@return string # For example `<direct    nil <    234 >    nil : glyph 0>`
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.tostring(d) end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6471-L6476](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6471-L6476)
---
---@return Node n
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.usedlist() end

---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L6480-L6484](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L6480-L6484)
---
---@return integer d
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.direct.usedlist() end

---
---Report some used values.
---
---Valid arguments are `dir`, `direction`, `glue`, `pdf_literal`, `pdf_action`, `pdf_window` and `color_stack`. Keep
---in mind that the setters normally expect a number, but this helper gives you a
---list of what numbers matter. For practical reason the `pagestate` values
---are also reported with this helper.
---
---__Example:__
---
---```lua
---node.values('dir') -- { "TLT", "TRT", "LTL", "RTT" },
---```
---
---__Reference:__
---
---* Corresponding C source code: [lnodelib.c#L3117-L3151](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lnodelib.c#L3117-L3151)
---
---@param type 'dir'|'direction'|'glue'|'pdf_literal'|'pdf_action'|'pdf_window'|'color_stack'|'pagestate'
---
---@return string[]|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/node.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function node.values(type) end ---
---The fontloader library is sort of independent of the rest in the sense that it
---can load font into a *Lua* table that then can be converted into a table suitable
---for *TeX*. The library is an adapted subset of *FontForge* and as such gives a
---similar view on a font (which has advantages when you want to debug).
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
fontloader = {}

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L2256-2283](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2256-2283)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontInfo
---@field fontname string # The *PostScript* name of the font, for example `NimbusRoman-Regular`.
---@field fullname string # The formal name of the font, for example `NimbusRoman-Regular`.
---@field familyname string # The family name this font belongs to, for example `Nimbus Roman`.
---@field weight string # A string indicating the color value of the font, for example `Book`.
---@field version string # The internal font version, for example `1.00`,
---@field italicangle number # The slant angle, for example `0.0`, `-12.0`, `-9.4626`, `-10.5`, `-15.5` or `11.0`.
---@field units_per_em number # `1000` for *PostScript*-based fonts, usually `2048` for *TrueType*, for example `1000`.
---@field pfminfo FontloaderPfminfo #
---@field names FontloaderLangName # https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2278-2282

---
---Get various information fields from an font file.
---
---This function returns either `nil`, or a `table`, or an array of
---small tables (in the case of a *TrueType* collection). The returned table(s) will
---contain some fairly interesting information items from the font(s) defined by the
---file.
---
---Getting information through this function is (sometimes much) more efficient than
---loading the font properly, and is therefore handy when you want to create a
---dictionary of available fonts based on a directory contents.
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L2257-L2284](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2257-L2284)
---
---@param filename string The path a font file.
---
---@return FontInfo|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fontloader.info(filename) end

---
---Parse a font file and return a table representing its contents. The optional argument is the name of the desired font in case of font collection files. The optional return value contains any parser error strings.
---
---Open an *OpenType* font.
---
---If you want to use an *OpenType* font, you have to get the metric information
---from somewhere. Using the `fontloader` library, the simplest way to get
---that information is thus:
---
---```lua
---function load_font (filename)
---  local metrics = nil
---  local font = fontloader.open(filename)
---  if font then
---     metrics = fontloader.to_table(font)
---     fontloader.close(font)
---  end
---  return metrics
---end
---
---myfont = load_font('/opt/tex/texmf/fonts/data/arial.ttf')
---```
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L198-L263](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L198-L263)
---
---@param filename string
---@param fontname? string
---
---@return FontloaderFont font # The first return value is a userdata representation of the font.
---@return table warnings # The second return value is a table containing any warnings and errors reported by fontloader while opening the font. In normal typesetting, you would probably ignore the second argument, but it can be useful for debugging purposes.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fontloader.open(filename, fontname) end

---
---If you want to use an \OPENTYPE\ font, you have to get the metric information
---from somewhere. Using the fontloader library, the simplest way to get
---that information is thus:
---
---__Example:__
---
---```lua
---function load_font (filename)
---  local metrics = nil
---  local font = fontloader.open(filename)
---  if font then
---     metrics = fontloader.to_table(font)
---     fontloader.close(font)
---  end
---  return metrics
---end
---
---myfont = load_font('/opt/tex/texmf/fonts/data/arial.ttf')
---```
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L2244-L2255](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2244-L2255)
---
---@param font FontloaderFont
---
---@return FontloaderFont|false f
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fontloader.to_table(font) end

---
---Discard a loaded font.
---
---__Example:__
---
---```lua
---local f = fontloader.open(
---              '/usr/share/fonts/opentype/urw-base35/NimbusRoman-Regular.otf')
---print(fontloader.to_table(f)) -- table: 0x3d23b50
---fontloader.close(f)
---print(fontloader.to_table(f)) -- false
---```
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L265-L280](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L265-L280)
---* Source file of the `LuaTeX` manual: [luatex-fontloader.tex#L149-153](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/4f2b914d365bab8a2747afe6e8c86d0f1c8475f7/manual/luatex-fontloader.tex#L149-153)
---
---@param font FontloaderFont
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fontloader.close(font) end

---
---Apply a feature file to a fontloader table.
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L313-L334](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L313-L334)
---
---@param font userdata
---@param filename string
---
---@return table errors
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fontloader.apply_featurefile(font, filename) end

---
---Apply an AFM file to a fontloader table.
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L336-L355](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L336-L355)
---
---@param font userdata
---@param filename string
---
---@return table errors
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fontloader.apply_afmfile(font, filename) end

---
---@alias FontloaderFontField
---|`table_version`
---|`fontname`
---|`fullname`
---|`familyname`
---|`weight`
---|`copyright`
---|`filename`
---|`version`
---|`italicangle`
---|`upos`
---|`uwidth`
---|`ascent`
---|`descent`
---|`uniqueid`
---|`glyphcnt`
---|`glyphmax`
---|`glyphmin`
---|`units_per_em`
---|`lookups`
---|`glyphs`
---|`hasvmetrics`
---|`onlybitmaps`
---|`serifcheck`
---|`isserif`
---|`issans`
---|`encodingchanged`
---|`strokedfont`
---|`use_typo_metrics`
---|`weight_width_slope_only`
---|`head_optimized_for_cleartype`
---|`uni_interp`
---|`map`
---|`origname`
---|`private`
---|`xuid`
---|`pfminfo`
---|`names`
---|`cidinfo`
---|`subfonts`
---|`comments`
---|`fontlog`
---|`cvt_names`
---|`ttf_tables`
---|`ttf_tab_saved`
---|`texdata`
---|`anchor_classes`
---|`kerns`
---|`vkerns`
---|`gsub`
---|`gpos`
---|`features`
---|`mm`
---|`chosenname`
---|`macstyle`
---|`fondname`
---|`design_size`
---|`fontstyle_id`
---|`fontstyle_name`
---|`design_range_bottom`
---|`design_range_top`
---|`strokewidth`
---|`mark_classes`
---|`creationtime`
---|`modificationtime`
---|`os2_version`
---|`sfd_version`
---|`math`
---|`validation_state`
---|`horiz_base`
---|`vert_base`
---|`extrema_bound`
---|`notdef_loc`

---
---__Example:__
---
---```lua
---local f = fontloader.open('/usr/share/fonts/opentype/urw-base35/NimbusRoman-Regular.otf')
---for _, value in ipairs(fontloader.fields(f)) do
---    print(value, f[value])
---end
---```
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L2491-L2511](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2491-L2511)
---
---@param font FontloaderFont
---
---@return FontloaderFontField[] fields
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fontloader.fields(font) end

---
---@alias FontloaderGlyphField
---|`name`
---|`unicode`
---|`boundingbox`
---|`vwidth`
---|`width`
---|`lsidebearing`
---|`class`
---|`kerns`
---|`vkerns`
---|`dependents`
---|`lookups`
---|`ligatures`
---|`comment`
---|`anchors`
---|`altuni`
---|`tex_height`
---|`tex_depth`
---|`is_extended_shape`
---|`italic_correction`
---|`top_accent`
---|`vert_variants`
---|`horiz_variants`
---|`mathkern`

---
------__Example:__
---
---```lua
---local f = fontloader.open('/usr/share/fonts/opentype/urw-base35/NimbusRoman-Regular.otf')
---for _, value in ipairs(fontloader.fields(f.glyphs[1])) do
---    print(value, f[value])
---end
---```---
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L2491-L2511](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2491-L2511)
---
---@param glyph FontloaderGlyph
---
---@return FontloaderGlyphField[] fields
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fontloader.fields(glyph) end

---
---__Example:__
---
---```lua
---local f = fontloader.open('/usr/share/fonts/opentype/urw-base35/NimbusRoman-Regular.otf')
---
---for _, value in ipairs(fontloader.fields(f)) do
---    print(value, f[value])
---end
---```
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1899-2241](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1899-2241)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderFont
---@field table_version string # Indicates the metrics version (currently `0.3`), for example `0.5`; Corresponding C source code: [luafflib.c#L2772-L2774](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2772-L2774).
---@field fontname string # The *PostScript* font name, for example `NimbusRoman-Regular`; Corresponding C source code: [luafflib.c#L2775-L2777](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2775-L2777).
---@field fullname string # The official (human-oriented) font name, for example `Nimbus Roman Regular`; Corresponding C source code: [luafflib.c#L2778-L2780](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2778-L2780).
---@field familyname string # The family name, for example `Nimbus Roman`; Corresponding C source code: [luafflib.c#L2781-L2783](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2781-L2783).
---@field weight string # The weight indicator, for example `Regular`; Corresponding C source code: [luafflib.c#L2784-L2786](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2784-L2786).
---@field copyright string # The copyright information, for example `Copyright (URW)++,Copyright 2014 by (URW)++ Design & Development`; Corresponding C source code: [luafflib.c#L2787-L2789](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2787-L2789).
---@field filename string # The file name; Corresponding C source code: [luafflib.c#L2790-L2792](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2790-L2792).
---@field version string # The font version; Corresponding C source code: [luafflib.c#L2793-L2795](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2793-L2795).
---@field italicangle integer # The slant angle; Corresponding C source code: [luafflib.c#L2796-L2798](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2796-L2798)
---@field units_per_em integer # `1000` for *PostScript*-based fonts, usually `2048` for *TrueType*; Corresponding C source code: [luafflib.c#L2827-L2829](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2827-L2829).
---@field ascent integer # The height of ascender in `units_per_em`; Corresponding C source code: [luafflib.c#L2805-L2807](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2805-L2807).
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2808-L2810
---@field descent integer # The depth of descender in `units_per_em`; Corresponding C source code: [luafflib.c#L2808-L2810](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2808-L2810).
---@field upos integer # Corresponding C source code: [luafflib.c#L2799-L2801](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2799-L2801)
---@field uwidth integer # Corresponding C source code: [luafflib.c#L2802-L2804](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2802-L2804)
---@field uniqueid integer # Corresponding C source code: [luafflib.c#L2811-L2813](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2811-L2813)
---@field glyphs FontloaderGlyph[] # The `glyphs` is an array containing the per-character information (quite a few of these are only present if non-zero); Corresponding C source code: [luafflib.c#L2838-L2845](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2838-L2845).
---@field glyphcnt integer # The number of included glyphs; Corresponding C source code: [luafflib.c#L2814-L2820](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2814-L2820).
---@field glyphmax integer # The maximum used index the glyphs array; Corresponding C source code: [luafflib.c#L2821-L2823](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2821-L2823).
---@field glyphmin integer # The minimum used index the glyphs array; Corresponding C source code: [luafflib.c#L2824-L2826](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2824-L2826).
---@field notdef_loc integer # The location of the `.notdef` glyph or `-1` when not present, for example `0`, `-1`, `854`; Corresponding C source code: [luafflib.c#L1920](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1920).
---@field hasvmetrics integer # Corresponding C source code: [luafflib.c#L2846-L2848](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2846-L2848).
---@field onlybitmaps integer # Corresponding C source code: [luafflib.c#L2849-L2851](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2849-L2851).
---@field serifcheck integer # Corresponding C source code: [luafflib.c#L2852-L2854](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2852-L2854).
---@field isserif integer # Corresponding C source code: [luafflib.c#L2855-L2857](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2855-L2857).
---@field issans integer # Corresponding C source code: [luafflib.c#L2858-L2860](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2858-L2860).
---@field encodingchanged integer # Corresponding C source code: [luafflib.c#L2861-L2863](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2861-L2863).
---@field strokedfont integer # Corresponding C source code: [luafflib.c#L2864-L2866](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2864-L2866).
---@field use_typo_metrics integer # Corresponding C source code: [luafflib.c#L2867-L2869](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2867-L2869).
---@field weight_width_slope_only integer # Corresponding C source code: [luafflib.c#L2870-L2872](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2870-L2872).
---@field head_optimized_for_cleartype integer # Corresponding C source code: [luafflib.c#L2873-L2875](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2873-L2875).
---@field uni_interp `unset`|`none`|`adobe`|`greek`|`japanese`|`trad_chinese`|`simp_chinese`|`korean`|`ams` # for example `none`; Corresponding C source code: [luafflib.c#L2876-L2878](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2876-L2878).
---@field origname string # The file name, as supplied by the user; Corresponding C source code: [luafflib.c#L2887-L2889](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2887-L2889).
---@field map FontloaderMap # Corresponding C source code: [luafflib.c#L2879-L2886](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2879-L2886).
---@field public private FontloaderPrivate # Corresponding C source code: [luafflib.c#L2890-L2897](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2890-L2897).
---@field xuid string # `[1021 618 1197637146 7437765]`; Corresponding C source code: [luafflib.c#L2041](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2041).
---@field pfminfo FontloaderPfminfo # Corresponding C source code: [luafflib.c#L2901-L2904](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2901-L2904).
---@field names FontloaderLangName # Corresponding C source code: [luafflib.c#L2905-L2912](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2905-L2912)
---@field cidinfo FontloaderCidinfo # Corresponding C source code: [luafflib.c#L2913-L2919](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2913-L2919).
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2920-L2930
---@field subfonts? table
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2931-L2933
---@field commments? string #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2934-L2936
---@field fontlog? string #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2937-L2947
---@field cvt_names? table #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2979-L2986
---@field anchor_classes FontloaderAnchorClasses #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2948-L2955
---@field ttf_tables? FontloaderTtfTables #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2956-L2963
---@field ttf_tab_saved? FontloaderTtfTables #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2964-L2978
---@field texdata? FontloaderTexdata #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2987-L2994
---@field kerns? FontloaderGlyph[] #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2995-L3002
---@field vkerns? FontloaderGlyph[] #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2830-L2837
---@field lookups FontloaderLockup #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L3011-L3018
---@field gpos FontloaderGpos #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L3019-L3026
---@field gsub? FontloaderGsub #
---
---@field features? unknown
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L3019-L3026
---@field mm? FontloaderMm #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L3027-L3029
---@field chosenname? string #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L3030-L3032
---@field macstyle integer # for example `-1`.
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L3033-L3035
---@field fondname? string #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L3036-L3038
---@field design_size integer # for example `0`.
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L3039-L3041
---@field fontstyle_id integer # for example `0`.
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L3042-L3049
---@field fontstyle_name? table #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2155
---@field design_range_bottom integer # for example `0`.
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L3053-L3055
---@field design_range_top integer # for example `0`.
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L3056-L3058
---@field strokewidth integer # for example `0`.
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L3059-L3072
---@field mark_classes? FontloaderMarkClasses|nil #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2171
---@field creationtime integer # for example `1495029639`.
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2172
---@field modificationtime integer # for example `1495029639`.
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2174
---@field os2_version integer # for example `3`.
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1705-L1764
---@field math? FontloaderMath #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2228
---@field validation_state? FontloaderValidationState #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2234
---@field horiz_base? FontloaderHorizVertBase #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2239
---@field vert_base? FontloaderHorizVertBase #
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2241
---@field extrema_bound integer # for example `0`.
---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1931-L1939
---@field truetype integer # signals a *TrueType* font

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L959-L1120](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L959-L1120)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderGlyph
---@field name string # The glyph name, for example `space`.
---@field unicode integer # The unicode code point, or -1 , for example `32`.
---@field boundingbox integer[] Array of four numbers, for example `{ 0, 0, 0, 0 }`.
---@field width integer # Only for horizontal fonts
---@field vwidth integer # Only for vertical fonts
---@field tsidebearing integer # Only for vertical ttf/otf fonts, and only if non-zero
---@field lsidebearing integer # Only if non-zero and not equal to boundingbox[1]
---@field class string # one of "none", "base", "ligature", "mark", "component" (if not present, the glyph class is “automatic”)
---@field kerns FontloaderGlyphKern[] only for horizontal fonts, if set
---@field vkerns table only for vertical fonts, if set
---@field dependents string[] linear array of glyph name strings, only if nonempty
---@field lookups table # only if nonempty
---@field ligatures table # only if nonempty
---@field anchors table # only if set
---@field comment string # only if set
---@field tex_height integer # only if set
---@field tex_depth integer # only if set
---@field italic_correction integer # only if set
---@field top_accent integer # only if set
---@field is_extended_shape integer # only if this character is part of a math extension list
---@field altuni table # alternate *Unicode* items
---@field vert_variants FontloaderVertHorizVariants #
---@field horiz_variants FontloaderVertHorizVariants #
---@field mathkern FontloaderMathkern #

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderGlyphKern
---@field char string # for example `afii10065`.
---@field lookup string # for example `pp_l_0_s`.
---@field off integer # for example `7`.

---
---The top-level map is a list of encoding mappings. Each of those is a table
---itself.
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1346-L1404](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1346-L1404)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderMap
---@field enccount integer # for example `65537`
---@field encmax integer # for example `65537`
---@field backmax integer # for example `855`
---@field remap FontloaderRemap #
---@field map table<integer, integer> # non-linear array of mappings, for example `{ [32] = 1, [33] = 2, [34] = 3 }`
---@field backmap integer[] # non-linear array of backward mappings, for example `{ 32, 33, 34, 35, 36, 37, 38, 8217 }`
---@field enc FontloaderEnc #

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1353-L1359](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1353-L1359)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderRemap
---@field firstenc integer #
---@field lastenc integer #
---@field infont integer #

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1284-L1336](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1284-L1336)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderEnc
---@field enc_name string #
---@field char_cnt integer #
---@field char_max integer #
---@field unicode integer[] # array of *Unicode* position numbers
---@field psnames string[] # array of *PostScript* glyph names
---@field builtin integer #
---@field hidden integer #
---@field only_1byte integer #
---@field has_1byte integer #
---@field has_2byte integer #
---@field is_unicodebmp integer # only if non-zero
---@field is_unicodefull integer # only if non-zero
---@field is_custom integer # only if non-zero
---@field is_original integer # only if non-zero
---@field is_compact integer # only if non-zero
---@field is_japanese integer # only if non-zero
---@field is_korean integer # only if non-zero
---@field is_tradchinese integer # only if non-zero [name?]
---@field is_simplechinese integer # only if non-zero
---@field low_page integer #
---@field high_page integer #
---@field iconv_name string #
---@field iso_2022_escape string #

---
---This is the font's private *PostScript* dictionary, if any. Keys and values are
---both strings.
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L2035-L2039](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2035-L2039)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FontloaderPrivate string<string, string>

---
---CID fonts (Character Identifier Fonts) were developed by Adobe to
---display Asian font formats with many different symbols. In CID fonts,
---CIDs (Character Identifiers) are used to address glyph descriptions
---within the font. A so-called CMap (Character Map) is used to
---establish correspondences between the CIDs and the character codes.
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L2053-L2058](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2053-L2058)
---* https://www.compart.com/de/glossar/cid-fonts
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderCidinfo
---@field registry string #
---@field ordering string #
---@field supplement integer # For example `0`.
---@field version number # For example `0`.

---
---The `pfminfo` table contains most of the OS/2 information.
---
---PFM stands for PostScript Font Metrics and is the suffix of a file
---associated with a PostScript Type 1 font. The PFM file contains
---metric data such as kerning values.
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1200-L1281](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1200-L1281)
---* https://www.typografie.info/3/wiki.html/p/pfm-r2/
---* https://github.com/fontforge/fontforge/blob/2d4ccf06b68ce1e31386cbfa5822d448378b6500/fontforge/splinefont.h#L1112-L1148
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderPfminfo
---@field pfmset integer # for example `0` or `1`.
---@field winascent_add integer # for example `0` or `1`.
---@field windescent_add integer # for example `0` or `1`.
---@field hheadascent_add integer # for example `0` or `1`.
---@field hheaddescent_add integer # for example `0` or `1`.
---@field typoascent_add integer # for example `0` or `1`.
---@field typodescent_add integer # for example `0` or `1`.
---@field subsuper_set integer # for example `0` or `1`.
---@field panose_set integer # for example `0` or `1`.
---@field hheadset integer # for example `0` or `1`.
---@field vheadset integer # for example `0` or `1`.
---@field pfmfamily integer # for example `17`, `65` or `81`.
---@field weight integer # for example `400`.
---@field width integer # for example `5`.
---@field avgwidth integer # for example `725`.
---@field firstchar integer # for example `32`.
---@field lastchar integer # for example `61440`.
---@field fstype integer # for example `0`.
---@field linegap integer # for example `0`.
---@field vlinegap integer # for example `0`.
---@field hhead_ascent integer # for example `870`.
---@field hhead_descent number # for example `-373`.
---@field os2_typoascent integer # The typographic ascender for this font. This field should be combined with the sTypoDescender and sTypoLineGap values to determine default line spacing. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#stypoascender
---@field os2_typodescent integer # The typographic descender for this font. This field should be combined with the sTypoAscender and sTypoLineGap values to determine default line spacing. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#stypodescender
---@field os2_typolinegap integer # The typographic line gap for this font. This field should be combined with the sTypoAscender and sTypoDescender values to determine default line spacing. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#stypolinegap
---@field os2_winascent integer # https://learn.microsoft.com/en-us/typography/opentype/spec/os2#uswinascent
---@field os2_windescent integer # https://learn.microsoft.com/en-us/typography/opentype/spec/os2#uswindescent
---@field os2_subxsize integer # The recommended horizontal size in font design units for subscripts for this font. Should be > 0. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#ysubscriptxsize
---@field os2_subysize integer # The recommended vertical size in font design units for subscripts for this font. Should be > 0. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#ysubscriptysize
---@field os2_subxoff integer # The recommended horizontal offset in font design units for subscripts for this font. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#ysubscriptxoffset
---@field os2_subyoff integer # The recommended vertical offset in font design units from the baseline for subscripts for this font. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#ysubscriptyoffset
---@field os2_supxsize integer # The recommended horizontal size in font design units for superscripts for this font. Should be > 0. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#ysuperscriptxsize
---@field os2_supysize integer # The recommended vertical size in font design units for superscripts for this font. Should be > 0. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#ysuperscriptysize
---@field os2_supxoff integer # The recommended horizontal offset in font design units for superscripts for this font. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#ysuperscriptxoffset
---@field os2_supyoff integer # The recommended vertical offset in font design units from the baseline for superscripts for this font. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#ysuperscriptyoffset
---@field os2_strikeysize integer # Thickness of the strikeout stroke in font design units. Should be > 0. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#ystrikeoutsize
---@field os2_strikeypos integer # The position of the top of the strikeout stroke relative to the baseline in font design units. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#ystrikeoutposition
---@field os2_family_class integer # This field provides a classification of font-family design. # https://learn.microsoft.com/en-us/typography/opentype/spec/os2#sfamilyclass
---@field os2_xheight integer # The height of lower case letters such as “x”
---@field os2_capheight integer # This metric specifies the distance between the baseline and the approximate height of uppercase letters measured in font design units. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#scapheight
---@field os2_defaultchar integer # This is the Unicode code point, in UTF-16 encoding, of a character that can be used for a default glyph if a requested character is not supported in the font. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#scapheight
---@field os2_breakchar integer # This is the Unicode code point, in UTF-16 encoding, of a character that can be used as a default break character. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#usbreakchar
---@field os2_vendor string # The four character identifier for the vendor of the given typeface. https://learn.microsoft.com/en-us/typography/opentype/spec/os2#achvendid
---@field codepages table # A two-number array of encoded code pages
---@field unicoderages table # A four-number array of encoded unicode ranges
---@field panose table #

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1222-L1232](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1222-L1232)
---* https://learn.microsoft.com/en-us/typography/opentype/spec/os2#panose
---* https://monotype.github.io/panose/pan1.htm
---* https://developer.apple.com/fonts/TrueType-Reference-Manual/RM06/Chap6OS2.html
---* https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/manual/luatex-fontloader.tex#L694-715
---* https://learn.microsoft.com/en-us/windows/win32/api/wingdi/ns-wingdi-panose
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderPanose
---@field familytype `Any`|`No Fit`|`Text and Display`|`Script`|`Decorative`|`Pictorial` # Values as in the *OpenType* font specification, for example `Text and Display`; Corresponding C source code: [luafflib.c#L1122-1124](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1122-1124).
---@field serifstyle `Any`|`No Fit`|`Cove`|`Obtuse Cove`|`Square Cove`|`Obtuse Square Cove`|`Square`|`Thin`|`Bone`|`Exaggerated`|`Triangle`|`Normal Sans`|`Obtuse Sans`|`Perp Sans`|`Flared`|`Rounded` # See the *OpenType* font specification for values, for example `Normal Sans`; Corresponding C source code: [luafflib.c#L1126-1131](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1126-1131).
---@field weight `Any`|`No Fit`|`Very Light`|`Light`|`Thin`|`Book`|`Medium`|`Demi`|`Bold`|`Heavy`|`Black`|`Nord` # For example `Medium`; Corresponding C source code: [luafflib.c#L1133-1136](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1133-1136).
---@field proportion `Any`|`No Fit`|`Old Style`|`Modern`|`Even Width`|`Expanded`|`Condensed`|`Very Expanded`|`Very Condensed`|`Monospaced` # Values as in the *OpenType* font specification, for example `Monospaced`; Corresponding C source code: [luafflib.c#L1138-1141](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1138-1141).
---@field contrast `Any`|`No Fit`|`None`|`Very Low`|`Low`|`Medium Low`|`Medium`|`Medium High`|`High`|`Very High` # Values as in the *OpenType* font specification, for example `Medium`; Corresponding C source code: [luafflib.c#L1143-1146](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1143-1146).
---@field strokevariation `Any`|`No Fit`|`Gradual/Diagonal`|`Gradual/Transitional`|`Gradual/Vertical`|`Gradual/Horizontal`|`Rapid/Vertical`|`Rapid/Horizontal`|`Instant/Vertical` # Values as in the *OpenType* font specification, for example `No Fit`; Corresponding C source code: [luafflib.c#L1148-1153](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1148-1153).
---@field armstyle `Any`|`No Fit`|`Straight Arms/Horizontal`|`Straight Arms/Wedge`|`Straight Arms/Vertical`|`Straight Arms/Single Serif`|`Straight Arms/Double Serif`|`Non-Straight Arms/Horizontal`|`Non-Straight Arms/Wedge`|`Non-Straight Arms/Vertical`|`Non-Straight Arms/Single Serif`|`Non-Straight Arms/Double Serif` # Values as in the *OpenType* font specification, for example `Straight Arms/Horizontal`; Corresponding C source code: [luafflib.c#L1155-1163](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1155-1163).
---@field letterform `Any`|`No Fit`|`Normal/Contact`|`Normal/Weighted`|`Normal/Boxed`|`Normal/Flattened`|`Normal/Rounded`|`Normal/Off Center`|`Normal/Square`|`Oblique/Contact`|`Oblique/Weighted`|`Oblique/Boxed`|`Oblique/Flattened`|`Oblique/Rounded`|`Oblique/Off Center`|`Oblique/Square` # Values as in the *OpenType* font specification, for example `Oblique/Weighted`; Corresponding C source code: [luafflib.c#L1165-1172](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1165-1172).
---@field midline `Any`|`No Fit`|`Standard/Trimmed`|`Standard/Pointed`|`Standard/Serifed`|`High/Trimmed`|`High/Pointed`|`High/Serifed`|`Constant/Trimmed`|`Constant/Pointed`|`Constant/Serifed`|`Low/Trimmed`|`Low/Pointed`|`Low/Serifed` # Values as in the *OpenType* font specification, for example `Standard/Pointed`; Corresponding C source code: [luafflib.c#L1174-1180](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1174-1180).
---@field xheight `Any`|`No Fit`|`Constant/Small`|`Constant/Standard`|`Constant/Large`|`Ducking/Small`|`Ducking/Standard`|`Ducking/Large` # Values as in the *OpenType* font specification, for example `Constant/Small`; Corresponding C source code: [luafflib.c#L1182-1185](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/a4b8e13d3879e95c21e1719af0c3e31722bedd4f/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1182-1185).
---

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1418-L1416](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1418-L1416)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderLangName
---@field lang string # The language for this entry, for example `Basque`, `Spanish Mexico`, `Portuguese (Brasil)`, `Spanish (Modern)`, `French Canadian` or `English (US)`
---@field names FontloaderTrueTypeNames # The `names` keys are the actual *TrueType* name strings.

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L108-L115](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L108-L115)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderTrueTypeNames
---@field copyright string # for example `Copyright (c) 2003 by Bitstream, Inc. All Rights Reserved. DejaVu changes are in public domain, math extensions are in public domain.`.
---@field family string # for example `DejaVu Math TeX Gyre`.
---@field subfamily string # for example `Bold`.
---@field uniqueid string # for example `Monotype:Arial Bold:Version 2.82 (Microsoft)`.
---@field fullname string # for example `DejaVuMathTeXGyre-Regular`.
---@field version string # for example `Version 2.82`.
---@field postscriptname string # for example `Arial-BoldMT`.
---@field trademark string # for example `Arial® Trademark of The Monotype Corporation plc registered in the US Pat & TM Off. and elsewhere.`.
---@field manufacturer string # for example `Monotype Typography`.
---@field designer string # for example `Monotype Type Drawing Office - Robin Nicholas, Patricia Saunders 1982`.
---@field descriptor string # for example `Contemporary sans serif design, Arial contains ...`.
---@field venderurl string # for example `http://www.monotype.com/html/mtname/ms_arial.html`.
---@field designerurl string # for example `http://www.monotype.com/html/mtname/ms_welcome.html`.
---@field license string # for example `NOTIFICATION OF LICENSE AGREEMENT\r\n\r\nThis typeface is the property of Monotype Typography and its use by you is covered under the terms of a license agreement. ...`.
---@field licenseurl string # for example `http://dejavu-fonts.org/wiki/License`.
---@field idontknow string
---@field preffamilyname string # for example `JetBrains Mono NL`.
---@field prefmodifiers string # for example `65 Bold` or `Light Italic` or `SemiCondensed SemiBold`.
---@field compatfull string # for example `Noto Looped Lao ExtraCondensed SemiBold`.
---@field sampletext string # for example `How razorback-jumping frogs can level six piqued gymnasts!`.
---@field cidfindfontname string # for example `Vemana2000`.
---@field wwsfamily string
---@field wwssubfamily string # for example ``.

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1448-L1452](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1448-L1452)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderAnchorClasses
---@field name string # a descriptive id of this anchor class
---@field lookup string #
---@field type `mark`|`mkmk`|`curs`|`mklg` # type is actually a lookup subtype, not a feature name. Officially, these strings should be gpos_mark2mark etc.

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L580-L643](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L580-L643)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderGposSub
---@field flags FontloaderGposFlags #
---@field name string #
---@field features FontloaderGposFeatures
---@field subtables FontloaderGposSubtables

---
---The `gpos` table has one array entry for each lookup. (The `gpos_`
---prefix is somewhat redundant.)
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L74-L76](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L74-L76)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderGpos: FontloaderGposSub
---@field type `gpos_single`|`gpos_pair`|`gpos_cursive`|`gpos_mark2base`|`gpos_mark2ligature`|`gpos_mark2mark`|`gpos_context`|`gpos_contextchain`

---
---The flags table has a true value for each of the lookup flags that is actually
---set:
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L587-L612](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L587-L612)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderGposFlags
---@field r2l boolean #
---@field ignorebaseglyphs boolean #
---@field ignoreligatures boolean #
---@field ignorecombiningmarks boolean #
---@field mark_class string #
---

---
---The features subtable items of gpos have
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L523-L531](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L523-L531)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderGposFeatures
---@field tag string #
---@field scripts FontloaderGposFeaturesScripts #

---
---The scripts table within features has:
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L492-L515](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L492-L515)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderGposFeaturesScripts
---@field script string #
---@field langs string[]
---

---
---The subtables table has:
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L541-L572](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L541-L572)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderGposSubtables
---@field name string #
---@field suffix string # (only if used) % used by gpos_single to get a default
---@field anchor_classes number # (only if used)
---@field vertical_kerning number # (only if used)
---@field kernclass FontloaderGposSubtablesKernclass # (only if used)
---

---
---Note: the kernclass (as far as we can see) always has one entry so it could be one level
---deep instead. Also the seconds start at `[2]` which is close to the fontforge
---internals so we keep that too.
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1469-L1518](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1469-L1518)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderGposSubtablesKernclass
---@field firsts string[]
---@field seconds string[]
---@field lookup string # or an array of associated lookup(s)
---@field offsets integer[]

---
---This has identical layout to the `gpos` table, except for the
---type:
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L56-L58](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L56-L58)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderGsub: FontloaderGposSub
---@field type `gsub_single`|`gsub_multiple`|`gsub_alternate`|`gsub_ligature`|`gsub_context`|`gsub_contextchain`|`gsub_reversecontextchain`
---

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1454-L1461](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1454-L1461)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderTtfTables
---@field tag string #
---@field len integer #
---@field maxlen integer #
---@field data integer #

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1841-L1898](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1841-L1898)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderMm
---@field axes table # array of axis names
---@field instance_count integer #
---@field positions table # array of instance positions (\#axes * instances )
---@field defweights table # array of default weights for instances
---@field cdv string #
---@field ndv string #
---@field axismaps table #

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1820-L1839](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1820-L1839)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderAxismaps
---@field blends table # an array of blend points
---@field designs table # an array of design values
---@field min integer #
---@field def integer #
---@field max integer #

---
---The keys in this table are mark class names, and the values are a
---space-separated string of glyph names in this class.
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L2159-L2169](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2159-L2169)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FontloaderMarkClasses string<string, string>

---
---The math table has the variables that are also discussed in the chapter about
---math
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1706-L1767](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1706-L1767)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderMath
---@field ScriptPercentScaleDown integer
---@field ScriptScriptPercentScaleDown integer
---@field DelimitedSubFormulaMinHeight integer
---@field DisplayOperatorMinHeight integer
---@field MathLeading integer
---@field AxisHeight integer
---@field AccentBaseHeight integer
---@field FlattenedAccentBaseHeight integer
---@field SubscriptShiftDown integer
---@field SubscriptTopMax integer
---@field SubscriptBaselineDropMin integer
---@field SuperscriptShiftUp integer
---@field SuperscriptShiftUpCramped integer
---@field SuperscriptBottomMin integer
---@field SuperscriptBaselineDropMax integer
---@field SubSuperscriptGapMin integer
---@field SuperscriptBottomMaxWithSubscript integer
---@field SpaceAfterScript integer
---@field UpperLimitGapMin integer
---@field UpperLimitBaselineRiseMin integer
---@field LowerLimitGapMin integer
---@field LowerLimitBaselineDropMin integer
---@field StackTopShiftUp integer
---@field StackTopDisplayStyleShiftUp integer
---@field StackBottomShiftDown integer
---@field StackBottomDisplayStyleShiftDown integer
---@field StackGapMin integer
---@field StackDisplayStyleGapMin integer
---@field StretchStackTopShiftUp integer
---@field StretchStackBottomShiftDown integer
---@field StretchStackGapAboveMin integer
---@field StretchStackGapBelowMin integer
---@field FractionNumeratorShiftUp integer
---@field FractionNumeratorDisplayStyleShiftUp integer
---@field FractionDenominatorShiftDown integer
---@field FractionDenominatorDisplayStyleShiftDown integer
---@field FractionNumeratorGapMin integer
---@field FractionNumeratorDisplayStyleGapMin integer
---@field FractionRuleThickness integer
---@field FractionDenominatorGapMin integer
---@field FractionDenominatorDisplayStyleGapMin integer
---@field SkewedFractionHorizontalGap integer
---@field SkewedFractionVerticalGap integer
---@field OverbarVerticalGap integer
---@field OverbarRuleThickness integer
---@field OverbarExtraAscender integer
---@field UnderbarVerticalGap integer
---@field UnderbarRuleThickness integer
---@field UnderbarExtraDescender integer
---@field RadicalVerticalGap integer
---@field RadicalDisplayStyleVerticalGap integer
---@field RadicalRuleThickness integer
---@field RadicalExtraAscender integer
---@field RadicalKernBeforeDegree integer
---@field RadicalKernAfterDegree integer
---@field RadicalDegreeBottomRaisePercent integer
---@field MinConnectorOverlap integer
---@field FractionDelimiterSize integer
---@field FractionDelimiterDisplayStyleSize integer

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L2183-L2229](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2183-L2229)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderValidationState
---@field bad_ps_fontname string
---@field bad_glyph_table table
---@field bad_cff_table table
---@field bad_metrics_table table
---@field bad_cmap_table table
---@field bad_bitmaps_table table
---@field bad_gx_table table
---@field bad_ot_table table
---@field bad_os2_version any
---@field bad_sfnt_header any

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1785-L1818](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1785-L1818)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderHorizVertBase
---@field tags table # an array of script list tags
---@field scripts FontloaderScripts #

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1785-L1818](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1785-L1818)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderScripts
---@field baseline table #
---@field default_baseline number #
---@field lang FontloaderLang #

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1769-L1777](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1769-L1777)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderLang
---@field tag string # a script tag
---@field ascent number #
---@field descent number #
---@field features table # The `features` points to an array of tables with the same layout except that in those nested tables, the tag represents a language.

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L692-L714](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L692-L714)
---
---An array of alternate *Unicode* values. Inside that array are hashes with:
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderAltuni
---@field unicode number # this glyph is also used for this unicode
---@field variant number # the alternative is driven by this unicode selector

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L914-L930](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L914-L930)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderVertHorizVariants
---@field variants string #
---@field italic_correction number #
---@field parts FontloaderParts[] # The `parts` table is an array of smaller tables.

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L914-L930](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L914-L930)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderParts
---@field component string #
---@field extender number #
---@field start number #
---@field end number #
---@field advance number #

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L943-L957](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L943-L957)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderMathkern
---@field top_right FontloaderMathkernSubtable #
---@field bottom_right FontloaderMathkernSubtable #
---@field top_left FontloaderMathkernSubtable #
---@field bottom_left FontloaderMathkernSubtable #

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L932-L941](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L932-L941)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderMathkernSubtable
---@field height number #
---@field kern number #

---
---# `kerns`
---
---Substructure is identical to the per-glyph subtable.

---
---# `vkerns`
---
---Substructure is identical to the per-glyph subtable.

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L2096-L2107](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L2096-L2107)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderTexdata
---@field type `unset`|`text`|`math`|`mathext`
---@field params table # 22 font numeric parameters

--
---Top-level `lookups` is quite different from the ones at character level.
---The keys in this hash are strings, the values the actual lookups, represented as
---dictionary tables.
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1628-L1661](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1628-L1661)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderLockup
---@field type string #
---@field format `glyphs`|`class`|`coverage`|`reversecoverage`
---@field tag string #
---@field current_class string[]
---@field before_class string[]
---@field after_class string[]
---@field rules FontloaderLockupRule[] # an array of rule items

---
---Rule items have one common item and one specialized item:
---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1558-L1626](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1558-L1626)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderLockupRule
---@field lookups string[] a linear array of lookup names
---@field glyphs FontloaderLockupGlyph[] only if the parent's format is `glyphs`
---@field class FontloaderLockupClass[] only if the parent's format is `class`
---@field coverage FontloaderLockupsCoverage[] only if the parent's format is `coverage`
---@field reversecoverage FontloaderLockupReverseCoverage[] only if the parent's format is `reversecoverage`

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1562-L1568](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1562-L1568)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderLockupGlyph
---@field names string #
---@field back string #
---@field fore string #

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1570-L1580](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1570-L1580)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderLockupClass
---@field before number[]
---@field after number[]

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1582-L1591](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1582-L1591)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderLockupsCoverage
---@field current string[]
---@field before string[]
---@field after string[]

---
---__Reference:__
---
---* Corresponding C source code: [luafflib.c#L1593-L1604](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafontloader/src/luafflib.c#L1593-L1604)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fontloader.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class FontloaderLockupReverseCoverage
---@field current string[]
---@field before string[]
---@field after string[]
---@field replacements string #
---
---# Binary input from files with `fio`
---
---There is a whole set of helpers for reading numbers and strings from a file:
---`fio.readcardinal1`, `fio.readcardinal2`, `fio.readcardinal3`, `fio.readcardinal4`, `fio.readcardinaltable`, `fio.readinteger1`, `fio.readinteger2`,
---`fio.readinteger3`, `fio.readinteger4`, `fio.readintegertable`, `fio.readfixed2`, `fio.readfixed4`,
---`fio.read2dot14`, `fio.setposition`, `fio.getposition`, `fio.skipposition`, `fio.readbytes`, `fio.readbytetable`. They work on normal *Lua* file handles.
---
---This library provides a set of functions for reading numbers from a file and
---in addition to the regular `io` library functions.
---
---|                            |                                     |
---|----------------------------|-------------------------------------|
---| `readcardinal1(f)`         | a 1 byte unsigned integer           |
---| `readcardinal2(f)`         | a 2 byte unsigned integer           |
---| `readcardinal3(f)`         | a 3 byte unsigned integer           |
---| `readcardinal4(f)`         | a 4 byte unsigned integer           |
---| `readcardinaltable(f,n,b)` | `n` cardinals of `b` bytes          |
---| `readinteger1(f)`          | a 1 byte signed integer             |
---| `readinteger2(f)`          | a 2 byte signed integer             |
---| `readinteger3(f)`          | a 3 byte signed integer             |
---| `readinteger4(f)`          | a 4 byte signed integer             |
---| `readintegertable(f,n,b)`  | `n` integers of `b` bytes           |
---| `readfixed2(f)`            | a 2 byte float (used in font files) |
---| `readfixed4(f)`            | a 4 byte float (used in font files) |
---| `read2dot14(f)`            | a 2 byte float (used in font files) |
---| `setposition(f,p)`         | goto position `p`                   |
---| `getposition(f)`           | get the current position            |
---| `skipposition(f,n)`        | skip `n` positions                  |
---| `readbytes(f,n)`           | `n` bytes                           |
---| `readbytetable(f,n)`       | `n` bytes                           |
---
---There are eight additional little endian variants for the `cardinal[1-4]`
---and `integer[1-4]` readers: `cardinal[1-4]le` and `integer[1-4]le`.
---
--- https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
fio = {}

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L114-L122](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L114-L122)
---
---@param f file* # A file handle.
---
---@return integer # a 1 byte unsigned integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readcardinal1(f) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L137-L147](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L137-L147)
---
---@param f file* # A file handle.
---
---@return integer # a 2 byte unsigned integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readcardinal2(f) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L187-L198](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L187-L198)
---
---@param f file* # A file handle.
---
---@return integer # a 3 byte unsigned integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readcardinal3(f) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L241-L253](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L241-L253)
---
---@param f file* # A file handle.
---
---@return integer # a 4 byte unsigned integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readcardinal4(f) end

---
---@param f file* # A file handle.
---@param n integer
---@param b integer
---
---@return table<integer, integer>
function fio.readcardinaltable(f, n, b) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L137-L147](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L137-L147)
---
---little endian variant
---
---@param f file* # A file handle.
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readcardinal1le(f) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L148-L158](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L148-L158)
---
---little endian variant
---
---@param f file* # A file handle.
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readcardinal2le(f) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L199-L210](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L199-L210)
---
---little endian variant
---
---@param f file* # A file handle.
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readcardinal3le(f) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L254-L266](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L254-L266)
---
---little endian variant
---
---@param f file* # A file handle.
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readcardinal4le(f) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L432-L442](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L432-L442)
---
---@param f file* # A file handle.
---
---@return integer # a 1 byte signed integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readinteger1(f) end

---
---__Reference:__
---
---* Corresponding C source code: [liolibext.c#L460-L471](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L460-L471)
---
---@param f file* # A file handle.
---
---@return integer # a 2 byte signed integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readinteger2(f) end

---
---@param f file* # A file handle.
---
---@return integer # a 3 byte signed integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readinteger3(f) end

---
---@param f file* # A file handle.
---
---@return integer # a 4 byte signed integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readinteger4(f) end

---
---@param f file* # A file handle.
---@param n integer
---@param b integer
---
---@return table<integer, integer>
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readintegertable(f, n, b) end

---
------* Corresponding C source code: [liolibext.c#L432-L442](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/liolibext.c#L432-L442)
---
---little endian variant
---
---@param f file* # A file handle.
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readinteger1le(f) end

---
---little endian variant
---
---@param f file* # A file handle.
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readinteger2le(f) end

---
---little endian variant
---
---@param f file* # A file handle.
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readinteger3le(f) end

---
---little endian variant
---
---@param f file* # A file handle.
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readinteger4le(f) end

---
---@param f file* # A file handle.
---
---@return number # a 2 byte float (used in font files)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readfixed2(f) end

---
---@param f file* # A file handle.
---
---@return number # a 4 byte float (used in font files)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readfixed4(f) end

---
---@param f file* # A file handle.
---
---@return number # a 2 byte float (used in font files)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.read2dot14(f) end

---
---goto position `p`
---
---@param f file* # A file handle.
---@param p integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.setposition(f, p) end

---
---get the current position
---
---@param f file* # A file handle.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.getposition(f) end

---
---skip `n` positions.
---
---@param f file* # A file handle.
---@param n integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.skipposition(f, n) end

---
---Read `n` bytes.
---
---@param f file* # A file handle.
---@param n integer
---
---@return integer ...
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readbytes(f, n) end

---
---Read `n` bytes as a table.
---
---@param f file* # A file handle.
---@param n integer
---
---@return table<integer, integer>
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readbytetable(f, n) end

---
---@param f file* # A file handle.
---
---@return string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/fio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function fio.readline(f) end
---
---This library contains variables and functions that are related to the *PDF*
---backend.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
pdf = {}

---
---Modify the map-file.
---
---This function modifies the map-file by adding, replacing, or removing all entries (lines) of <map_file>.
---
---@param map_file string # File name of map-file.
---The first character of <map_file> has a special meaning:
--- * `'+'`  Add new entries, if they do not exist already.
--- * `'='`  Replace existing entries.
--- * `'-'`  Remove entries, if they exist.
--- * `''`   Flush map file first, then insert new entries.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.mapfile(map_file) end

---
---Modify the map-file.
---
---This function modifies the map-file by adding, replacing, or removing exactly one entry (line).
---
---@param map_line string # Entry (line) to be added, replace, or removed.
---The first character of <map_line> has a special meaning:
--- * `'+'`  Insert new entry, if it doesn't exist already.
--- * `'='`  Replace existing entry.
--- * `'-'`  Remove entry, if it exists.
--- * `''`   Flush map file first, then insert new entry.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.mapline(map_line) end

---
---Insert string <catalog> into the `/Catalog` dictionary.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with metadata. The value is written to the *PDF*
---file directly after the token registers set at the *TeX* end are written.
---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L612](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L612)
---
---@see pdf.getcatalog
---
---@param catalog string # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setcatalog(catalog) end

---
---Return string <catalog> that was set by `pdf.setcatalog()`.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with metadata. The value is written to the *PDF*
---file directly after the token registers set at the *TeX* end are written.
---
---@see pdf.setcatalog
---
---@return string | nil catalog # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getcatalog() end

---
---Insert string <info> into the `/Info` dictionary.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with metadata. The value is written to the *PDF*
---file directly after the token registers set at the *TeX* end are written.
---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L613](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L613)
---
---@see pdf.getinfo
---
---@param info string # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setinfo(info) end

---
---Return string <info> that was set by `pdf.setinfo()`.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with metadata. The value is written to the *PDF*
---file directly after the token registers set at the *TeX* end are written.
---
---@see pdf.setinfo
---
---@return string | nil info # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getinfo() end

---
---Insert string <names> into the `/Names` dictionary of the `/Catalog`.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with metadata. The value is written to the *PDF*
---file directly after the token registers set at the *TeX* end are written.
---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L614](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L614)
---
---@see pdf.getnames
---
---@param names string # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setnames(names) end

---
---Return string <names> that was set by `pdf.setnames()`.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with metadata. The value is written to the *PDF*
---file directly after the token registers set at the *TeX* end are written.
---
---@see pdf.setnames
---
---@return string | nil names # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getnames() end

---
---Insert string <trailer> into the `trailer` dictionary.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with metadata. The value is written to the *PDF*
---file directly after the token registers set at the *TeX* end are written.
---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L615](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L615)
---
---@see pdf.gettrailer
---
---@param trailer string # String.
---FIXME: pdf.settrailer() not working. Bug?
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.settrailer(trailer) end

---
---Return string <trailer> that was set by `pdf.settrailer()`.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with metadata. The value is written to the *PDF*
---file directly after the token registers set at the *TeX* end are written.
---
---@see pdf.settrailer
---
---@return string | nil trailer # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.gettrailer() end

---
---Insert string <attributes> into the current `/Page` dictionary.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with page resources. The variables have no interaction with the corresponding *PDF*
---backend token register. They are written to the *PDF* file directly after the
---token registers set at the *TeX* end are written.
---
------* Corresponding C source code: [lpdflib.c#L610](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L610)
---
---@see pdf.getpageattributes
---
---@param attributes string # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setpageattributes(attributes) end

---
---Return string <attributes> that was set by `pdf.setpageattributes()`.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with page resources. The variables have no interaction with the corresponding *PDF*
---backend token register. They are written to the *PDF* file directly after the
---token registers set at the *TeX* end are written.
---
---@see pdf.setpageattributes
---
---@return string | nil attributes # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getpageattributes() end

---
---Insert string <resources> into the `/Resources` dictionary of the current page.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with page resources. The variables have no interaction with the corresponding *PDF*
---backend token register. They are written to the *PDF* file directly after the
---token registers set at the *TeX* end are written.
---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L609](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L609)
---
---@see pdf.getpageresources
---
---@param resources string # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setpageresources(resources) end

---
---Return string <resources> that was set by `pdf.setpageresources()`.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with page resources. The variables have no interaction with the corresponding *PDF*
---backend token register. They are written to the *PDF* file directly after the
---token registers set at the *TeX* end are written.
---
---@see pdf.setpageresources
---
---@return string | nil resources # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getpageresources() end

---
---Insert string <attributes> into the `/Pages` dictionary of the `/Catalog`.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with page resources. The variables have no interaction with the corresponding *PDF*
---backend token register. They are written to the *PDF* file directly after the
---token registers set at the *TeX* end are written.
---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L611](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L611)
---
---@see pdf.setpageattributes
---
---@param attributes string # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setpagesattributes(attributes) end

---
---Return string <attributes> that was set by `pdf.setpageattributes()`.
---
---This function complements the corresponding *PDF* backend token lists dealing
---with page resources. The variables have no interaction with the corresponding *PDF*
---backend token register. They are written to the *PDF* file directly after the
---token registers set at the *TeX* end are written.
---
---@see pdf.setpageattributes
---
---@return string | nil attributes # String.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getpagesattributes() end

---
---This function complements the corresponding *PDF* backend token lists dealing
---with reuseable boxes and images. The variables have no interaction with the
---corresponding *PDF* backend token register. They are written to the *PDF*
---file directly after the token registers set at the *TeX* end are written.
---
---@see pdf.getxformattributes
---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L617](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L617)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setxformattributes() end

---
---This function complements the corresponding *PDF* backend token lists dealing
---with reuseable boxes and images. The variables have no interaction with the
---corresponding *PDF* backend token register. They are written to the *PDF*
---file directly after the token registers set at the *TeX* end are written.
---
---@see pdf.setxformattributes
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getxformattributes() end

---
---This function complements the corresponding *PDF* backend token lists dealing
---with reuseable boxes and images. The variables have no interaction with the
---corresponding *PDF* backend token register. They are written to the *PDF*
---file directly after the token registers set at the *TeX* end are written.
---
---@see pdf.getxformresources
---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L616](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L616)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setxformresources() end

---
---This function complements the corresponding *PDF* backend token lists dealing
---with reuseable boxes and images. The variables have no interaction with the
---corresponding *PDF* backend token register. They are written to the *PDF*
---file directly after the token registers set at the *TeX* end are written.
---
---@see pdf.setxformresources
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getxformresources() end

---
---Set major version number of the PDF file format.
---
---Note: This function sets the version number of the file header. It does not set the `/Version` key of the `/Catalog` as recommended for PDF-1.4 and later.
---
---The major version is
---normally 1 but when set to 2 some data will not be written to the file in order
---to comply with the standard. What minor version you set depends on what *PDF*
---features you use. This is out of control of *LuaTeX*.
---
---One can set the major version number to 2 but we cannot guarantee that the engine
---adapts itself correctly, because there is no public and free specification that
---we know of. Also, user constructed annotations are not checked and just passed
---to the file. On the other hand, the *PDF* that the engine generated is rather
---simple and not that version depending.
---
---@see pdf.getmajorversion
---
---@param n integer # Major version number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setmajorversion(n) end

---
---Return major version number of the PDF file format.
---
---@see pdf.setmajorversion
---
---@return integer n # Major version number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getmajorversion() end

---
---Set minor version number of the PDF file format.
---
---Note: This function sets the version number of the file header. It does not set the `/Version` key of the `/Catalog` as recommended for PDF-1.4 and later.
---
---@see pdf.getmajorversion
---
---@param n integer # Minor version number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setminorversion(n) end

---
---Return minor version number of the PDF file format.
---
---@see pdf.setminorversion
---
---@return integer n # Minor version number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getminorversion() end

---
---Return the `/CreationDate` string of the `/Info` dictionary.
---
---@return string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getcreationdate() end

---
---Set error level for inserting *PDF* files.
---
---A warning/error is issued if the inserted *PDF* file has a newer version number than the main *PDF* file.
--- * `level <= 0`: Issue a warning.
--- * `level > 0`: Issue an error.
---@param level integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setinclusionerrorlevel(level) end

---
---Return value set by `pdf.setinclusionerrorlevel()`.
---
---@return integer level # Error level.
---
---@see pdf.setinclusionerrorlevel
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getinclusionerrorlevel() end

---
---Set <ignore> status for inserting images.
--- * `ignore ~= 0`: Issue a warning if image file type is unknown.
--- * `ignore == 0`: Issue an error if image file type is unknown.
---LuaTeX can handle these file types: pdf, png, jpg, jbig2
---@param ignore integer # Ignore image.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setignoreunknownimages(ignore) end

---
---Return <ignore> status.
---
---@return integer ignore # Ignor image.
---
---@see pdf.setignoreunknownimages
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getignoreunknownimages() end

---
---Set bit field suppressing certain informational keys in the *PDF* file.
---
---@param flags integer # Bit field.
--- --------------------------------------------------------------
--- Bit  Meaning
--- ---  ---------------------------------------------------------
---   1  `/PTEX.FullBanner` in `/Info` dictionary of the trailer.
---   2  `/PTEX.FileName` in `/XObject` dictionary of an image. (1)
---   4  `/PTEX.PageNumber` in `/XObject` dictionary of an image.
---   8  `/PTEX.InfoDict` in `/XObject` dictionary of an image.
---  16  `/Creator` in `/Info` dictionary of the trailer.
---  32  `/CreationDate` in `/Info` dictionary of the trailer.
---  64  `/ModDate` in `/Info` dictionary of the trailer.
--- 128  `/Producer` in `/Info` dictionary of the trailer.
--- 256  `/Trapped` in `/Info` dictionary of the trailer.
--- 512  `/ID` of the trailer.
--- --------------------------------------------------------------
--- (1) Seems to work only if '\pdfvariable imageaddfilename = 1'
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setsuppressoptionalinfo(flags) end

---
---Return bit field set by `pdf.setsuppressoptionalinfo()`.
---
---@return integer flags # Bit field.
---
---@see pdf.setsuppressoptionalinfo
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getsuppressoptionalinfo() end

---
---Set `/ID` of the trailer.
---
---@param id string # Trailer ID.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.settrailerid(id) end

---
---Return `/ID` of the trailer.
---
---@return string id # Trailer ID.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.gettrailerid() end

---
---Add or remove `/CIDSet` entry in `/FontDescriptor`.
--- * `n ~= 0`: Remove `/CIDSet` entry.
--- * `n == 0`: Add `/CIDSet` entry.
---
---@param n integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setomitcidset(n) end

---
---Return <flag> set by `pdf.setomitcidset()`.
---
---@return integer flag
---
---@see pdf.setomitcidset
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getomitcidset() end

---
---Add or remove `/CharSet` entry in `/FontDescriptor`.
--- * `n ~= 0`: Remove `/CharSet` entry.
--- * `n == 0`: Add `/CharSet` entry.
---
---@param n integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setomitcharset(n) end

---
---Return value set by `pdf.setomitcharset()`.
---
---@return integer n
---
---@see pdf.setomitcharset
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getomitcharset() end

---
---Set compression level of streams.
---
---@param level integer # Minimum: 0, Maximum: 9
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setcompresslevel(level) end

---
---Return compression level of streams.
---
---@return integer level # Compression level.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getcompresslevel() end

---
---Set compression level of objects.
---
---@param level integer # Minimum: 0, Maximum: 9
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setobjcompresslevel(level) end

---
---Return compression level of objects.
---
---@return integer level # Compression level.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getobjcompresslevel() end

---
---Switch for recompressing streams of `/XObject`s.
---
---@param switch integer # 0: don't recompress, 1: do recompress.
---FIXME: pdf.setrecompress() not working. Bug?
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setrecompress(switch) end

---
---Return value set by `pdf.setrecompress()`.
---
---@return integer switch # 0: don't recompress, 1: do recompress.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getrecompress() end

---
---Add or remove `/ToUnicode` entry in a font dictionary.
---
---@param n integer
--- * `n ~= 0`: Add `/ToUnicode` entry.
--- * `n == 0`: Remove `/ToUnicode` entry.
--- Note: Use `\pdfextension glyphtounicode` to add a CMap mapping from character codes to Unicode values.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setgentounicode(n) end

---
---Return value set by `pdf.setgentounicode()`.
---
---@return integer n
---@see pdf.setgentounicode
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getgentounicode() end

---
---Set number of decimal digits used for writing numbers (floats) in the *PDF* file.
---
---@param ndigits integer # Number of decimal digits.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setdecimaldigits(ndigits) end

---
---Return value set by `pdf.setdecimaldigits()`.
---
---@return integer ndigits # Number of digits.
---
---@see pdf.setdecimaldigits
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getdecimaldigits() end

---
---Set resolution of PK fonts.
---@param resolution integer
---@param fixed_dpi integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setpkresolution(resolution, fixed_dpi) end

---
---Return resolution of PK fonts.
---
---@return integer resolution
---@return integer fixed_dpi
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getpkresolution() end

---
---Return object number of latest created objects.
---
---@return integer objnum # Number of latest created object.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getlastobj() end

---
---Return object number of latest created `/Annot` object (subtype `/Link`) that was created by `/pdfextension startlink`.
---
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getlastlink() end

---
---Return object number of latest created `/Annot` object that was created by `pdf.reserveobj('annot')`.
---
---@return integer objnum # Number of latest created object.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getlastannot() end

---
---Warning! Undocumented code!<p>
---TODO: Please contribute
---https://github.com/Josef-Friedrich/LuaTeX_Lua-API#how-to-contribute
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getretval() end

---
---Return the object number of the latest created pdf object.
---
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getmaxobjnum() end

---
---Return a string describing the object type.
---
---@param objnum integer # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getobjtype(objnum) end

---
---Return the number of the font resource name.
---The `/Font` entry of a `/Resource` dictionary maps resource names to font dictionaries.
---LuaTeX uses resource names like `/F36` or `/F52`, where the number corresponds to
---the font ID.
---
---@param id integer # Font ID.
---
---@return integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getfontname(id) end

---
---Return the object number of the `/Font` dictionary.
---
---@param id integer # Font ID.
---
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getfontobjnum(id) end

---
---Return the font size in scaled points.
---
---@param id integer # Font ID.
---
---@return integer # Font size.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getfontsize(id) end

---
---Warning! Undocumented code!<p>
---TODO: Please contribute
---https://github.com/Josef-Friedrich/LuaTeX_Lua-API#how-to-contribute
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getxformname() end

---
---Set horizontal and vertical offset from the top left corner of the page.
---
---@param h integer # Horizontal offset in scaled points.
---@param v integer # Vertical offset in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setorigin(h, v) end

---
---Return values set by `pdf.setorigin()`.
---
---@return integer h # Horizontal offset in scaled points.
---@return integer v # Vertical offset in scaled points.
---
---@see pdf.setorigin
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getorigin() end

---
---Set default resolution of a bitmap image.
---
---@param dpi integer # Resolution.
--- LuaTeX determines the size of an image as follows:
--- 1. It uses <width> and <height> parameter of `\useimageresource`.
--- 2. If (1) is missing, it uses the resolution given in the metadata of the image.
--- 3. If (2) is missing, it uses the resolution given by this function.
--- 4. If (3) is missing, it uses a resolution of 72 dpi.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setimageresolution(dpi) end

---
---Return value set by `pdf.setimageresolution()`.
---
---@return integer dpi # Resolution.
---
---@see pdf.setimageresolution
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getimageresolution() end

---
---Set margin size of hyperlinks.
---
---@param margin integer # Margin in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setlinkmargin(margin) end

---
---Return value set by `pdf.setlinkmargin()`.
---
---@return integer margin # Margin in scaled points.
---@see pdf.setlinkmargin
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getlinkmargin() end

---
---Set margin size of destination.
---
---@param margin integer # Margin in scaled points.
---FIXME pdf.setdestmargin() is not working. Bug?
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setdestmargin(margin) end

---
---Return value set by `pdf.setdestmargin()`.
---
---@return integer margin # Margin in scaled points.
---@see pdf.setdestmargin
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getdestmargin() end

---
---Set size of margin for threads.
---
---@param margin integer # Margin in scaled points.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setthreadmargin(margin) end

---
---Return value set by `pdf.setthreadmargin()`.
---
---@return integer margin # Margin in scaled points.
---
---@see pdf.setthreadmargin
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getthreadmargin() end

---
---Warning! Undocumented code!<p>
---TODO: Please contribute
---https://github.com/Josef-Friedrich/LuaTeX_Lua-API#how-to-contribute
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setxformmargin() end

---
---Warning! Undocumented code!<p>
---TODO: Please contribute
---https://github.com/Josef-Friedrich/LuaTeX_Lua-API#how-to-contribute
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getxformmargin() end

---
---Return coordinates of the current position. The origin of the coordinate system is at the lower left corner of the page.
---
---@return integer h # Horizontal position in scaled points.
---@return integer v # Vertical position in scaled points.
---
---Note: This function is mainly used inside `\latelua` calls.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getpos() end

---
---Return horizontal coordinate of the current position. The origin of the coordinate system is at the lower left corner of the page.
---
---@return integer h # Horizontal position in scaled points.
---Note: This function is mainly used inside `\latelua` calls.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.gethpos() end

---
---Return vertical coordinate of the current position. The origin of the coordinate system is at the lower left corner of the page.
---
---@return integer v # Vertical position in scaled points.
---
---Note: This function is mainly used inside `\latelua` calls.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getvpos() end

---
---Returns `true` is a CTM is currently used.
---
---@return boolean
---
---Note: This function is mainly used inside `\latelua` calls.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.hasmatrix() end

---
---Return the CTM (current transformation matrix) at the current position.
--- ┌         ┐
--- │ a  b  0 │
--- │ c  d  0 │
--- │ e  f  1 │
--- └         ┘
---@return integer a
---@return integer b
---@return integer c
---@return integer d
---@return integer e
---@return integer f
---Note: This function is mainly used inside `\latelua` calls.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getmatrix() end

---
---Write a string into the contents stream of the current page. If LuaTeX is currently inside a text block (**BT**...**ET**), close the text block first.
---
---@param str string # String.
---
---Note: This function is mainly used inside `\latelua` calls.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.print(str) end

---
---Write a string into the contents stream of the current page.
---
---@param type string # Type.
---@param str string # String.
---Parameter <type> determines how the string is written:
--- * `text`: String <str> is written inside a text block (**BT**...**ET**).
--- * `page`: String <str> is written outside a text block (**BT**...**ET**).
--- * `origin`: Like `page` but the CTM is inserted just before the string <str>.
--- * `direct`: String <str> is written directly, independ of whether this is inside or outside a text block.
--- * `raw`: Like `direct`.
----Note: This function is mainly used inside `\latelua` calls.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.print(type, str) end

---
---Create an object and write it immediately to the pdf file.
---
---The created object looks like this:
--->  <objnum> 0 obj
--->    <str>
--->  endobj
---@param objnum? integer # Object number (optional argument).
---@param str string # Contents of the object.
---
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.immediateobj(str) end

---
---Create an object and write it immediately to the pdf file.
---The created object looks like this:
--->  <objnum> 0 obj
--->    <contents of file <filename>>
--->  endobj
---@param objnum? integer # Object number (optional argument).
---@param file string # Literal string `"file"`.
---@param filename string # File name.
---
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.immediateobj(objnum, file, filename) end

---
---Create an object and write it immediately to the pdf file.
---
---The created object looks like this:
--->   <objnum> 0 obj
--->   <<
--->   <streamdict>
--->   /Length 123 >>
--->   stream
--->   <streamcontents>
--->   endstream
--->   endobj
---@param objnum? integer # Object number (optional argument).
---@param stream string # Literal string `"stream"`.
---@param streamcontents string # Contents of the stream.
---@param streamdict string # Stream dictionary.
---
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.immediateobj(stream, streamcontents, streamdict) end

---
---Create an object and write it immediately to the pdf file.
---
---The created object looks like this:
--->   <objnum> 0 obj
--->   <<
--->   <contents of `file` <filename>>
--->   /Length 123 >>
--->   stream
--->   <streamcontents>
--->   endstream
--->   endobj
---@param objnum? integer # Object number (optional argument).
---@param streamfile string # Literal string `"streamfile"`.
---@param filename string # File name.
---@param streamdict string # Stream dictionary.
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.immediateobj(streamfile, filename, streamdict) end

---
---Create an object.
---
---This object is written to the pdf file only if it is referenced later by `pdf.refobj()`
---The created object looks like this:
--->  <objnum> 0 obj
--->    <str>
--->  endobj
---@param objnum? integer # Object number (optional argument).
---@param str string # Contents of the object.
---
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.obj(str) end

---
---Create an object.
---
---This object is written to the pdf file only if it is referenced later by `pdf.refobj()`
---The created object looks like this:
--->  <objnum> 0 obj
--->    <contents of file <filename>>
--->  endobj
---
---@param objnum? integer # Object number (optional argument).
---@param file string # Literal string `"file"`.
---@param filename string # File name.
---
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.obj(objnum, file, filename) end

---
---Create an object.
---
---This object is written to the pdf file only if it is referenced later by `pdf.refobj()`
---The created object looks like this:
--->   <objnum> 0 obj
--->   <<
--->   <streamdict>
--->   /Length 123 >>
--->   stream
--->   <streamcontents>
--->   endstream
--->   endobj
---@param objnum? integer # Object number (optional argument).
---@param stream string # Literal string `"stream"`.
---@param streamcontents string # Contents of the stream.
---@param streamdict string # Stream dictionary.
---
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.obj(stream, streamcontents, streamdict) end

---
---Create an object.
---
---This object is written to the pdf file only if it is referenced later by `pdf.refobj()`
---The created object looks like this:
--->   <objnum> 0 obj
--->   <<
--->   <contents of `file` <filename>>
--->   /Length 123 >>
--->   stream
--->   <streamcontents>
--->   endstream
--->   endobj
---@param objnum? integer # Object number (optional argument).
---@param streamfile string # Literal string `"streamfile"`.
---@param filename string # File name.
---@param streamdict string # Stream dictionary.
---
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.obj(streamfile, filename, streamdict) end

---
---Create an object.
---
---@param keyvals table # Object specification.
---> keyvals = {
--->    type           = <string>, -- `'stream'` (stream object), `'raw'` (non-stream object)
--->    immediate      = <boolean>,
--->    objnum         = <number>, -- object number
--->    attr           = <string>, -- stream dictionary
--->    compresslevel  = <number>,
--->    objcompression = <boolean>,
--->    file           = <string>, -- contents of object/stream
--->    string         = <string>, -- contents of object/stream
--->    nolength       = <boolean>, -- omit `/Length` entry of stream dictionary
---> }
--- Keys `string` and `file` are mutual exclusive.
--- Key `nolength` omits `/Length` entry of the stream dictionary only if a `/Length` entry is given in the `attr` key.
---
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.obj(keyvals) end

---
---Write referenced object to pdf file.
---
---@param objnum integer # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.refobj(objnum) end

---
---Create an empty object.
---
---Use `pdf.obj()` or `pdf.immediateobj()` to define this object.
---
---@return integer objnum # Object number.
---
---@see pdf.obj
---@see pdf.immediateobj
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.reserveobj() end

---
---Create a annotation object.
---
---Annotation object are not written to file directly but must be registered with `pdf.registerannot()`.
---
---@param annot string # Literal string `'annot'`.
---
---@return integer objnum # Object number.
---@see pdf.registerannot
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.reserveobj(annot) end

---
---Return object number of page <pagenum>.
---
---This can be a forwar reference, i.e. page <pagenum> doesn't have to be created yet.
---
---@param pagenum integer # Page number.
---
---@return integer objnum # Object number.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getpageref(pagenum) end

---
---Add an `/Annot` object to the `/Annots` dictionary.
---
---@param objnum integer # Object number of `/Annot` object.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.registerannot(objnum) end

---
---Allocate a new color stack and returns it's id. The arguments
---are the same as for the similar backend extension primitive.
---
---```
---pdf.newcolorstack("0 g","page",true) --
---```
---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L1166-L1199](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L1166-L1199)
---
---@param s string
---@param literal_mode 'page'|'direct'|'origin'|'text'|'raw'
---@param page_start boolean
---
---@return integer color_stack_id
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.newcolorstack(s, literal_mode, page_start) end

---
---Force some additional code into the font resource.
---
---It can for
---instance be used to add a custom `ToUnicode` vector to a bitmap file.
---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L1202-L1216](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L1202-L1216)
---@param font_id integer
---@param pdf_code integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setfontattributes(font_id, pdf_code) end

---
---@deprecated
---@see pdf.getfontname
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.fontname() end

---
---@deprecated
---@see pdf.getpdffontsize
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.fontsize() end

---
---Return how many objects were created so far.
---
---@return integer n # Number of objects.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.maxobjnum() end

---
---Return number of object that are written or not yet written to file.
---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L1270-L1285](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L1270-L1285)
---
---@return integer written_obj # Object written to file.
---@return integer not_written_obj # Object not yet written to file.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.getnofobjects() end

---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L1218-L1235](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L1218-L1235)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.includechar() end

---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L1237-L1242](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L1237-L1242)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.includefont() end

---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L1244-L1268](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L1244-L1268)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.includeimage() end

---
---@deprecated
---@see pdf.getobjtype
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.objtype() end

---
---@deprecated
---@see pdf.getpageref
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.pageref() end

---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L871-L879](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L871-L879)
---
---@param force boolean
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.setforcefile(force) end

---
---__Reference:__
---
---* Corresponding C source code: [lpdflib.c#L1294-L1298](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lpdflib.c#L1294-L1298)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.settypeonewidemode() end

---
---@deprecated
---@see pdf.getxformname
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/pdf.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function pdf.xformname() end

---
---The definitions are developed in this repository: https://github.com/LuaCATS/luafilesystem
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luafilesystem/src/lfs.c
---Changes to upstream: global lfs table---
---@class LuaFileSystem.Attributes
---@field [LuaFileSystem.AttributeName] any
---@field ['mode'] LuaFileSystem.AttributeMode

---
---@alias LuaFileSystem.AttributeMode
---|'file'
---|'directory'
---|'link'
---|'socket'
---|'char device'
---|"block device"
---|"named pipe"

---
---@alias LuaFileSystem.AttributeName
---|'dev' -- on Unix systems, this represents the device that the inode resides on. On Windows systems, represents the drive number of the disk containing the file
---|'ino' -- on Unix systems, this represents the inode number. On Windows systems this has no meaning
---|'mode' -- string representing the associated protection mode (the values could be file, directory, link, socket, named pipe, char device, block device or other)
---|'nlink' -- number of hard links to the file
---|'uid' -- user-id of owner (Unix only, always 0 on Windows)
---|'gid' -- group-id of owner (Unix only, always 0 on Windows)
---|'rdev' -- on Unix systems, represents the device type, for special file inodes. On Windows systems represents the same as dev
---|'access' -- time of last access
---|'modification' -- time of last data modification
---|'change' -- time of last file status change
---|'size' -- file size, in bytes
---|'permissions' -- file permissions string
---|'blocks' -- block allocated for file; (Unix only)
---|'blksize' -- optimal file system I/O blocksize; (Unix only)

---
---@class LuaFileSystem
lfs = {}

--[[
Returns a table with the file attributes corresponding to filepath (or nil followed by an error message and a system-dependent error code in case of error). If the second optional argument is given and is a string, then only the value of the named attribute is returned (this use is equivalent to lfs.attributes(filepath)[request_name], but the table is not created and only one attribute is retrieved from the O.S.). if a table is passed as the second argument, it (result_table) is filled with attributes and returned instead of a new table. The attributes are described as follows; attribute mode is a string, all the others are numbers, and the time related attributes use the same time reference of os.time:
]]
---@overload fun(filepath:string):LuaFileSystem.Attributes
---@overload fun(filepath:string, result_table:LuaFileSystem.Attributes)
---@param filepath string
---@param request_name LuaFileSystem.AttributeName
---@return string|integer|LuaFileSystem.AttributeMode
function lfs.attributes(filepath, request_name) end

--[[
Changes the current working directory to the given path.
Returns true in case of success or nil plus an error string.
]]
---@param path string
---@return boolean, string
function lfs.chdir(path) end

--[[
Creates a lockfile (called lockfile.lfs) in path if it does not exist and returns the lock. If the lock already exists checks if it's stale, using the second parameter (default for the second parameter is INT_MAX, which in practice means the lock will never be stale. To free the the lock call lock:free().
In case of any errors it returns nil and the error message. In particular, if the lock exists and is not stale it returns the "File exists" message.
]]
---@param path string
---@param seconds_stale? number
---@return boolean, string
function lfs.lock_dir(path, seconds_stale) end

---
---Returns a string with the current working directory or nil plus an error string.
---@return string
function lfs.currentdir() end

--[[
Lua iterator over the entries of a given directory. Each time the iterator is called with dir_obj it returns a directory entry's name as a string, or nil if there are no more entries. You can also iterate by calling dir_obj:next(), and explicitly close the directory before the iteration finished with dir_obj:close(). Raises an error if path is not a directory.
]]
---@param path string
---@return fun():string
function lfs.dir(path) end

--[[
Locks a file or a part of it. This function works on open files; the file handle should be specified as the first argument. The string mode could be either r (for a read/shared lock) or w (for a write/exclusive lock). The optional arguments start and length can be used to specify a starting point and its length; both should be numbers.
Returns true if the operation was successful; in case of error, it returns nil plus an error string.
]]
---@param filehandle file*
---@param mode openmode
---@param start? integer
---@param length? integer
---@return boolean, string
function lfs.lock(filehandle, mode, start, length) end

--[[
    Creates a link. The first argument is the object to link to and the second is the name of the link. If the optional third argument is true, the link will by a symbolic link (by default, a hard link is created).
]]
---@param old string
---@param new string
---@param symlink? boolean
---@return boolean, string
function lfs.link(old, new, symlink) end

--[[
    Creates a new directory. The argument is the name of the new directory.
    Returns true in case of success or nil, an error message and a system-dependent error code in case of error.
]]
---@param dirname string
---@return boolean, string
function lfs.mkdir(dirname) end

--[[
    Removes an existing directory. The argument is the name of the directory.
    Returns true in case of success or nil, an error message and a system-dependent error code in case of error.
]]
---@param dirname string
---@return boolean, string
function lfs.rmdir(dirname) end

--[[
Sets the writing mode for a file. The mode string can be either "binary" or "text". Returns true followed the previous mode string for the file, or nil followed by an error string in case of errors. On non-Windows platforms, where the two modes are identical, setting the mode has no effect, and the mode is always returned as binary.
]]
---@param file string
---@param mode 'binary'|'text'
---@return boolean, string
function lfs.setmode(file, mode) end

--[[
Identical to lfs.attributes except that it obtains information about the link itself (not the file it refers to). It also adds a target field, containing the file name that the symlink points to. On Windows this function does not yet support links, and is identical to lfs.attributes.
]]
---@param filepath string
---@param request_name? LuaFileSystem.AttributeName
---@return LuaFileSystem.Attributes
function lfs.symlinkattributes(filepath, request_name) end

--[[
Set access and modification times of a file. This function is a bind to utime function. The first argument is the filename, the second argument (atime) is the access time, and the third argument (mtime) is the modification time. Both times are provided in seconds (which should be generated with Lua standard function os.time). If the modification time is omitted, the access time provided is used; if both times are omitted, the current time is used.
Returns true in case of success or nil, an error message and a system-dependent error code in case of error.
]]
---@param filepath string
---@param atime? integer
---@param mtime? integer
---@return boolean, string
function lfs.touch(filepath, atime, mtime) end

--[[
Unlocks a file or a part of it. This function works on open files; the file handle should be specified as the first argument. The optional arguments start and length can be used to specify a starting point and its length; both should be numbers.
Returns true if the operation was successful; in case of error, it returns nil plus an error string.
]]
---@param filehandle file*
---@param start? integer
---@param length? integer
---@return boolean, string
function lfs.unlock(filehandle, start, length) end

---
---Changes to upstream: global zlib table

---
---The definitions are developed in this repository: https://github.com/LuaCATS/lzlib

---
---`lzlib`, by Tiago Dionizio.
---
---__Reference:__
---
---* Old location: http://luaforge.net/projects/lzlib/.
---* More recent git repo (archived): https://github.com/LuaDist/lzlib
---* On luarocks: https://luarocks.org/modules/hisham/lzlib
---
---😱 [Types](https://github.com/LuaCATS/lzlib/blob/main/library/lzlib.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lzlib/pulls)
zlib = {}

---
---Return the zlib version.
---
---__Example:__
---
---```lua
---local version = zlib.version()
---assert(version == '1.2.13')
---```
---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L331-L335](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L331-L335)
---
---@return string version # For example `1.2.13`
---
---😱 [Types](https://github.com/LuaCATS/lzlib/blob/main/library/lzlib.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lzlib/pulls)
function zlib.version() end

---
---Return the initial `adler32` value.
---
---Call to update the `adler32` value, `adler32` is the current value, `buffer` is passed
---to `adler32` zlib function and the updated value is returned.
---
---This function is not related to compression but is exported anyway because it might be useful in applications using the compression library.
---
---__Example:__
---
---```lua
---local adler = zlib.adler32()
---assert(adler == 1.0)
---adler = zlib.adler32(adler, 'some text')
---assert(adler == 300417946.0)
---adler = zlib.adler32(adler, 'some text')
---assert(adler == 1144063795.0)
---adler = zlib.adler32(adler, 'some text')
---assert(adler == 2530937548.0)
---```
---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L338-L355](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L338-L355)
---
---@param adler32? number
---@param buffer? string
---
---@return number adler32
---
---😱 [Types](https://github.com/LuaCATS/lzlib/blob/main/library/lzlib.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lzlib/pulls)
function zlib.adler32(adler32, buffer) end

---
---Return the initial `crc32` (Cyclic Redundancy Check) value.
---
---Call to update the `crc32` value, `crc32` is the current value, `buffer` is passed
---to `crc32` zlib function and the updated value is returned.
---
---This function is not related to compression but is exported anyway because it might be useful in applications using the compression library.
---
---__Example:__
---
---```lua
---local crc = zlib.crc32()
---assert(crc == 0.0)
---crc = zlib.crc32(crc, 'some text')
---assert(crc == 1337638330.0)
---crc = zlib.crc32(crc, 'some text')
---assert(crc == 2768805016.0)
---crc = zlib.crc32(crc, 'some text')
---assert(crc == 1021719064.0)
---```
---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L358-L375](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L358-L375)
---
---@param crc32? number
---@param buffer? string
---
---@return number crc32
---
---😱 [Types](https://github.com/LuaCATS/lzlib/blob/main/library/lzlib.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lzlib/pulls)
function zlib.adler32(crc32, buffer) end

---
---Return a string containing the compressed buffer according to the given parameters.
---
---__Example:__
---
---```lua
---local orig = [[
---Lorem ipsum dolor sit amet, consetetur sadipscing elitr, sed diam
---nonumy eirmod tempor invidunt ut labore et dolore magna aliquyam erat,
---sed diam voluptua.
---]]
---assert(string.len(zlib.compress(orig)) < string.len(orig))
---```
---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L380-L441](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L380-L441)
---
---@param buffer string
---@param level? integer # The compression level must be `-1` (default compression), or between `0` and `9`: `1` gives best speed, `9` gives best compression, `0` gives no compression at all (the input data is simply copied a block at a time).
---@param method? integer # The method parameter is the compression method. It must be `8` (`Z_DEFLATED`) in this version of the library.
---@param window_bits? integer # `The window_bits` parameter is the base two logarithm of the maximum window size (the size of the history buffer). It should be in the range `8..15` for this version of the library, default `15`.
---@param mem_level? integer - default `8`
---@param strategy? integer - default `Z_DEFAULT_STRATEGY`
---
---@return string buffer
---
---😱 [Types](https://github.com/LuaCATS/lzlib/blob/main/library/lzlib.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lzlib/pulls)
function zlib.compress(buffer, level, method, window_bits, mem_level, strategy) end

---
---Return the decompressed stream after processing the given buffer.
---
---__Example:__
---
---```lua
---local uncompressed = 'Lorem ipsum'
---local compressed = zlib.compress(uncompressed)
---local result = zlib.decompress(compressed)
---assert(result == uncompressed)
---```
---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L445-L502](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L445-L502)
---
---@param buffer string
---@param window_bits? integer # `The window_bits` parameter is the base two logarithm of the maximum window size (the size of the history buffer). It should be in the range `8..15` for this version of the library, default `15`.
---
---@return string
---
---😱 [Types](https://github.com/LuaCATS/lzlib/blob/main/library/lzlib.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lzlib/pulls)
function zlib.decompress(buffer, window_bits) end

---
---Return a deflate stream.
---
---In the upstream project this function is called `zlib.deflate()`.
---
---__Example:__
---
---```lua
---local z_stream = zlib.compressobj(1)
---local result = z_stream:compress('test')
---assert(type(result) == 'string')
---```
---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L156-L173](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L156-L173)
---
---@param level? integer # The compression level must be `-1` (default compression), or between `0` and `9`: `1` gives best speed, `9` gives best compression, `0` gives no compression at all (the input data is simply copied a block at a time).
---@param method? integer # The method parameter is the compression method. It must be `8` (`Z_DEFLATED`) in this version of the library.
---@param window_bits? integer # `The window_bits` parameter is the base two logarithm of the maximum window size (the size of the history buffer). It should be in the range `8..15` for this version of the library, default `15`.
---@param mem_level? integer - default `8`
---@param strategy? integer - default `Z_DEFAULT_STRATEGY`
---
---@return ZStream
---
---😱 [Types](https://github.com/LuaCATS/lzlib/blob/main/library/lzlib.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lzlib/pulls)
function zlib.compressobj(level, method, window_bits, mem_level, strategy) end

---
---Return an inflate stream.
---
---In the upstream project this function is called `zlib.inflate()`.
---
---__Example:__
---
---```lua
---local z_stream_com = zlib.compressobj(1)
---local compressed = z_stream_com:compress('test')
---compressed = compressed .. z_stream_com:flush()
---
---local z_stream_decomp = zlib.decompressobj()
---local result = z_stream_decomp:decompress(compressed)
---assert(result == 'test')
---```
---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L177-L191](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L177-L191)
---
---@param window_bits? integer # `The window_bits` parameter is the base two logarithm of the maximum window size (the size of the history buffer). It should be in the range `8..15` for this version of the library, default `15`.
---
---@return ZStream
---
---😱 [Types](https://github.com/LuaCATS/lzlib/blob/main/library/lzlib.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lzlib/pulls)
function zlib.decompressobj(window_bits) end

---
---😱 [Types](https://github.com/LuaCATS/lzlib/blob/main/library/lzlib.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lzlib/pulls)
---@class ZStream
local ZStream = {}

---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L307-L322](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L307-L322)
---
function ZStream:adlerreset() end

---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L231-L263](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L231-L263)
---
---@param buffer string
---
---@return string buffer
function ZStream:compress(buffer) end

---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L195-L227](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L195-L227)
---
---@param buffer string
---
---@return string buffer
function ZStream:decompress(buffer) end

---
---Flush the output for deflate streams.
---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L267-L303](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L267-L303)
---
---😱 [Types](https://github.com/LuaCATS/lzlib/blob/main/library/lzlib.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lzlib/pulls)
function ZStream:flush() end

---
---Close the stream.
---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L138-L143](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L138-L143)
---
---😱 [Types](https://github.com/LuaCATS/lzlib/blob/main/library/lzlib.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/lzlib/pulls)
function ZStream:close() end

---
---__Reference:__
---
---* Corresponding C source code: [lzlib.c#L147-L152](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luazlib/lzlib.c#L147-L152)
---
---@return number adler
function ZStream.adler() end

---
---https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/trunk/source/texk/web2c/luatexdir/luasocket/src/mime.lua
---Changes to upstream: global mime table

---
---The definitions are developed in this repository: https://github.com/LuaCATS/luasocket

---
---https://lunarmodules.github.io/luasocket/mime.html
---
---The `mime` namespace offers filters that apply and remove common
---content transfer encodings, such as Base64 and Quoted-Printable.
---It also provides functions to break text into lines and change
---the end-of-line convention.
---MIME is described mainly in
---RFC 2045,
---2046,
---2047,
---2048, and
---2049.
---
---All functionality provided by the MIME module
---follows the ideas presented in
---
---LTN012, Filters sources and sinks.
---
---To obtain the `mime` namespace, run:
---
---```
----- loads the MIME module and everything it requires
---local mime = require("mime")
---```
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
mime = {}

---
---Returns a filter that decodes data from a given transfer content
---encoding.
---
---In the Quoted-Printable case, the user can specify whether the data is
---textual or binary, by passing the `mode` strings "`text`" or
---"`binary`". `Mode` defaults to "`text`".
---
---@param name 'base64'|'quoted-printable'
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.decode(name) end

---
---Returns a filter that encodes data according to a given transfer content
---encoding.
---
---In the Quoted-Printable case, the user can specify whether the data is
---textual or binary, by passing the `mode` strings "`text`" or
---"`binary`". `Mode` defaults to "`text`".
---
---@param name 'base64'|'quoted-printable'
---@param mode? 'text'|'binary'
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.encode(name, mode) end

---
---Converts most common end-of-line markers to a specific given marker.
---
---`Marker` is the new marker. It defaults to CRLF, the canonic
---end-of-line marker defined by the MIME standard.
---
---The function returns a filter that performs the conversion.
---
---Note: There is no perfect solution to this problem. Different end-of-line
---markers are an evil that will probably plague developers forever.
---This function, however, will work perfectly for text created with any of
---the most common end-of-line markers, i.e. the Mac OS (CR), the Unix (LF),
---or the DOS (CRLF) conventions. Even if the data has mixed end-of-line
---markers, the function will still work well, although it doesn't
---guarantee that the number of empty lines will be correct.
---
---@param marker string
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.normalize(marker) end

---
---Creates and returns a filter that performs stuffing of SMTP messages.
---
---Note: The `smtp.send`function
---uses this filter automatically. You don't need to chain it with your
---source, or apply it to your message body.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.stuff() end

---
---Returns a filter that breaks data into lines.
---
---The "`text`" line-wrap filter simply breaks text into lines by
---inserting CRLF end-of-line markers at appropriate positions.
---`Length` defaults 76.
---The "`base64`" line-wrap filter works just like the default
---"`text`" line-wrap filter with default length.
---The function can also wrap "`quoted-printable`" lines, taking care
---not to break lines in the middle of an escaped character. In that case, the
---line length is fixed at 76.
---
---For example, to create an encoding filter for the Quoted-Printable transfer content encoding of text data, do the following:
---
---```
---qp = ltn12.filter.chain(
---  mime.normalize(),
---  mime.encode("quoted-printable"),
---  mime.wrap("quoted-printable")
---)
---```
---
---Note: To break into lines with a different end-of-line convention, apply
---a normalization filter after the line break filter.
---
---@param name 'text'|'base64'|'quoted-printable'
---@param length integer
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.wrap(name, length) end

---
---Low-level filter to perform Base64 encoding.
---
---If `D` is `nil`, `A` is padded with
---the encoding of the remaining bytes of `C`.
---
---Note: The simplest use of this function is to encode a string into it's
---Base64 transfer content encoding. Notice the extra parenthesis around the
---call to `mime.b64`, to discard the second return value.
---
---```
---print((mime.b64("diego:password")))
-----&gt; ZGllZ286cGFzc3dvcmQ=
---```
---
---@param C string
---@param D? string
---
---@return string A - `A` is the encoded version of the largest prefix of `C..D` that can be encoded unambiguously.
---@return string B -  `B` has the remaining bytes of `C..D`, before encoding.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.b64(C, D) end

---
---Low-level filter to perform SMTP stuffing and enable transmission of
---messages containing the sequence "CRLF.CRLF".
---
---Note: The message body is defined to begin with
---an implicit CRLF. Therefore, to stuff a message correctly, the
---first `m` should have the value 2.
---
---```
---print((string.gsub(mime.dot(2, ".\r\nStuffing the message.\r\n.\r\n."), "\r\n", "\\n")))
-----&gt; ..\nStuffing the message.\n..\n..
---```
---
---Note: The `smtp.send`function
---uses this filter automatically. You don't need to
---apply it again.
---
---@param m integer - '`m`' should tell the same, but for the previous chunk.
---@param B? string
---
---@return string A - `A` is the stuffed version of `B`.
---@return integer n - `n` gives the number of characters from the sequence CRLF seen in the end of `B`.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.dot(m, B) end

---
---A, B = mime.eol(C [, D, marker])
---
---Low-level filter to perform end-of-line marker translation.
---For each chunk, the function needs to know if the last character of the
---previous chunk could be part of an end-of-line marker or not. This is the
---context the function receives besides the chunk.  An updated version of
---the context is returned after each new chunk.
---
---```
----- translates the end-of-line marker to UNIX
---unix = mime.eol(0, dos, "\n")
---```
---
---@param C integer # `C` is the ASCII value of the last character of the previous chunk, if it was a candidate for line break, or 0 otherwise.
---@param D? string
---@param marker string `Marker` gives the new end-of-line marker and defaults to CRLF.
---
---@return string A # `A` is the translated version of `D`.
---@return string B # `B` is the same as `C`, but for the current chunk.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.eol(C, D, marker) end

---
---Low-level filter to perform Quoted-Printable encoding.
---
---Note: The simplest use of this function is to encode a string into it's
---Quoted-Printable transfer content encoding.
---Notice the extra parenthesis around the call to `mime.qp`, to discard the second return value.
---
---```
---print((mime.qp("ma��")))
-----&gt; ma=E7=E3=
---```
---
---@param C string
---@param D? string If `D` is `nil`, `A` is padded with the encoding of the remaining bytes of `C`.
---@param marker string Throughout encoding, occurrences of CRLF are replaced by the `marker`, which itself defaults to CRLF.
---
---@return string A # `A` is the encoded version of the largest prefix of `C..D` that can be encoded unambiguously.
---@return string B # `B` has the remaining bytes of `C..D`, before decoding.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.qp(C, D, marker) end

---
---Low-level filter to break Quoted-Printable text into lines.
---
---Note: Besides breaking text into lines, this function makes sure the line
---breaks don't fall in the middle of an escaped character combination. Also,
---this function only breaks lines that are bigger than `length` bytes.
---
---@param n integer `n` should tell how many bytes are left for the first line of `B`
---@param B? string
---@param length? any # broken into lines of at most `length` bytes (defaults to 76).
---
---@return string A # `A` is a copy of `B`, broken into lines of at most `length` bytes (defaults to 76).
---@return integer m # returns the number of bytes left in the last line of `A`.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.qpwrp(n, B, length) end

---
---Low-level filter to perform Base64 decoding.
---
---If `D` is `nil`, `A` is the empty string
---and `B` returns whatever couldn't be decoded.
---
---Note: The simplest use of this function is to decode a string from it's
---Base64 transfer content encoding.
---Notice the extra parenthesis around the call to `mime.unqp`, to discard the second return value.
---
---```
---print((mime.unb64("ZGllZ286cGFzc3dvcmQ=")))
-----&gt; diego:password
---```
---
---@param C string
---@param D? string
---
---@return string A # `A` is the decoded version of the largest prefix of `C..D` that can be decoded unambiguously.
---@return string B # `B` has the remaining bytes of `C..D`, before decoding.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.unb64(C, D) end

---
---Low-level filter to remove the Quoted-Printable transfer content encoding
---from data.
---
---Note: The simplest use of this function is to decode a string from it's
---Quoted-Printable transfer content encoding.
---Notice the extra parenthesis around the call to `mime.unqp`, to discard the second return value.
---
---```
---print((mime.qp("ma=E7=E3=")))
-----&gt; ma��
---```
---
---@param C string
---@param D? string
---
---@return string A # `A` is the decoded version of the largest prefix of `C..D` that can be decoded unambiguously. If `D` is `nil`, `A` is augmented with the encoding of the remaining bytes of `C`.
---@return string B # `B` has the remaining bytes of `C..D`, before decoding.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.unqp(C, D) end

---
---Low-level filter to break text into lines with CRLF marker.
---Text is assumed to be in the `normalize` form.
---
---Note: This function only breaks lines that are bigger than
---`length` bytes. The resulting line length does not include the CRLF
---marker.
---
---@param n integer `n` should tell how many bytes are left for the first line of `B`
---@param B? string
---@param length? any # broken into lines of at most `length` bytes (defaults to 76).
---
---@return string A # `A` is a copy of `B`, broken into lines of at most `length` bytes (defaults to 76).
---@return integer m # returns the number of bytes left in the last line of `A`.
---
---😱 [Types](https://github.com/LuaCATS/luasocket/blob/main/library/mime.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/LuaCATS/luasocket/pulls)
function mime.wrp(n, B, length) end

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
img = {}

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class Image
---@field attr string # the image attributes for *LuaTeX*
---@field bbox table # table with 4 boundingbox dimensions `llx`, `lly`, `urx` and `ury` overruling the `pagebox` entry
---@field colordepth number # the number of bits used by the color space
---@field colorspace number # the color space object number
---@field depth number # the image depth for *LuaTeX*
---@field filename string # the image file name
---@field filepath string # the full (expanded) file name of the image
---@field height number # the image height for *LuaTeX*
---@field imagetype string # one of `pdf`, `png`, `jpg`, `jp2` or `jbig2`
---@field index number # the *PDF* image name suffix
---@field objnum number # the *PDF* image object number
---@field page number # the identifier for the requested image page
---@field pagebox string # the requested bounding box, one of `none`, `media`, `crop`, `bleed`, `trim`, `art`
---@field pages number # the total number of available pages
---@field rotation number # the image rotation from included *PDF* file, in multiples of 90 deg.
---@field stream string # the raw stream data for an `/Xobject` `/Form` object
---@field transform number # the image transform, integer number 0..7
---@field orientation number # the (jpeg) image orientation, integer number 1..8 (0 for unset)
---@field width number # the image width for *LuaTeX*
---@field xres number # the horizontal natural image resolution (in \DPI)
---@field xsize number # the natural image width
---@field yres number # the vertical natural image resolution (in \DPI)
---@field ysize number # the natural image height
---@field visiblefilename string # when set, this name will find its way in the *PDF* file as `PTEX` specification; when an empty string is assigned nothing is written to file; otherwise the natural filename is taken
---@field userpassword string # the userpassword needed for opening a *PDF* file
---@field ownerpassword string # the ownerpassword needed for opening a *PDF* file
---@field keepopen boolean # keep the *PDF* file open
---@field nobbox boolean # don't add a boundingbox specification for streams
---@field nolength boolean # don't add length key nor compress for streams
---@field nosize boolean # don't add size fields for streams

---
---Create a userdata object of type “image”.
---
---```
---<image> var = img.new()
---<image> var = img.new(<table> image_spec)
---```
---
---This function creates a userdata object of type “image”. The `image_spec` argument is optional. If it is given, it must be a table, and that
---table must contain a `filename` key. A number of other keys can also be
---useful, these are explained below.
---
---You can either say
---
---```
---a = img.new()
---```
---
---followed by
---
---```
---a.filename = "foo.png"
---```
---
---or you can put the file name (and some or all of the other keys) into a table
---directly, like so:
---
---```
---a = img.new({filename='foo.pdf', page=1})
---```
---
---The generated `<image>` userdata object allows access to a set of
---user-specified values as well as a set of values that are normally filled in
---and updated automatically by *LuaTeX* itself. Some of those are derived from the
---actual image file, others are updated to reflect the *PDF* output status of the
---object.
---
---There is one required user-specified field: the file name (`filename`). It
---can optionally be augmented by the requested image dimensions (`width`,
---`depth`, `height`), user-specified image attributes (`attr`),
---the requested *PDF* page identifier (`page`), the requested boundingbox
---(`pagebox`) for *PDF* inclusion, the requested color space object (`colorspace`).
---
---The function `img.new` does not access the actual image file, it just
---creates the `<image>` userdata object and initializes some memory
---structures. The `<image>` object and its internal structures are
---automatically garbage collected.
---
---Once the image is scanned, all the values in the `<image>` except `width`, `height` and `depth`, become frozen, and you cannot change
---them any more.
---
---You can use `pdf.setignoreunknownimages(1)` (or at the *TeX* end the `pdfvariable` `ignoreunknownimages`) to get around a quit when no known
---image type is found (based on name or preamble). Beware: this will not catch
---invalid images and we cannot guarantee side effects. A zero dimension image is
---still included when requested. No special flags are set. A proper workflow will
---not rely in such a catch but make sure that images are valid.
---
---__Reference:__
---
---* Corresponding C source code: [limglib.c#L75-L102](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/limglib.c#L75-L102)
---
---@param image_spec? ImageSpec
---
---@return Image
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function img.new(image_spec) end

---
---__Reference:__
---
---* Corresponding C source code: [limglib.c#L301-L304](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/limglib.c#L301-L304)
---@return ImageSpec keys
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function img.fields() end

---
---Alias for img.fields()
---
---__Reference:__
---
---* Corresponding C source code: [limglib.c#L301-L304](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/limglib.c#L301-L304)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function img.keys() end

---
---Must be a table, and that table must contain a `filename` key.
---
---__Reference:__
---
---* Corresponding C source code: [luatex-api.h#L506-L532](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/luatex-api.h#L506-L532)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class ImageSpec
---@field attr? string # the image attributes for *LuaTeX*
---@field bbox? table # table with 4 boundingbox dimensions `llx`, `lly`, `urx` and `ury` overruling the `pagebox` entry
---@field colordepth? number # the number of bits used by the color space
---@field colorspace? number # the color space object number
---@field depth? number # the image depth for *LuaTeX*
---@field filename string # the image file name
---@field filepath? string # the full (expanded) file name of the image
---@field height? number # the image height for *LuaTeX*
---@field imagetype? string # one of `pdf`, `png`, `jpg`, `jp2` or `jbig2`
---@field index? number # the *PDF* image name suffix
---@field objnum? number # the *PDF* image object number
---@field page? number # the identifier for the requested image page
---@field pagebox? string # the requested bounding box, one of `none`, `media`, `crop`, `bleed`, `trim`, `art`
---@field pages? number # the total number of available pages
---@field rotation? number # the image rotation from included *PDF* file, in multiples of 90 deg.
---@field stream? string # the raw stream data for an `/Xobject` `/Form` object
---@field transform? number # the image transform, integer number 0..7
---@field orientation? number # the (jpeg) image orientation, integer number 1..8 (0 for unset)
---@field width? number # the image width for *LuaTeX*
---@field xres? number # the horizontal natural image resolution (in \DPI)
---@field xsize? number # the natural image width
---@field yres? number # the vertical natural image resolution (in \DPI)
---@field ysize? number # the natural image height
---@field visiblefilename? string # when set, this name will find its way in the *PDF* file as `PTEX` specification; when an empty string is assigned nothing is written to file; otherwise the natural filename is taken
---@field userpassword? string # the userpassword needed for opening a *PDF* file
---@field ownerpassword? string # the ownerpassword needed for opening a *PDF* file
---@field keepopen? boolean # keep the *PDF* file open
---@field nobbox? boolean # don't add a boundingbox specification for streams
---@field nolength? boolean # don't add length key nor compress for streams
---@field nosize? boolean # don't add size fields for streams

---
---When you say `img.scan(a)` for a new image, the file is scanned, and
---variables such as `xsize`, `ysize`, image `type`, number of
---`pages`, and the resolution are extracted. Each of the `width`, `height`, `depth` fields are set up according to the image dimensions, if
---they were not given an explicit value already. An image file will never be
---scanned more than once for a given image variable. With all subsequent `img.scan(a)` calls only the dimensions are again set up (if they have been
---changed by the user in the meantime).
---
---For ease of use, you can do right-away a
---
---```lua
---<image> a = img.scan { filename = "foo.png" }
---```
---
---without a prior `img.new`.
---
---Nothing is written yet at this point, so you can do `a=img.scan`, retrieve
---the available info like image width and height, and then throw away `a`
---again by saying `a=nil`. In that case no image object will be reserved in
---the PDF, and the used memory will be cleaned up automatically.
---
---__Reference:__
---
---* Corresponding C source code: [limglib.c#L145-L157](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/limglib.c#L145-L157)
---
---@param image Image|ImageSpec
---
---@return Image
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function img.scan(image) end

---
---If you say `a = b`, then both variables point to the same `<image>`
---object. if you want to write out an image with different sizes, you can do
---`b = img.copy(a)`.
---
---Afterwards, `a` and `b` still reference the same actual image
---dictionary, but the dimensions for `b` can now be changed from their
---initial values that were just copies from `a`.
---
---__Reference:__
---
---* Corresponding C source code: [limglib.c#L104-L114](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/limglib.c#L104-L114)
---
---@param image Image|ImageSpec
---
---@return Image
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function img.copy(image) end

---
---Allocate a *PDF* object number and generate a rule node of
---subtype `image` and put it into the output list.
---
---By this the
---image `a` is placed into the page stream, and the image file is written out
---into an image stream object after the shipping of the current page is finished.
---
---Again you can do a terse call like
---
---```
---img.write { filename = "foo.png" }
---```
---
---The `<image>` variable is returned in case you want it for later
---processing. You can also write an object.
---
---__Reference:__
---
---* Corresponding C source code: [limglib.c#L267-L271](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/limglib.c#L267-L271)
---
---@param image Image|ImageSpec
---
---@return Image
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function img.write(image) end

---
---Allocate a *PDF* object number and write the
---image file for image `a` out immediately into the *PDF* file as
---an image stream object (like with `immediate` `\pdfximage`).
---
---The object
---number of the image stream dictionary is then available by the `objnum`
---key. No `pdf_refximage` whatsit node is generated. You will need an
---`img.write(a)` or `img.node(a)` call to let the image appear on the
---page, or reference it by another trick; else you will have a dangling image
---object in the *PDF* file.
---
---Also here you can do a terse call like
---
---```
---a = img.immediatewrite { filename = "foo.png" }
---```
---
---The `<image>` variable is returned and you will most likely need it.
---
---__Reference:__
---
---* Corresponding C source code: [limglib.c#L273-L282](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/limglib.c#L273-L282)
---
---@param image Image|ImageSpec
---
---@return Image
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function img.immediatewrite(image) end

---
---Copie an object from a (*PDF*) image
---file.
---
---This features is experimental and might disappear.
---
---__Reference:__
---
---* Corresponding C source code: [limglib.c#L284-L293](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/limglib.c#L284-L293)
---
---@param image Image|ImageSpec
---@param objnum integer
---
---@return Image
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function img.immediatewriteobject(image, objnum) end

---
---Allocate a *PDF* object number and return a whatsit node of
---subtype `pdf_refximage`, filled with the image parameters `width`,
---`height`, `depth`, and `objnum`.
---
---Also here you can do a terse
---call like:
---
---```
---n = img.node ({ filename = "foo.png" })
---```
---
---This example outputs an image:
---
---```
---node.write(img.node{filename="foo.png"})
---```
---
---__Reference:__
---
---* Corresponding C source code: [limglib.c#L295-L299](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/limglib.c#L295-L299)
---
---@param image Image|ImageSpec
---
---@return Node
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function img.node(image) end

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias ImgType `pdf` | `png` | `jpg` | `jp2` | `jbig2`

---
---Return a list with the supported image file type names, currently
---these are `pdf`, `png`, `jpg`, `jp2` (JPEG 2000), and
---`jbig2`.
---
---```lua
---types = img.types()
---```
---
---__Reference:__
---
---* Corresponding C source code: [limglib.c#L306-L309](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/limglib.c#L306-L309)
---
---@return ImgType[]
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function img.types() end

---
---__Reference:__
---
---* Corresponding C source code: [luatex-api.h#L534-L540](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/luatex-api.h#L534-L540)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias ImgBox `media` | `crop` | `bleed` | `trim` | `art`

---
---Return a list with the supported *PDF* page box names, currently
---these are `media`, `crop`, `bleed`, `trim`, and `art`, all in lowercase.
---
---```lua
---boxes = img.boxes()
---```
---
---__Reference:__
---
---* Corresponding C source code: [limglib.c#L311-L314](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/limglib.c#L311-L314)
---
---@return ImgBox[]
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/img.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function img.boxes() end ---
---This library provides the interface to *LuaTeX*'s structure representing a
---language, and the associated functions.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
lang = {}

---
---The language object is the first argument to most of the other functions in the `lang` library.
---
---__Reference:__
---
---* Corresponding C source code: [texlang.c#L55-62](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lang/texlang.c#L55-62)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@class Language
Language = {}

---
---Create a new language object, with an optional fixed id number.
---
---Create a new userdata object of type `<language>`.
---
---These functions can also be used as if they were object methods, using the colon
---syntax.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L27-L48](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L27-L48)
---
---@param id? integer # Without an argument, the next available internal id number will be assigned to this object. With an argument, an object will be created that links to the internal language with that id number.
---
---@return Language # The language object.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.new(id) end

---
---Return the current internal `language` id number.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L50-L56](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L50-L56)
---
---@param language Language # The language object.
---
---@return integer # The number returned is the internal `language` id number this object refers to.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.id(language) end

---
---Return the current internal `language` id number.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L50-L56](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L50-L56)
---
---@return integer # The number returned is the internal `language` id number this object refers to.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Language:id() end

---
---Add hyphenation exceptions.
---
---```lua
---local l = lang.new()
---lang.hyphenation(l, "man-u-script")
---print(lang.hyphenation(l)) -- man-u-script
---lang.hyphenation(l, "ca-tas-tro-phe")
---print(lang.hyphenation(l)) -- man-u-script ca-tas-tro-phe
---```
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L86-L104](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L86-L104)
---
---@param language Language # The language object.
---@param hyphenation_exceptions string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.hyphenation(language, hyphenation_exceptions) end

---
---Add hyphenation exceptions.
---
---```lua
---local l = lang.new()
---l:hyphenation("man-u-script")
---print(l:hyphenation()) -- man-u-script
---l:hyphenation("ca-tas-tro-phe")
---print(l:hyphenation()) -- man-u-script ca-tas-tro-phe
---```
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L86-L104](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L86-L104)
---
---@param hyphenation_exceptions string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Language:hyphenation(hyphenation_exceptions) end

---
---Get the hyphenation exceptions.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L86-L104](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L86-L104)
---
---@param language Language # The language object.
---
---@return string|nil hyphenation_exceptions
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.hyphenation(language) end

---
---Get the hyphenation exceptions.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L86-L104](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L86-L104)
---
---@return string|nil hyphenation_exceptions
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Language:hyphenation() end

---
---Clear the set of hyphenation exceptions.
---
---@param language Language
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L215-L221](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L215-L221)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.clear_hyphenation(language) end

---
---Clear the exception dictionary (string) for this language.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L223-L241](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L223-L241)
---
---@param language Language # The language object.
---@param o string
---
---@return string n
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.clean(language, o) end

---
---Clear the exception dictionary (string) for this language.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L223-L241](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L223-L241)
---
---@param o string # The language object.
---
---@return string n
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.clean(o) end

---
---Add additional patterns for this language object.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L58-L76](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L58-L76)
---
---@param language Language # The language object.
---@param patterns string # For example `.ab3a .abb2 .ab5erk`
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.patterns(language, patterns) end

---
---Add additional patterns for this language object.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L58-L76](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L58-L76)
---
---@param patterns string # For example `.ab3a .abb2 .ab5erk`
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Language:patterns(patterns) end

---
---Return the current set of patterns.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L58-L76](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L58-L76)
---
---@param language Language # The language object.
---
---@return string|nil pattterns # For example `.ab3a .abb2 .ab5erk`
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.patterns(language) end

---
---Return the current set of patterns.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L58-L76](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L58-L76)
---
---@return string|nil pattterns # For example `.ab3a .abb2 .ab5erk`
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function Language:patterns() end

---
---Clear the set of hyphenation patterns.
---
---Clear the pattern dictionary for a language.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L78-L84](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L78-L84)
---
---@param language Language
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.clear_patterns(language) end

---
---Set the value of the *TeX* parameter
---`\hyphenationmin`.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L199-L213](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L199-L213)
---
---@param language Language
---@param min integer
function lang.hyphenationmin(language, min) end

---
---Get the value of the *TeX* parameter
---`\hyphenationmin`.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L199-L213](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L199-L213)
---
---@param language Language # The language object.
---
---@return integer min
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.hyphenationmin(language) end

---
---Set the “pre-break” hyphen characters for implicit hyphenation in this language.
---
---The
---intial values are decimal 45 (hyphen) and decimal 0 (indicating emptiness).
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L106-L120](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L106-L120)
---
---@param language Language # The language object.
---@param char integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.prehyphenchar(language, char) end

---
---Get the “pre-break” hyphen characters for implicit hyphenation in this language.
---
---The
---intial values are decimal 45 (hyphen) and decimal 0 (indicating emptiness).
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L106-L120](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L106-L120)
---
---@param language Language # The language object.
---
---@return integer char
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.prehyphenchar(language) end

---
---Set the “post-break” hyphen characters for implicit hyphenation in this language.
---
---The
---intial values are decimal 45 (hyphen) and decimal 0 (indicating emptiness).
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L122-L136](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L122-L136)
---
---@param language Language # The language object.
---@param char integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.posthyphenchar(language, char) end

---
---Get the “post-break” hyphen characters for implicit hyphenation in this language.
---
---The
---intial values are decimal 45 (hyphen) and decimal 0 (indicating emptiness).
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L122-L136](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L122-L136)
---
---@param language Language # The language object.
---
---@return integer char
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.posthyphenchar(language) end

---
---Set the “pre-break” hyphen
---characters for explicit hyphenation in this language.
---
---The hyphen
---character is initially
---decimal `0` (indicating emptiness).
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L138-L152](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L138-L152)
---
---@param language Language # The language object.
---@param char integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.preexhyphenchar(language, char) end

---
---Get the “pre-break” hyphen
---characters for explicit hyphenation in this language.
---
---The hyphen
---character is initially
---decimal `0` (indicating emptiness).
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L138-L152](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L138-L152)
---
---@param language Language # The language object.
---
---@return integer char
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.preexhyphenchar(language) end

---
---Set the “post-break” hyphen
---characters for explicit hyphenation in this language.
---
---The hyphen
---character is initially
---decimal `0` (indicating emptiness).
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L183-L197](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L183-L197)
---
---@param language Language # The language object.
---@param char integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.postexhyphenchar(language, char) end

---
---Get the “post-break” hyphen
---characters for explicit hyphenation in this language.
---
---The hyphen
---character is initially
---decimal `0` (indicating emptiness).
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L183-L197](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L183-L197)
---
---@param language Language # The language object.
---
---@return integer char
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.postexhyphenchar(language) end

---
---Hyphenate a node list.
---
---Insert hyphenation points (discretionary nodes) in a node list.
---
---If
---`tail` is given as argument, processing stops on that node. Currently,
---`success` is always true if `head` (and `tail`, if specified)
---are proper nodes, regardless of possible other errors.
---
---Hyphenation works only on “characters”, a special subtype of all the glyph
---nodes with the node subtype having the value `1`. Glyph modes with
---different subtypes are not processed.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L243-L261](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L243-L261)
---
---@param head Node
---@param tail? Node
---
---@return boolean success
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.hyphenate(head, tail) end

---
---Set hj codes.
---
---When you set a hjcode the current sets get initialized unless the set was already
---initialized due to `\savinghyphcodes` being larger than zero.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L154-L169](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L154-L169)
---
---@param language Language # The language object.
---@param char integer
---@param used_char integer
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.sethjcode(language, char, used_char) end

---
---Query hj codes.
---
---__Reference:__
---
---* Corresponding C source code: [llanglib.c#L171-L181](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/llanglib.c#L171-L181)
---
---@param language Language # The language object.
---@param char number
---
---@return integer used_char
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/lang.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function lang.gethjcode(language, char) end ---
---# Extra `os` library functions
---
---The `os` library has a few extra functions and variables: `os.selfdir`, `os.exec`, `os.spawn`, `os.setenv`,
---`os.env`, `os.gettimeofday`, `os.times`, `os.tmpdir`, `os.type`, `os.name` and `os.uname`,
---that we will discuss here.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
os = {}

---
---A variable that holds the directory path of the
---actual executable.
---
---For example: `\directlua {tex.sprint(os.selfdir)`}.
---
---__Reference:__
---
---* Corresponding C source code: [luainit.c#L169-L186](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/luainit.c#L169-L186)
---
---For example `/usr/local/texlive/bin/x86_64-linux`
---@string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
os.selfdir = ""

---
---A variation on `os.execute`.
---
---Here
---`commandline` can be either a single string or a single table.
---
---* If the argument is a table *LuaTeX* first checks if there is a value at
---  integer index zero. If there is, this is the command to be executed.
---  Otherwise, it will use the value at integer index one. If neither are
---  present, nothing at all happens.
---
---* The set of consecutive values starting at integer 1 in the table are the
---  arguments that are passed on to the command (the value at index 1 becomes
---  `arg[0]`). The command is searched for in the execution path, so
---  there is normally no need to pass on a fully qualified path name.
---
---* If the argument is a string, then it is automatically converted into a
---  table by splitting on whitespace. In this case, it is impossible for the
---  command and first argument to differ from each other.
---
---* In the string argument format, whitespace can be protected by putting
---  (part of) an argument inside single or double quotes. One layer of quotes
---  is interpreted by *LuaTeX*, and all occurrences of `\"`, `\'`
---  or `\\` within the quoted text are unescaped. In the table format,
---  there is no string handling taking place.
---
---This function normally does not return control back to the *Lua* script: the
---command will replace the current process. However, it will return the two
---values `nil` and `error` if there was a problem while
---attempting to execute the command.
---
---On *MS Windows*, the current process is actually kept in memory until after the
---execution of the command has finished. This prevents crashes in situations
---where *TeXLua* scripts are run inside integrated *TeX* environments.
---
---The original reason for this command is that it cleans out the current
---process before starting the new one, making it especially useful for use in
---*TeX*LUA.
---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L429-L501](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/loslibext.c#L429-L501)
---
---@see os.execute
---
---@param commandline string|table
---
---@return integer|nil exit_status
---@return string|nil error
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function os.exec(commandline) end

---
---A returning version of `os.exec`,
---with otherwise identical calling conventions.
---
---If the command ran ok, then the return value is the exit status of the
---command. Otherwise, it will return the two values `nil` and `error`.
---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L510-L596](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/loslibext.c#L510-L596)
---
---@see os.execute
---
---@param commandline string|table
---
---@return integer|nil exit_status
---@return string|nil error
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function os.spawn(commandline) end

---
---This function is similar to `io.popen`
---but with a preliminary check of the command by mean of `kpse.check_permission`
---
---If the command ran ok, then the return value is the same of `io.popen`;
---otherwise it will return the two values `nil` and `error`.
---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L1093-1121](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/5650c067de62cb7d4aaca44f30c8e9115c51bfc6/source/texk/web2c/luatexdir/lua/loslibext.c#L1093-1121)
---
---@see io.popen
---@see kpse.check_permission
---
---@param commandline string
---@param mode 'r'|'w'
---
---@return integer|nil exit_status
---@return string|nil error
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function os.kpsepopen(commandline, mode) end

---
---Set a variable in the environment.
---
---Passing
---`nil` instead of a value string will remove the variable.
---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L600-L627](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/loslibext.c#L600-L627)
---
---@param key string
---@param value string|nil
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function os.setenv(key, value) end

---
---A hash table containing a dump of the variables and
---values in the process environment at the start of the run.
---
---It is writeable,
---but the actual environment is `not` updated automatically.
---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L630-L661](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/loslibext.c#L630-L661)
---
---```lua
---{
---  HOME = "/home/user",
---  HOSTNAME = "myhost",
---  INFOPATH = "/usr/local/texlive/texmf-dist/doc/info",
---  LANG = "en_US.UTF-8",
---  LANGUAGE = "en_US:en",
---  OS = "Linux",
---  PAPERSIZE = "a4",
---  PATH = "/bin:/sbin",
---  TERM = "xterm-256color",
---  USER = "user",
---  USERNAME = "user",
---  ...
---}
---```
---@type table<string, string>
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
os.env = {}

---
---Suspend execution for second intervals.
---
---__Example:__
---
---```lua
---local start_time = os.gettimeofday()
---os.sleep(1)
---local end_time = os.gettimeofday()
---assert(end_time - start_time > 1)
---
---os.sleep(1) -- Sleep 1 second
---os.sleep(1, 10) -- Sleep 1 decisecond
---os.sleep(1, 100) -- Sleep 1 centisecond
---os.sleep(1, 1000) -- Sleep 1 millisecond
---```
---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L663-L673](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/loslibext.c#L663-L673)
---* Corresponding Lualibs source code: [lualibs-os.lua#L580-L587](https://github.com/latex3/lualibs/blob/26fe094de645fdee79f65d9fc93040a53cb97272/lualibs-os.lua#L580-L587)
---
---@param interval number # By default seconds.
---@param unit? number # Sleep `interval / unit` seconds.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/lualibs/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function os.sleep(interval, unit) end

---
---Return the current “UNIX time”, but as a
---float.
---
---This function is not available on the *SunOS* platforms, so do not use
---this function for portable documents.
---
---__Example:__
---
---```lua
---local time = os.gettimeofday()
---assert(time > 1682153121.3217, time)
---```
---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L905-L927](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/loslibext.c#L905-L927)
---
---@return number # for example `1673390071.0893`
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function os.gettimeofday() end

---
---@class Times
---@field cstime number # system time of children
---@field cutime number # user time of children
---@field stime number # system time
---@field utime number # user time

---
---Return the current process times according to the
---UNIX C library function “times”.
---
---This function is not available on
---the MS Windows and *SunOS* platforms, so do not use this function for
---portable documents.
---
---```lua
---{
---  cstime = 0.0,
---  cutime = 0.0,
---  stime = 0.01,
---  utime = 0.12
---}
---```
---
---__Example:__
---
---```lua
---local times = os.times()
---assert(type(times.cstime) == 'number')
---assert(type(times.cutime) == 'number')
---assert(type(times.stime) == 'number')
---assert(type(times.utime) == 'number')
---```
---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L872-L894](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/loslibext.c#L872-L894)
---
---@return Times
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function os.times() end

---
---Create a directory in the “current directory”
---with the name `luatex.XXXXXX` where the `X`-es are replaced by a
---unique string.
---
---The function also returns this string, so you can `lfs.chdir()` into it, or `nil` if it failed to create the directory.
---The user is responsible for cleaning up at the end of the run, it does not
---happen automatically.
---
---__Example:__
---
---```lua
---lfs.chdir('/tmp')
---local dir, err = os.tmpdir()
---assert(type(dir) == 'string')
---assert(lfs.isdir('/tmp/' .. dir))
---
---dir, err = os.tmpdir('tmp.XXXXXX')
---assert(type(dir) == 'string')
---assert(lfs.isdir('/tmp/' .. dir))
---
---dir, err = os.tmpdir('tmp.X')
---assert(dir == nil)
---assert(err == 'Invalid argument to os.tmpdir()', err)
---
---dir, err = os.tmpdir('tmp.XXXXXXX_suffix')
---assert(dir == nil)
---assert(err == 'Invalid argument to os.tmpdir()', err)
---```
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-lua.tex#L531-L538](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-lua.tex#L531-L538)
---* Corresponding C source code: [loslibext.c#L971-L997](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/loslibext.c#L971-L997)
---
---@param template? string # for example `luatex.XXXXXX`
---
---@return string|nil
---@return nil|string error
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function os.tmpdir(template) end

---
---A string that gives a global indication of the class of
---operating system.
---
---The possible values are currently `windows`, `unix`, and `msdos` (you are unlikely to find this value “in the
---wild”).
---
---__Example:__
---
---```lua
---assert(os.type == 'unix')
---```
---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L40-L91](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/loslibext.c#L40-L91)
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
os.type = ""

---
---A string that gives a more precise indication of the
---operating system.
---
---These possible values are not yet fixed, and for `os.type` values `windows` and `msdos`, the `os.name`
---values are simply `windows` and `msdos`
---
---The list for the type `unix` is more precise: `linux`, `freebsd`, `kfreebsd`, `cygwin`, `openbsd`, `solaris`, `sunos` (pre-solaris), `hpux`, `irix`, `macosx`, `gnu` (hurd), `bsd` (unknown, but BSD-like), `sysv` (unknown, but SYSV-like), `generic` (unknown).
---
---__Example:__
---
---```lua
---assert(os.name == 'linux')
---```
---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L40-L91](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/loslibext.c#L40-L91)
---
---@type string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
os.name = ""

---
---@class Uname
---@field sysname string # for example `x86_64`
---@field machine string # for example `zotac`
---@field release string # for example `5.15.0-57-generic`
---@field version string # for example `#63-Ubuntu SMP Thu Nov 24 13:43:17 UTC 2022`
---@field nodename string # for example `mypc`

---
---Return a table with specific operating system
---information acquired at runtime.
---
---The keys in the returned table are all
---string values, and their names are: `sysname`, `machine`, `release`, `version`, and `nodename`.
---
---__Example:__
---
---```lua
---local uname = os.uname()
---assert(type(uname.machine) == 'string')
---assert(type(uname.nodename) == 'string')
---assert(type(uname.release) == 'string')
---assert(type(uname.sysname) == 'string')
---assert(type(uname.version) == 'string')
---```
---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L849-L868](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/loslibext.c#L849-L868)
---
---```lua
---{
---  machine = "x86_64",
---  nodename = "mypc",
---  release = "5.15.0-57-generic",
---  sysname = "Linux",
---  version = "#63-Ubuntu SMP Thu Nov 24 13:43:17 UTC 2022"
---}
---```
---
---@return Uname
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function os.uname() end

---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L1157-1170](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/b6437de2fa62bb25e17f162e624e8d815fc4d88b/source/texk/web2c/luatexdir/lua/loslibext.c#L1157-1170)
---
---@see socket.sleep
---
---@param time number
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function os.socketsleep(time) end

---
---__Reference:__
---
---* Corresponding C source code: [loslibext.c#L1171-1179](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/b6437de2fa62bb25e17f162e624e8d815fc4d88b/source/texk/web2c/luatexdir/lua/loslibext.c#L1171-1179)
---
---@see socket.gettime
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/os.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function os.socketgettime() end
---
---The `texio` library takes care of the low-level I/O interface: writing to the log file
---and/or console.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
texio = {}

---
---The optional `target` can be one of three possibilities: `term`, `log` or `term and log`.
---@alias WriteTarget
---|'term'
---|'log'
---|'term and log'

---
---Write all given strings to a target.
---
---__Reference:__
---
---* Corresponding C source code: [ltexiolib.c#L119-L126](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexiolib.c#L119-L126)
---
---@param target WriteTarget
---@param ... string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function texio.write(target, ...) end

---
---Write all given strings to the same
---location(s) *TeX* writes messages to at this moment.
---
---If `batchmode` is in
---effect, it writes only to the log, otherwise it writes to the log and the
---terminal.
---
---__Reference:__
---
---* Corresponding C source code: [ltexiolib.c#L119-L126](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexiolib.c#L119-L126)
---
---@param ... string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function texio.write(...) end

---
---Write all given strings to a target.
---
---It makes sure that the given
---strings will appear at the beginning of a new line.
---
---__Reference:__
---
---* Corresponding C source code: [ltexiolib.c#L128-L135](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexiolib.c#L128-L135)
---
---@param target WriteTarget
---@param ... string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function texio.write_nl(target, ...) end

---
---Write all given strings to the same
---location(s) *TeX* writes messages to at this moment. It makes sure that the given
---strings will appear at the beginning of a new line.
---
---Note: If several strings are given, and if the first of these strings is or might
---be one of the targets above, the `target` must be specified explicitly to
---prevent *Lua* from interpreting the first string as the target.
---
---__Reference:__
---
---* Corresponding C source code: [ltexiolib.c#L128-L135](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexiolib.c#L128-L135)
---
---@param ... string
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function texio.write_nl(...) end

---
---Disable `^^` escaping of control characters.
---
---__Reference:__
---
---* Corresponding C source code: [ltexiolib.c#L148-L156](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexiolib.c#L148-L156)
---
---@param escape_controls boolean
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function texio.setescape(escape_controls) end

---
---Force a jump back to *TeX*.
---
---This function should be used with care. It acts as `endinput` but at
---the *Lua* end. Normally
---*Lua* will just collect prints and at the end bump an input level and flush these
---prints. This function can help you stay at the current level but you need to know
---what you're doing (or more precise: what *TeX* is doing with input).
---
---__Reference:__
---
---* Corresponding C source code: [ltexiolib.c#L158-L171](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/ltexiolib.c#L158-L171)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/texio.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function texio.closeinput() end

---
---Current version
---
---2025/07/24 v0.2.0
------
---This library has functions that register, find and list callbacks. Callbacks are
---*Lua* functions that are called in well defined places. There are two kind of
---callbacks: those that mix with existing functionality, and those that (when
---enabled) replace functionality. In mosty cases the second category is expected to
---behave similar to the built in functionality because in a next step specific
---data is expected. For instance, you can replace the hyphenation routine. The
---function gets a list that can be hyphenated (or not). The final list should be
---valid and is (normally) used for constructing a paragraph. Another function can
---replace the ligature builder and/or kerner. Doing something else is possible
---but in the end might not give the user the expected outcome.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-callbacks.tex#L17-L26](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-callbacks.tex#L17-L26)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
callback = {}

---
---__Reference:__
---
---* Corresponding C source code: [lcallbacklib.c#L33-L90](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lcallbacklib.c#L33-L90)
---
---Source: `callback.list()`
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias CallbackName
---|"append_to_vlist_filter"
---|"build_page_insert"
---|"buildpage_filter"
---|"call_edit"
---|"contribute_filter"
---|"define_font"
---|"find_cidmap_file"
---|"find_data_file"
---|"find_enc_file"
---|"find_font_file"
---|"find_format_file"
---|"find_image_file"
---|"find_map_file"
---|"find_opentype_file"
---|"find_output_file"
---|"find_pk_file"
---|"find_read_file"
---|"find_truetype_file"
---|"find_type1_file"
---|"find_vf_file"
---|"find_write_file"
---|"finish_pdffile"
---|"finish_pdfpage"
---|"hpack_quality"
---|"hyphenate"
---|"input_level_string"
---|"insert_local_par"
---|"kerning"
---|"ligaturing"
---|"linebreak_filter"
---|"make_extensible"
---|"mlist_to_hlist"
---|"new_graf"
---|"open_read_file"
---|"page_order_index"
---|"post_linebreak_filter"
---|"pre_dump
---|"pre_linebreak_filter"
---|"pre_output_filter"
---|"process_input_buffer"
---|"process_jobname"
---|"process_output_buffer"
---|"process_pdf_image_content"
---|"process_rule"
---|"provide_charproc_data"
---|"read_cidmap_file"
---|"read_data_file"
---|"read_enc_file"
---|"read_font_file"
---|"read_map_file"
---|"read_opentype_file"
---|"read_pk_file"
---|"read_truetype_file"
---|"read_type1_file"
---|"read_vf_file"
---|"show_error_hook"
---|"show_error_message"
---|"show_lua_error_hook"
---|"show_warning_message"
---|"start_file"
---|"start_page_number"
---|"start_run"
---|"stop_file"
---|"stop_page_number"
---|"stop_run"
---|"vpack_filter"
---|"vpack_quality"
---|"wrapup_run

---
---__Reference:__
---
---* Corresponding C source code: [](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lcallbacklib.c#L198-L208)
---
-- #define CALLBACK_BOOLEAN        'b'
-- #define CALLBACK_INTEGER        'd'
-- #define CALLBACK_FLOAT          'f'
-- #define CALLBACK_LINE           'l'
-- #define CALLBACK_STRNUMBER      's'
-- #define CALLBACK_STRING         'S'
-- #define CALLBACK_RESULT         'R' /* a string but nil is also ok */
-- #define CALLBACK_CHARNUM        'c'
-- #define CALLBACK_LSTRING        'L'
-- #define CALLBACK_NODE           'N'
-- #define CALLBACK_DIR            'D'

---
---Register a callback. Passing `nil` removes an existing callback. Returns `nil`, `error` on failure.
---
---The first thing you need to do is registering a callback:
---
---Here the `callback_name` is a predefined callback name, see below.
---
---*LuaTeX* internalizes the callback function in such a way that it does not matter
---if you redefine a function accidentally.
---
---Callback assignments are always global. You can use the special value `nil`
---instead of a function for clearing the callback.
---
---For some minor speed gain, you can assign the boolean `false` to the
---non-file related callbacks, doing so will prevent *LuaTeX* from executing
---whatever it would execute by default (when no callback function is registered at
---all). Be warned: this may cause all sorts of grief unless you know `exactly` what you are doing!
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-callbacks.tex#L28-L54](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-callbacks.tex#L28-L54)
---* Corresponding C source code: [lcallbacklib.c#L517-L557](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lcallbacklib.c#L517-L557)
---
---@param callback_name CallbackName
---@param func function|nil|false
---
---@return integer|nil id # The function returns the internal `id` of the callback or `nil`, if the callback could not be registered.
---@return string error # In the latter case, `error` contains an error message, otherwise it is `nil`. The function returns `No such callback exists.` if a wrong callback name was specified.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function callback.register(callback_name, func) end

---
---Produce a list of all known callback names.
---
---The keys in the table are the known callback names, the value is a boolean where
---`true` means that the callback is currently set (active).
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-callbacks.tex#L56-L62](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-callbacks.tex#L56-L62)
---* Corresponding C source code: [lcallbacklib.c#L584-L599](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lcallbacklib.c#L584-L599)
---
---@return table<string, boolean> info
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function callback.list() end

---
---If the callback is not set, `find` returns `nil`.
---
---__Reference:__
---
---* Source file of the `LuaTeX` manual: [luatex-callbacks.tex#L64-L68](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/manual/luatex-callbacks.tex#L64-L68)
---* Corresponding C source code: [lcallbacklib.c#L559-L582](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/lua/lcallbacklib.c#L559-L582)
---
---@param callback_name CallbackName
---
---@return function|nil f
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
function callback.find(callback_name) end

---
---Find a file for `input` (0) or `openin` (higher integers).
---
---Arguments:
---
---__id_number__
---
---This number is zero for the log or `input` files. For *TeX*'s `read`
---or `write` the number is incremented by one, so `\read0` becomes 1.
---
---__asked_name__
---
---This is the user-supplied filename, as found by `input`, `openin`
---or `openout`.
---
---Return value:
---
---__actual_name__
---
---This is the filename used. For the very first file that is read in by *TeX*, you
---have to make sure you return an `actual_name` that has an extension and
---that is suitable for use as `jobname`. If you don't, you will have to
---manually fix the name of the log file and output file after *LuaTeX* is finished,
---and an eventual format filename will become mangled. That is because these file
---names depend on the jobname.
---
---You have to return `nil` if the file cannot be found.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindReadFileCallback fun(id_number: integer, asked_name: string): actual_name: string|nil

---
---Find a file for writing to the log file (0) or with `write` (higher integers).
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindWriteFileCallback fun(id_number: integer, asked_name: string): actual_name: string|nil

---
---Find a font metrics file.
---
---The `asked_name` is an *OTF* or *TFM* font metrics file.
---
---Return `nil` if the file cannot be found.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindFontFileCallback fun(asked_name: string): actual_name: string|nil

---
---The `asked_name` is the *PDF* or *DVI* file for writing.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindOutputFileCallback fun(asked_name: string): actual_name: string|nil

---
---Find the format file.
---
---The `asked_name` is a format file for reading (the format file for writing
---is always opened in the current directory).
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindFormatFileCallback fun(asked_name: string): actual_name: string|nil

---
---Find a VF (virtual font) file.
---
---Like `find_font_file`, but for virtual fonts. This applies to both *ALEPH's
---*OVF* files and traditional Knuthian *VF* files.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindVfFileCallback fun(asked_name: string): actual_name: string|nil

---
---Like `find_font_file`, but for map files.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindMapFileCallback fun(asked_name: string): actual_name: string|nil

---
---Find a font encoding file.
---
---Like `find_font_file`, but for enc files.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindEncFileCallback fun(asked_name: string): actual_name: string|nil

---
---Find a PK font bitmap file.
---
---Like `find_font_file`, but for pk bitmap files. This callback takes two
---arguments: `name` and `dpi`. In your callback you can decide to
---look for:
---
---```
---<base res>dpi/<fontname>.<actual res>pk
---```
---
---but other strategies are possible. It is up to you to find a “reasonable”
---bitmap file to go with that specification.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindPkFileCallback fun(asked_name: string, dpi: integer): actual_name: string|nil

---
---Find an input data file for PDF attachment.
---
---Like `find_font_file`, but for embedded files (`\pdfobj file '...'`).
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindDataFileCallback fun(asked_name: string): actual_name: string|nil

---
---Find an OpenType font file.
---
---Like `find_font_file`, but for *OpenType* font files.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindOpentypeFileCallback fun(asked_name: string): actual_name: string|nil

---
---Find a TrueType font file.
---
---The `asked_name` is a font file. This callback is called while *LuaTeX* is
---building its internal list of needed font files, so the actual timing may
---surprise you. Your return value is later fed back into the matching `read_file` callback.
---
---Strangely enough, `find_type1_file` is also used for *OpenType* (OTF)
---fonts.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindTruetypeFileCallback fun(asked_name: string): actual_name: string|nil

---
---Find an Type1 (PostScript) font file.
---
---The `asked_name` is a font file. This callback is called while *LuaTeX* is
---building its internal list of needed font files, so the actual timing may
---surprise you. Your return value is later fed back into the matching `read_file` callback.
---
---Strangely enough, `find_type1_file` is also used for *OpenType* (OTF)
---fonts.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindType1FileCallback fun(asked_name: string): actual_name: string|nil

---
---Find an image file for inclusion.
---
---The `asked_name` is an image file. Your return value is used to open a file
---from the hard disk, so make sure you return something that is considered the name
---of a valid file by your operating system.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias FindImageFileCallback fun(asked_name: string): actual_name: string|nil

---
---*LuaTeX* never looks at the rest of the table, so you can use it to store your
---private per-file data. Both the callback functions will receive the table as
---their only argument.
---
---@class OpenReadFileEnv
local OpenReadFileEnv = {}

---
---Read a line from a file opened with the `open_read_file` callback. The argument is the return value from `open_read_file`
---
---*LuaTeX* will run this function whenever it needs a new input line from the file.
---
---Your function should return either a string or `nil`. The value `nil`
---signals that the end of file has occurred, and will make *TeX* call the optional
---@param env OpenReadFileEnv
---
---@return string|nil line
function OpenReadFileEnv.reader(env) end

---
---Close a file opened with the `open_read_file` callback. The argument is the return value from the `open_read_file`.
---
---*LuaTeX* will run this optional function when it decides to close the file.
---
---@param env OpenReadFileEnv
---
function OpenReadFileEnv.close(env) end

---
---Open a file for reading. The returned table should define key functions for  `reader` and `close`.
---
---The filename returned by a previous `find_read_file` or the return value of
---`kpse.find_file()` if there was no such callback defined.
---
---@alias OpenReadFileCallback fun(file_name: string): OpenReadFileEnv

---
---callback for the loading of binary data files.
---The `name` will normally be a full path name as it is returned by either
---one of the file discovery callbacks or the internal version of `kpse.find_file()`.
---
---__success__
---
---Return `false` when a fatal error occurred (e.g.\ when the file cannot be
---found, after all).
---
---__data__
---
---The bytes comprising the file.
---
---__data_size__
---
---The length of the `data`, in bytes.
---
---Return an empty string and zero if the file was found but there was a
---reading problem.
---@alias ReadFileCallback fun(file_name: string): success: boolean, data: string, data_size: integer

---
---Read a TFM metrics file. Return `true`,  the data, and the data length for success, `false` otherwise
---
---ofm or tfm files
---@alias ReadFontFileCallback ReadFileCallback

---
---Read a VF metrics file.
---
---virtual fonts
---@alias ReadVfFileCallback ReadFileCallback

---
---Read a font map file.
---
--- map files
---@alias ReadMapFileCallback ReadFileCallback

---
---Read a font encoding file.
---
---encoding files
---@alias ReadEncFileCallback ReadFileCallback

---
---Read a font bitmap PK file.
---
---pk bitmap files
---@alias ReadPkFileCallback ReadFileCallback

---
---Read a data file.
---
---embedded files (as is possible with *PDF* objects)
---@alias ReadDataFileCallback ReadFileCallback

---
---Read a TrueType font.
---*TrueType* font files
---@alias ReadTruetypeFileCallback ReadFileCallback

---
---Read a Type1 font.
---
---*Type1* font files
---@alias ReadType1FileCallback ReadFileCallback

---
---Read an OpenType font.
---
---*OpenType* font files
---@alias ReadOpentypeFileCallback ReadFileCallback

---
---Modify the encoding of the input buffer.
---
---This callback allows you to change the contents of the line input buffer just
---before *LuaTeX* actually starts looking at it.
---
---If you return `nil`, *LuaTeX* will pretend like your callback never
---happened. You can gain a small amount of processing time from that. This callback
---does not replace any internal code.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias ProcessInputBufferCallback fun(buffer: string): string|nil

---
---Modify the encoding of the output buffer.
---
---This callback allows you to change the contents of the line output buffer just
---before *LuaTeX* actually starts writing it to a file as the result of a `write` command. It is only called for output to an actual file (that is,
---excluding the log, the terminal, and so called `write` 18 calls).
---
---If you return `nil`, *LuaTeX* will pretend like your callback never
---happened. You can gain a small amount of processing time from that. This callback
---does not replace any internal code.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias ProcessOutputBufferCallback fun(buffer: string): adjusted_buffer: string|nil

---
---This callback allows you to change the jobname given by `jobname` in *TeX*
---and `tex.jobname` in Lua. It does not affect the internal job name or the
---name of the output or log files.
---
---The only argument is the actual job name; you should not use `tex.jobname`
---inside this function or infinite recursion may occur. If you return `nil`,
---*LuaTeX* will pretend your callback never happened. This callback does not
---replace any internal code.
---
---__Reference:__
---
---* Corresponding C source code: [textoken.c#L2511](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/textoken.c#L2511)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias ProcessJobnameCallback fun(jobname: string): string|nil

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias ContributeFilterExtrainfo
---|'pre_box' # interline material is being added
---|'pre_adjust' # `vadjust` material is being added
---|'box' # a typeset box is being added (always called)
---|'adjust' # `vadjust` material is being added

---
---This callback is called when *LuaTeX* adds contents to list:
---
---The string reports the group code. From this you can deduce from
---what list you can give a treat.
---
---__Reference:__
---
---* Corresponding C source code: [luatexcallbackids.h#L163](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/luatexcallbackids.h#L163)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias ContributeFilterCallback fun(extrainfo: ContributeFilterExtrainfo)

---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias BuildpageFilterExtrainfo
---|'alignment' # a (partial) alignment is being added
---|'after_output' # an output routine has just finished
---|'new_graf' # the beginning of a new paragraph
---|'vmode_par' # `par` was found in vertical mode
---|'hmode_par' # `par` was found in horizontal mode
---|'insert' # an insert is added
---|'penalty' # a penalty (in vertical mode)
---|'before_display' # immediately before a display starts
---|'after_display' # a display is finished
---|'end' # *LuaTeX* is terminating (it's all over)

---
---Process objects as they are added to the main vertical list. The string argument gives some context.
---
---This callback is called whenever *LuaTeX* is ready to move stuff to the main
---vertical list. You can use this callback to do specialized manipulation of the
---page building stage like imposition or column balancing.
---
---The string `extrainfo` gives some additional information about what *TeX*'s
---state is with respect to the “current page”.
---
---__Reference:__
---
---* Corresponding C source code: [align.c#L1281](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/align.c#L1281)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias BuildpageFilterCallback fun(extrainfo: ContributeFilterExtrainfo)

---
---This callback is called when the pagebuilder adds an insert. There is not much
---control over this mechanism but this callback permits some last minute
---manipulations of the spacing before an insert, something that might be handy when
---for instance multiple inserts (types) are appended in a row.
---
---__Reference:__
---
---* `n`  the insert class
---* `i`  the order of the insert
---
---The return value is a number indicating the skip register to use for the
---prepended spacing. This permits for instance a different top space (when `i` equals one) and intermediate space (when `i` is larger than one). Of
---course you can mess with the insert box but you need to make sure that *LuaTeX*
---is happy afterwards.
---
---__Reference:__
---
---* Corresponding C source code: [buildpage.c#L507](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/buildpage.c#L507)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias BuildPageInsertCallback fun(n: integer, i: integer): integer

---
---The string called `groupcode` identifies the nodelist's context within
---*TeX*'s processing. The range of possibilities is given in the table below, but
---not all of those can actually appear in `pre_linebreak_filter`, some are
---for the `hpack_filter` and `vpack_filter` callbacks that will be
---explained in the next two paragraphs.
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias PreLinebreakFilterGroupCode
---|'' # main vertical list
---|'hbox' # hbox` in horizontal mode
---|'adjusted_hbox' #hbox` in vertical mode
---|'vbox' # vbox`
---|'vtop' # vtop' #
---|'align' # halign` or `valign`
---|'disc' # discretionaries
---|'insert' # packaging an insert
---|'vcenter' # vcenter`
---|'local_box' # localleftbox` or `localrightbox`
---|'split_off' # top of a `vsplit`
---|'split_keep' # remainder of a `vsplit`
---|'align_set' # alignment cell
---|'fin_row' # alignment row

---
---As for all the callbacks that deal with nodes, the return value can be one of
---three things:
---
---__Reference:__
---
---* boolean `true` signals successful processing
---* `<node>` signals that the “head” node should be replaced by the
---  returned node
---* boolean `false` signals that the “head” node list should be
---  ignored and flushed from memory
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias NodeCallbackReturn true|false|Node

---
---Alter a node list before linebreaking takes place. The string argument gives some context.
---
---This callback is called just before *LuaTeX* starts converting a list of nodes
---into a stack of `hbox`es, after the addition of `parfillskip`.
---
---```lua
------@type PreLinebreakFilter
---function(head, groupcode)
---  --- true|false|node
---  return true
---end
---```
---
---This callback does not replace any internal code.
---
---__Reference:__
---
---* Corresponding C source code: [linebreak.c#L124](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/source/texk/web2c/luatexdir/tex/linebreak.c#L124)
---
---😱 [Types](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/blob/main/library/luatex/callback.lua) incomplete or incorrect? 🙏 [Please contribute!](https://github.com/Josef-Friedrich/LuaTeX_Lua-API/pulls)
---@alias PreLinebreakFilterCallback fun(head: Node, groupcode: PreLinebreakFilterGroupCode): NodeCallbackReturn

---
---Override the linebreaking algorithm. The boolean is `true` if this is a pre-display break.
---
---This callback replaces *LuaTeX*'s line breaking algorithm.
---
---The returned node is the head of the list that will be added to the main vertical
---list, the boolean argument is true if this paragraph is interrupted by a
---following math display.
---
---If you return something that is not a `<node>`, *LuaTeX* will apply the
---internal linebreak algorithm on the list that starts at `<head>`.
---Otherwise, the `<node>` you return is supposed to be the head of a list of
---nodes that are all allowed in vertical mode, and at least one of those has to
---represent a hbox. Failure to do so will result in a fatal error.
---
---Setting this callback to `false` is possible, but dangerous, because it is
---possible you will end up in an unfixable “deadcycles loop”.
---
---__Reference:__
---
---* Corresponding C source code: [luanode.c#L125](https://gitlab.lisn.upsaclay.fr/texlive/luatex/-/blob/f52b099f3e01d53dc03b315e1909245c3d5418d3/sour