/*
 * Copyright (C) 2013-2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <wtf/DataLog.h>
#include <wtf/MonotonicTime.h>
#include <wtf/Vector.h>
#include <wtf/text/StringHasherInlines.h>


namespace TestWebKitAPI {

static bool dumpHashingSpeedComparison = false;

TEST(WTF, StringHasher)
{
    auto generateLatin1Array = [&](size_t size) {
        auto array = std::unique_ptr<Latin1Character[]>(new Latin1Character[size]);
        for (size_t i = 0; i < size; i++)
            array[i] = i;
        return array;
    };

    auto generateUTF16Array = [&](size_t size) {
        auto array = std::unique_ptr<char16_t[]>(new char16_t[size]);
        for (size_t i = 0; i < size; i++)
            array[i] = i;
        return array;
    };

    StringHasher hash;
    unsigned max8Bit = std::numeric_limits<uint8_t>::max();
    for (size_t size = 0; size <= max8Bit; size++) {
        std::unique_ptr<const Latin1Character[]> arr1 = generateLatin1Array(size);
        std::unique_ptr<const char16_t[]> arr2 = generateUTF16Array(size);
        unsigned left = StringHasher::computeHashAndMaskTop8Bits(std::span { arr1.get(), size });
        unsigned right = StringHasher::computeHashAndMaskTop8Bits(std::span { arr2.get(), size });
        ASSERT_EQ(left, right);

        for (size_t i = 0; i < size; i++)
            hash.addCharacter(arr2.get()[i]);
        unsigned result1 = hash.hashWithTop8BitsMasked();
        ASSERT_EQ(right, result1);
    }
}

TEST(WTF, StringHasher_SuperFastHash_VS_WYHash)
{
    if (!dumpHashingSpeedComparison)
        return;

    size_t sum = 0;
    dataLogLn("char16_t");
    for (size_t size = 2; size < (1 << 16); size *= 2) {
        Vector<char16_t> vector(size, [](size_t index) {
            return index & 0x7f;
        });
        sum += SuperFastHash::computeHashAndMaskTop8Bits(vector.span());
        sum += WYHash::computeHashAndMaskTop8Bits(vector.span());
        sum += StringHasher::computeHashAndMaskTop8Bits(vector.span());
        auto start = MonotonicTime::now();
        for (unsigned i = 0; i < 1e5; ++i)
            sum += SuperFastHash::computeHashAndMaskTop8Bits(vector.span());
        dataLogLn("SFH ", size, " -> ", MonotonicTime::now() - start);
        start = MonotonicTime::now();
        for (unsigned i = 0; i < 1e5; ++i)
            sum += WYHash::computeHashAndMaskTop8Bits(vector.span());
        dataLogLn("WYH ", size, " -> ", MonotonicTime::now() - start);
        start = MonotonicTime::now();
        for (unsigned i = 0; i < 1e5; ++i)
            sum += StringHasher::computeHashAndMaskTop8Bits(vector.span());
        dataLogLn("STH ", size, " -> ", MonotonicTime::now() - start);
    }

    dataLogLn("Latin1Character");
    for (size_t size = 2; size < (1 << 16); size *= 2) {
        Vector<Latin1Character> vector(size, [](size_t index) {
            return index & 0x7f;
        });
        sum += SuperFastHash::computeHashAndMaskTop8Bits(vector.span());
        sum += WYHash::computeHashAndMaskTop8Bits(vector.span());
        sum += StringHasher::computeHashAndMaskTop8Bits(vector.span());
        auto start = MonotonicTime::now();
        for (unsigned i = 0; i < 1e5; ++i)
            sum += SuperFastHash::computeHashAndMaskTop8Bits(vector.span());
        dataLogLn("SFH ", size, " -> ", MonotonicTime::now() - start);
        start = MonotonicTime::now();
        for (unsigned i = 0; i < 1e5; ++i)
            sum += WYHash::computeHashAndMaskTop8Bits(vector.span());
        dataLogLn("WYH ", size, " -> ", MonotonicTime::now() - start);
        start = MonotonicTime::now();
        for (unsigned i = 0; i < 1e5; ++i)
            sum += StringHasher::computeHashAndMaskTop8Bits(vector.span());
        dataLogLn("STH ", size, " -> ", MonotonicTime::now() - start);
    }
    dataLogLn(sum);
}

} // namespace TestWebKitAPI
