// SPDX-License-Identifier: GPL-2.0-only
//
// Regmap tables and other data for Cirrus Logic CS48L32 audio DSP.
//
// Copyright (C) 2018, 2020, 2022, 2025 Cirrus Logic, Inc. and
//               Cirrus Logic International Semiconductor Ltd.

#include <linux/array_size.h>
#include <linux/build_bug.h>
#include <linux/device.h>
#include <linux/linear_range.h>
#include <linux/regmap.h>
#include <linux/regulator/consumer.h>
#include <sound/cs48l32.h>
#include <sound/cs48l32_registers.h>

#include "cs48l32.h"

static const struct reg_sequence cs48l32_reva_patch[] = {
	{ 0x00001044, 0x0005000f },
	{ 0x00001c34, 0x000037e8 },
	{ 0x000046d8, 0x00000fe0 },
};

int cs48l32_apply_patch(struct cs48l32 *cs48l32)
{
	int ret;

	ret = regmap_register_patch(cs48l32->regmap, cs48l32_reva_patch,
				    ARRAY_SIZE(cs48l32_reva_patch));
	if (ret < 0)
		return dev_err_probe(cs48l32->dev, ret, "Failed to apply patch\n");

	return 0;
}

static const struct reg_default cs48l32_reg_default[] = {
	{ 0x00000c08, 0xe1000001 }, /* GPIO1_CTRL1 */
	{ 0x00000c0c, 0xe1000001 }, /* GPIO2_CTRL1 */
	{ 0x00000c10, 0xe1000001 }, /* GPIO3_CTRL1 */
	{ 0x00000c14, 0xe1000001 }, /* GPIO4_CTRL1 */
	{ 0x00000c18, 0xe1000001 }, /* GPIO5_CTRL1 */
	{ 0x00000c1c, 0xe1000001 }, /* GPIO6_CTRL1 */
	{ 0x00000c20, 0xe1000001 }, /* GPIO7_CTRL1 */
	{ 0x00000c24, 0xe1000001 }, /* GPIO8_CTRL1 */
	{ 0x00000c28, 0xe1000001 }, /* GPIO9_CTRL1 */
	{ 0x00000c2c, 0xe1000001 }, /* GPIO10_CTRL1 */
	{ 0x00000c30, 0xe1000001 }, /* GPIO11_CTRL1 */
	{ 0x00000c34, 0xe1000001 }, /* GPIO12_CTRL1 */
	{ 0x00000c38, 0xe1000001 }, /* GPIO13_CTRL1 */
	{ 0x00000c3c, 0xe1000001 }, /* GPIO14_CTRL1 */
	{ 0x00000c40, 0xe1000001 }, /* GPIO15_CTRL1 */
	{ 0x00000c44, 0xe1000001 }, /* GPIO16_CTRL1 */
	{ 0x00001020, 0x00000000 }, /* OUTPUT_SYS_CLK */
	{ 0x00001044, 0x0005000f }, /* AUXPDM_CTRL */
	{ 0x0000105c, 0x00000000 }, /* AUXPDM_CTRL2 */
	{ 0x00001400, 0x00000002 }, /* CLOCK32K */
	{ 0x00001404, 0x00000404 }, /* SYSTEM_CLOCK1 */
	{ 0x00001420, 0x00000003 }, /* SAMPLE_RATE1 */
	{ 0x00001424, 0x00000003 }, /* SAMPLE_RATE2 */
	{ 0x00001428, 0x00000003 }, /* SAMPLE_RATE3 */
	{ 0x0000142c, 0x00000003 }, /* SAMPLE_RATE4 */
	{ 0x00001c00, 0x00000002 }, /* FLL1_CONTROL1 */
	{ 0x00001c04, 0x88203004 }, /* FLL1_CONTROL2 */
	{ 0x00001c08, 0x00000000 }, /* FLL1_CONTROL3 */
	{ 0x00001c0c, 0x21f05001 }, /* FLL1_CONTROL4 */
	{ 0x00001ca0, 0x00000c04 }, /* FLL1_GPIO_CLOCK */
	{ 0x00002000, 0x00000006 }, /* CHARGE_PUMP1 */
	{ 0x00002408, 0x000003e4 }, /* LDO2_CTRL1 */
	{ 0x00002410, 0x000000e6 }, /* MICBIAS_CTRL1 */
	{ 0x00002418, 0x00000222 }, /* MICBIAS_CTRL5 */
	{ 0x00002710, 0x00004600 }, /* IRQ1_CTRL_AOD */
	{ 0x00004000, 0x00000000 }, /* INPUT_CONTROL */
	{ 0x00004008, 0x00000400 }, /* INPUT_RATE_CONTROL */
	{ 0x0000400c, 0x00000000 }, /* INPUT_CONTROL2 */
	{ 0x00004020, 0x00050020 }, /* INPUT1_CONTROL1 */
	{ 0x00004024, 0x00000000 }, /* IN1L_CONTROL1 */
	{ 0x00004028, 0x10800080 }, /* IN1L_CONTROL2 */
	{ 0x00004044, 0x00000000 }, /* IN1R_CONTROL1 */
	{ 0x00004048, 0x10800080 }, /* IN1R_CONTROL2 */
	{ 0x00004060, 0x00050020 }, /* INPUT2_CONTROL1 */
	{ 0x00004064, 0x00000000 }, /* IN2L_CONTROL1 */
	{ 0x00004068, 0x10800000 }, /* IN2L_CONTROL2 */
	{ 0x00004084, 0x00000000 }, /* IN2R_CONTROL1 */
	{ 0x00004088, 0x10800000 }, /* IN2R_CONTROL2 */
	{ 0x00004244, 0x00000002 }, /* INPUT_HPF_CONTROL */
	{ 0x00004248, 0x00000022 }, /* INPUT_VOL_CONTROL */
	{ 0x00004300, 0x00000000 }, /* AUXPDM_CONTROL1 */
	{ 0x00004304, 0x00000000 }, /* AUXPDM_CONTROL2 */
	{ 0x00004308, 0x00010008 }, /* AUXPDM1_CONTROL1 */
	{ 0x00004310, 0x00010008 }, /* AUXPDM2_CONTROL1 */
	{ 0x00004688, 0x00000000 }, /* ADC1L_ANA_CONTROL1 */
	{ 0x0000468c, 0x00000000 }, /* ADC1R_ANA_CONTROL1 */
	{ 0x00006000, 0x00000000 }, /* ASP1_ENABLES1 */
	{ 0x00006004, 0x00000028 }, /* ASP1_CONTROL1 */
	{ 0x00006008, 0x18180200 }, /* ASP1_CONTROL2 */
	{ 0x0000600c, 0x00000002 }, /* ASP1_CONTROL3 */
	{ 0x00006010, 0x03020100 }, /* ASP1_FRAME_CONTROL1 */
	{ 0x00006014, 0x07060504 }, /* ASP1_FRAME_CONTROL2 */
	{ 0x00006020, 0x03020100 }, /* ASP1_FRAME_CONTROL5 */
	{ 0x00006024, 0x07060504 }, /* ASP1_FRAME_CONTROL6 */
	{ 0x00006030, 0x00000020 }, /* ASP1_DATA_CONTROL1 */
	{ 0x00006040, 0x00000020 }, /* ASP1_DATA_CONTROL5 */
	{ 0x00006080, 0x00000000 }, /* ASP2_ENABLES1 */
	{ 0x00006084, 0x00000028 }, /* ASP2_CONTROL1 */
	{ 0x00006088, 0x18180200 }, /* ASP2_CONTROL2 */
	{ 0x0000608c, 0x00000002 }, /* ASP2_CONTROL3 */
	{ 0x00006090, 0x03020100 }, /* ASP2_FRAME_CONTROL1 */
	{ 0x000060a0, 0x03020100 }, /* ASP2_FRAME_CONTROL5 */
	{ 0x000060b0, 0x00000020 }, /* ASP2_DATA_CONTROL1 */
	{ 0x000060c0, 0x00000020 }, /* ASP2_DATA_CONTROL5 */
	{ 0x00008200, 0x00800000 }, /* ASP1TX1_INPUT1 */
	{ 0x00008204, 0x00800000 }, /* ASP1TX1_INPUT2 */
	{ 0x00008208, 0x00800000 }, /* ASP1TX1_INPUT3 */
	{ 0x0000820c, 0x00800000 }, /* ASP1TX1_INPUT4 */
	{ 0x00008210, 0x00800000 }, /* ASP1TX2_INPUT1 */
	{ 0x00008214, 0x00800000 }, /* ASP1TX2_INPUT2 */
	{ 0x00008218, 0x00800000 }, /* ASP1TX2_INPUT3 */
	{ 0x0000821c, 0x00800000 }, /* ASP1TX2_INPUT4 */
	{ 0x00008220, 0x00800000 }, /* ASP1TX3_INPUT1 */
	{ 0x00008224, 0x00800000 }, /* ASP1TX3_INPUT2 */
	{ 0x00008228, 0x00800000 }, /* ASP1TX3_INPUT3 */
	{ 0x0000822c, 0x00800000 }, /* ASP1TX3_INPUT4 */
	{ 0x00008230, 0x00800000 }, /* ASP1TX4_INPUT1 */
	{ 0x00008234, 0x00800000 }, /* ASP1TX4_INPUT2 */
	{ 0x00008238, 0x00800000 }, /* ASP1TX4_INPUT3 */
	{ 0x0000823c, 0x00800000 }, /* ASP1TX4_INPUT4 */
	{ 0x00008240, 0x00800000 }, /* ASP1TX5_INPUT1 */
	{ 0x00008244, 0x00800000 }, /* ASP1TX5_INPUT2 */
	{ 0x00008248, 0x00800000 }, /* ASP1TX5_INPUT3 */
	{ 0x0000824c, 0x00800000 }, /* ASP1TX5_INPUT4 */
	{ 0x00008250, 0x00800000 }, /* ASP1TX6_INPUT1 */
	{ 0x00008254, 0x00800000 }, /* ASP1TX6_INPUT2 */
	{ 0x00008258, 0x00800000 }, /* ASP1TX6_INPUT3 */
	{ 0x0000825c, 0x00800000 }, /* ASP1TX6_INPUT4 */
	{ 0x00008260, 0x00800000 }, /* ASP1TX7_INPUT1 */
	{ 0x00008264, 0x00800000 }, /* ASP1TX7_INPUT2 */
	{ 0x00008268, 0x00800000 }, /* ASP1TX7_INPUT3 */
	{ 0x0000826c, 0x00800000 }, /* ASP1TX7_INPUT4 */
	{ 0x00008270, 0x00800000 }, /* ASP1TX8_INPUT1 */
	{ 0x00008274, 0x00800000 }, /* ASP1TX8_INPUT2 */
	{ 0x00008278, 0x00800000 }, /* ASP1TX8_INPUT3 */
	{ 0x0000827c, 0x00800000 }, /* ASP1TX8_INPUT4 */
	{ 0x00008300, 0x00800000 }, /* ASP2TX1_INPUT1 */
	{ 0x00008304, 0x00800000 }, /* ASP2TX1_INPUT2 */
	{ 0x00008308, 0x00800000 }, /* ASP2TX1_INPUT3 */
	{ 0x0000830c, 0x00800000 }, /* ASP2TX1_INPUT4 */
	{ 0x00008310, 0x00800000 }, /* ASP2TX2_INPUT1 */
	{ 0x00008314, 0x00800000 }, /* ASP2TX2_INPUT2 */
	{ 0x00008318, 0x00800000 }, /* ASP2TX2_INPUT3 */
	{ 0x0000831c, 0x00800000 }, /* ASP2TX2_INPUT4 */
	{ 0x00008320, 0x00800000 }, /* ASP2TX3_INPUT1 */
	{ 0x00008324, 0x00800000 }, /* ASP2TX3_INPUT2 */
	{ 0x00008328, 0x00800000 }, /* ASP2TX3_INPUT3 */
	{ 0x0000832c, 0x00800000 }, /* ASP2TX3_INPUT4 */
	{ 0x00008330, 0x00800000 }, /* ASP2TX4_INPUT1 */
	{ 0x00008334, 0x00800000 }, /* ASP2TX4_INPUT2 */
	{ 0x00008338, 0x00800000 }, /* ASP2TX4_INPUT3 */
	{ 0x0000833c, 0x00800000 }, /* ASP2TX4_INPUT4 */
	{ 0x00008980, 0x00000000 }, /* ISRC1INT1_INPUT1 */
	{ 0x00008990, 0x00000000 }, /* ISRC1INT2_INPUT1 */
	{ 0x000089a0, 0x00000000 }, /* ISRC1INT3_INPUT1 */
	{ 0x000089b0, 0x00000000 }, /* ISRC1INT4_INPUT1 */
	{ 0x000089c0, 0x00000000 }, /* ISRC1DEC1_INPUT1 */
	{ 0x000089d0, 0x00000000 }, /* ISRC1DEC2_INPUT1 */
	{ 0x000089e0, 0x00000000 }, /* ISRC1DEC3_INPUT1 */
	{ 0x000089f0, 0x00000000 }, /* ISRC1DEC4_INPUT1 */
	{ 0x00008a00, 0x00000000 }, /* ISRC2INT1_INPUT1 */
	{ 0x00008a10, 0x00000000 }, /* ISRC2INT2_INPUT1 */
	{ 0x00008a40, 0x00000000 }, /* ISRC2DEC1_INPUT1 */
	{ 0x00008a50, 0x00000000 }, /* ISRC2DEC2_INPUT1 */
	{ 0x00008a80, 0x00000000 }, /* ISRC3INT1_INPUT1 */
	{ 0x00008a90, 0x00000000 }, /* ISRC3INT2_INPUT1 */
	{ 0x00008ac0, 0x00000000 }, /* ISRC3DEC1_INPUT1 */
	{ 0x00008ad0, 0x00000000 }, /* ISRC3DEC2_INPUT1 */
	{ 0x00008b80, 0x00800000 }, /* EQ1_INPUT1 */
	{ 0x00008b84, 0x00800000 }, /* EQ1_INPUT2 */
	{ 0x00008b88, 0x00800000 }, /* EQ1_INPUT3 */
	{ 0x00008b8c, 0x00800000 }, /* EQ1_INPUT4 */
	{ 0x00008b90, 0x00800000 }, /* EQ2_INPUT1 */
	{ 0x00008b94, 0x00800000 }, /* EQ2_INPUT2 */
	{ 0x00008b98, 0x00800000 }, /* EQ2_INPUT3 */
	{ 0x00008b9c, 0x00800000 }, /* EQ2_INPUT4 */
	{ 0x00008ba0, 0x00800000 }, /* EQ3_INPUT1 */
	{ 0x00008ba4, 0x00800000 }, /* EQ3_INPUT2 */
	{ 0x00008ba8, 0x00800000 }, /* EQ3_INPUT3 */
	{ 0x00008bac, 0x00800000 }, /* EQ3_INPUT4 */
	{ 0x00008bb0, 0x00800000 }, /* EQ4_INPUT1 */
	{ 0x00008bb4, 0x00800000 }, /* EQ4_INPUT2 */
	{ 0x00008bb8, 0x00800000 }, /* EQ4_INPUT3 */
	{ 0x00008bbc, 0x00800000 }, /* EQ4_INPUT4 */
	{ 0x00008c00, 0x00800000 }, /* DRC1L_INPUT1 */
	{ 0x00008c04, 0x00800000 }, /* DRC1L_INPUT2 */
	{ 0x00008c08, 0x00800000 }, /* DRC1L_INPUT3 */
	{ 0x00008c0c, 0x00800000 }, /* DRC1L_INPUT4 */
	{ 0x00008c10, 0x00800000 }, /* DRC1R_INPUT1 */
	{ 0x00008c14, 0x00800000 }, /* DRC1R_INPUT2 */
	{ 0x00008c18, 0x00800000 }, /* DRC1R_INPUT3 */
	{ 0x00008c1c, 0x00800000 }, /* DRC1R_INPUT4 */
	{ 0x00008c20, 0x00800000 }, /* DRC2L_INPUT1 */
	{ 0x00008c24, 0x00800000 }, /* DRC2L_INPUT2 */
	{ 0x00008c28, 0x00800000 }, /* DRC2L_INPUT3 */
	{ 0x00008c2c, 0x00800000 }, /* DRC2L_INPUT4 */
	{ 0x00008c30, 0x00800000 }, /* DRC2R_INPUT1 */
	{ 0x00008c34, 0x00800000 }, /* DRC2R_INPUT2 */
	{ 0x00008c38, 0x00800000 }, /* DRC2R_INPUT3 */
	{ 0x00008c3c, 0x00800000 }, /* DRC2R_INPUT4 */
	{ 0x00008c80, 0x00800000 }, /* LHPF1_INPUT1 */
	{ 0x00008c84, 0x00800000 }, /* LHPF1_INPUT2 */
	{ 0x00008c88, 0x00800000 }, /* LHPF1_INPUT3 */
	{ 0x00008c8c, 0x00800000 }, /* LHPF1_INPUT4 */
	{ 0x00008c90, 0x00800000 }, /* LHPF2_INPUT1 */
	{ 0x00008c94, 0x00800000 }, /* LHPF2_INPUT2 */
	{ 0x00008c98, 0x00800000 }, /* LHPF2_INPUT3 */
	{ 0x00008c9c, 0x00800000 }, /* LHPF2_INPUT4 */
	{ 0x00008ca0, 0x00800000 }, /* LHPF3_INPUT1 */
	{ 0x00008ca4, 0x00800000 }, /* LHPF3_INPUT2 */
	{ 0x00008ca8, 0x00800000 }, /* LHPF3_INPUT3 */
	{ 0x00008cac, 0x00800000 }, /* LHPF3_INPUT4 */
	{ 0x00008cb0, 0x00800000 }, /* LHPF4_INPUT1 */
	{ 0x00008cb4, 0x00800000 }, /* LHPF4_INPUT2 */
	{ 0x00008cb8, 0x00800000 }, /* LHPF4_INPUT3 */
	{ 0x00008cbc, 0x00800000 }, /* LHPF4_INPUT4 */
	{ 0x00009000, 0x00800000 }, /* DSP1RX1_INPUT1 */
	{ 0x00009004, 0x00800000 }, /* DSP1RX1_INPUT2 */
	{ 0x00009008, 0x00800000 }, /* DSP1RX1_INPUT3 */
	{ 0x0000900c, 0x00800000 }, /* DSP1RX1_INPUT4 */
	{ 0x00009010, 0x00800000 }, /* DSP1RX2_INPUT1 */
	{ 0x00009014, 0x00800000 }, /* DSP1RX2_INPUT2 */
	{ 0x00009018, 0x00800000 }, /* DSP1RX2_INPUT3 */
	{ 0x0000901c, 0x00800000 }, /* DSP1RX2_INPUT4 */
	{ 0x00009020, 0x00800000 }, /* DSP1RX3_INPUT1 */
	{ 0x00009024, 0x00800000 }, /* DSP1RX3_INPUT2 */
	{ 0x00009028, 0x00800000 }, /* DSP1RX3_INPUT3 */
	{ 0x0000902c, 0x00800000 }, /* DSP1RX3_INPUT4 */
	{ 0x00009030, 0x00800000 }, /* DSP1RX4_INPUT1 */
	{ 0x00009034, 0x00800000 }, /* DSP1RX4_INPUT2 */
	{ 0x00009038, 0x00800000 }, /* DSP1RX4_INPUT3 */
	{ 0x0000903c, 0x00800000 }, /* DSP1RX4_INPUT4 */
	{ 0x00009040, 0x00800000 }, /* DSP1RX5_INPUT1 */
	{ 0x00009044, 0x00800000 }, /* DSP1RX5_INPUT2 */
	{ 0x00009048, 0x00800000 }, /* DSP1RX5_INPUT3 */
	{ 0x0000904c, 0x00800000 }, /* DSP1RX5_INPUT4 */
	{ 0x00009050, 0x00800000 }, /* DSP1RX6_INPUT1 */
	{ 0x00009054, 0x00800000 }, /* DSP1RX6_INPUT2 */
	{ 0x00009058, 0x00800000 }, /* DSP1RX6_INPUT3 */
	{ 0x0000905c, 0x00800000 }, /* DSP1RX6_INPUT4 */
	{ 0x00009060, 0x00800000 }, /* DSP1RX7_INPUT1 */
	{ 0x00009064, 0x00800000 }, /* DSP1RX7_INPUT2 */
	{ 0x00009068, 0x00800000 }, /* DSP1RX7_INPUT3 */
	{ 0x0000906c, 0x00800000 }, /* DSP1RX7_INPUT4 */
	{ 0x00009070, 0x00800000 }, /* DSP1RX8_INPUT1 */
	{ 0x00009074, 0x00800000 }, /* DSP1RX8_INPUT2 */
	{ 0x00009078, 0x00800000 }, /* DSP1RX8_INPUT3 */
	{ 0x0000907c, 0x00800000 }, /* DSP1RX8_INPUT4 */
	{ 0x0000a400, 0x00000000 }, /* ISRC1_CONTROL1 */
	{ 0x0000a404, 0x00000000 }, /* ISRC1_CONTROL2 */
	{ 0x0000a510, 0x00000000 }, /* ISRC2_CONTROL1 */
	{ 0x0000a514, 0x00000000 }, /* ISRC2_CONTROL2 */
	{ 0x0000a620, 0x00000000 }, /* ISRC3_CONTROL1 */
	{ 0x0000a624, 0x00000000 }, /* ISRC3_CONTROL2 */
	{ 0x0000a800, 0x00000000 }, /* FX_SAMPLE_RATE */
	{ 0x0000a808, 0x00000000 }, /* EQ_CONTROL1 */
	{ 0x0000a80c, 0x00000000 }, /* EQ_CONTROL2 */
	{ 0x0000a810, 0x0c0c0c0c }, /* EQ1_GAIN1 */
	{ 0x0000a814, 0x0000000c }, /* EQ1_GAIN2 */
	{ 0x0000a818, 0x03fe0fc8 }, /* EQ1_BAND1_COEFF1 */
	{ 0x0000a81c, 0x00000b75 }, /* EQ1_BAND1_COEFF2 */
	{ 0x0000a820, 0x000000e0 }, /* EQ1_BAND1_PG */
	{ 0x0000a824, 0xf1361ec4 }, /* EQ1_BAND2_COEFF1 */
	{ 0x0000a828, 0x00000409 }, /* EQ1_BAND2_COEFF2 */
	{ 0x0000a82c, 0x000004cc }, /* EQ1_BAND2_PG */
	{ 0x0000a830, 0xf3371c9b }, /* EQ1_BAND3_COEFF1 */
	{ 0x0000a834, 0x0000040b }, /* EQ1_BAND3_COEFF2 */
	{ 0x0000a838, 0x00000cbb }, /* EQ1_BAND3_PG */
	{ 0x0000a83c, 0xf7d916f8 }, /* EQ1_BAND4_COEFF1 */
	{ 0x0000a840, 0x0000040a }, /* EQ1_BAND4_COEFF2 */
	{ 0x0000a844, 0x00001f14 }, /* EQ1_BAND4_PG */
	{ 0x0000a848, 0x0563058c }, /* EQ1_BAND5_COEFF1 */
	{ 0x0000a84c, 0x00000000 }, /* EQ1_BAND5_COEFF1 + 4 */
	{ 0x0000a850, 0x00004000 }, /* EQ1_BAND5_PG */
	{ 0x0000a854, 0x0c0c0c0c }, /* EQ2_GAIN1 */
	{ 0x0000a858, 0x0000000c }, /* EQ2_GAIN2 */
	{ 0x0000a85c, 0x03fe0fc8 }, /* EQ2_BAND1_COEFF1 */
	{ 0x0000a860, 0x00000b75 }, /* EQ2_BAND1_COEFF2 */
	{ 0x0000a864, 0x000000e0 }, /* EQ2_BAND1_PG */
	{ 0x0000a868, 0xf1361ec4 }, /* EQ2_BAND2_COEFF1 */
	{ 0x0000a86c, 0x00000409 }, /* EQ2_BAND2_COEFF2 */
	{ 0x0000a870, 0x000004cc }, /* EQ2_BAND2_PG */
	{ 0x0000a874, 0xf3371c9b }, /* EQ2_BAND3_COEFF1 */
	{ 0x0000a878, 0x0000040b }, /* EQ2_BAND3_COEFF2 */
	{ 0x0000a87c, 0x00000cbb }, /* EQ2_BAND3_PG */
	{ 0x0000a880, 0xf7d916f8 }, /* EQ2_BAND4_COEFF1 */
	{ 0x0000a884, 0x0000040a }, /* EQ2_BAND4_COEFF2 */
	{ 0x0000a888, 0x00001f14 }, /* EQ2_BAND4_PG */
	{ 0x0000a88c, 0x0563058c }, /* EQ2_BAND5_COEFF1 */
	{ 0x0000a890, 0x00000000 }, /* EQ2_BAND5_COEFF1 + 4 */
	{ 0x0000a894, 0x00004000 }, /* EQ2_BAND5_PG */
	{ 0x0000a898, 0x0c0c0c0c }, /* EQ3_GAIN1 */
	{ 0x0000a89c, 0x0000000c }, /* EQ3_GAIN2 */
	{ 0x0000a8a0, 0x03fe0fc8 }, /* EQ3_BAND1_COEFF1 */
	{ 0x0000a8a4, 0x00000b75 }, /* EQ3_BAND1_COEFF2 */
	{ 0x0000a8a8, 0x000000e0 }, /* EQ3_BAND1_PG */
	{ 0x0000a8ac, 0xf1361ec4 }, /* EQ3_BAND2_COEFF1 */
	{ 0x0000a8b0, 0x00000409 }, /* EQ3_BAND2_COEFF2 */
	{ 0x0000a8b4, 0x000004cc }, /* EQ3_BAND2_PG */
	{ 0x0000a8b8, 0xf3371c9b }, /* EQ3_BAND3_COEFF1 */
	{ 0x0000a8bc, 0x0000040b }, /* EQ3_BAND3_COEFF2 */
	{ 0x0000a8c0, 0x00000cbb }, /* EQ3_BAND3_PG */
	{ 0x0000a8c4, 0xf7d916f8 }, /* EQ3_BAND4_COEFF1 */
	{ 0x0000a8c8, 0x0000040a }, /* EQ3_BAND4_COEFF2 */
	{ 0x0000a8cc, 0x00001f14 }, /* EQ3_BAND4_PG */
	{ 0x0000a8d0, 0x0563058c }, /* EQ3_BAND5_COEFF1 */
	{ 0x0000a8d4, 0x00000000 }, /* EQ3_BAND5_COEFF1 + 4 */
	{ 0x0000a8d8, 0x00004000 }, /* EQ3_BAND5_PG */
	{ 0x0000a8dc, 0x0c0c0c0c }, /* EQ4_GAIN1 */
	{ 0x0000a8e0, 0x0000000c }, /* EQ4_GAIN2 */
	{ 0x0000a8e4, 0x03fe0fc8 }, /* EQ4_BAND1_COEFF1 */
	{ 0x0000a8e8, 0x00000b75 }, /* EQ4_BAND1_COEFF2 */
	{ 0x0000a8ec, 0x000000e0 }, /* EQ4_BAND1_PG */
	{ 0x0000a8f0, 0xf1361ec4 }, /* EQ4_BAND2_COEFF1 */
	{ 0x0000a8f4, 0x00000409 }, /* EQ4_BAND2_COEFF2 */
	{ 0x0000a8f8, 0x000004cc }, /* EQ4_BAND2_PG */
	{ 0x0000a8fc, 0xf3371c9b }, /* EQ4_BAND3_COEFF1 */
	{ 0x0000a900, 0x0000040b }, /* EQ4_BAND3_COEFF2 */
	{ 0x0000a904, 0x00000cbb }, /* EQ4_BAND3_PG */
	{ 0x0000a908, 0xf7d916f8 }, /* EQ4_BAND4_COEFF1 */
	{ 0x0000a90c, 0x0000040a }, /* EQ4_BAND4_COEFF2 */
	{ 0x0000a910, 0x00001f14 }, /* EQ4_BAND4_PG */
	{ 0x0000a914, 0x0563058c }, /* EQ4_BAND5_COEFF1 */
	{ 0x0000a918, 0x00000000 }, /* EQ4_BAND5_COEFF1 + 4 */
	{ 0x0000a91c, 0x00004000 }, /* EQ4_BAND5_PG */
	{ 0x0000aa30, 0x00000000 }, /* LHPF_CONTROL1 */
	{ 0x0000aa34, 0x00000000 }, /* LHPF_CONTROL2 */
	{ 0x0000aa38, 0x00000000 }, /* LHPF1_COEFF */
	{ 0x0000aa3c, 0x00000000 }, /* LHPF2_COEFF */
	{ 0x0000aa40, 0x00000000 }, /* LHPF3_COEFF */
	{ 0x0000aa44, 0x00000000 }, /* LHPF4_COEFF */
	{ 0x0000ab00, 0x00000000 }, /* DRC1_CONTROL1 */
	{ 0x0000ab04, 0x49130018 }, /* DRC1_CONTROL2 */
	{ 0x0000ab08, 0x00000018 }, /* DRC1_CONTROL3 */
	{ 0x0000ab0c, 0x00000000 }, /* DRC1_CONTROL4 */
	{ 0x0000ab14, 0x00000000 }, /* DRC2_CONTROL1 */
	{ 0x0000ab18, 0x49130018 }, /* DRC2_CONTROL2 */
	{ 0x0000ab1c, 0x00000018 }, /* DRC2_CONTROL3 */
	{ 0x0000ab20, 0x00000000 }, /* DRC2_CONTROL4 */
	{ 0x0000b000, 0x00000000 }, /* TONE_GENERATOR1 */
	{ 0x0000b004, 0x00100000 }, /* TONE_GENERATOR2 */
	{ 0x0000b400, 0x00000000 }, /* COMFORT_NOISE_GENERATOR */
	{ 0x0000b800, 0x00000000 }, /* US_CONTROL */
	{ 0x0000b804, 0x00002020 }, /* US1_CONTROL */
	{ 0x0000b808, 0x00000000 }, /* US1_DET_CONTROL */
	{ 0x0000b814, 0x00002020 }, /* US2_CONTROL */
	{ 0x0000b818, 0x00000000 }, /* US2_DET_CONTROL */
	{ 0x00018110, 0x00000700 }, /* IRQ1_MASK_1 */
	{ 0x00018114, 0x00000004 }, /* IRQ1_MASK_2 */
	{ 0x00018120, 0x03ff0000 }, /* IRQ1_MASK_5 */
	{ 0x00018124, 0x00000103 }, /* IRQ1_MASK_6 */
	{ 0x00018128, 0x003f0000 }, /* IRQ1_MASK_7 */
	{ 0x00018130, 0xff00000f }, /* IRQ1_MASK_9 */
	{ 0x00018138, 0xffff0000 }, /* IRQ1_MASK_11 */
};

static bool cs48l32_readable_register(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case CS48L32_DEVID:
	case CS48L32_REVID:
	case CS48L32_OTPID:
	case CS48L32_SFT_RESET:
	case CS48L32_CTRL_IF_DEBUG3:
	case CS48L32_MCU_CTRL1:
	case CS48L32_GPIO1_CTRL1 ... CS48L32_GPIO16_CTRL1:
	case CS48L32_OUTPUT_SYS_CLK:
	case CS48L32_AUXPDM_CTRL:
	case CS48L32_AUXPDM_CTRL2:
	case CS48L32_CLOCK32K:
	case CS48L32_SYSTEM_CLOCK1 ... CS48L32_SYSTEM_CLOCK2:
	case CS48L32_SAMPLE_RATE1 ... CS48L32_SAMPLE_RATE4:
	case CS48L32_FLL1_CONTROL1 ... CS48L32_FLL1_GPIO_CLOCK:
	case CS48L32_CHARGE_PUMP1:
	case CS48L32_LDO2_CTRL1:
	case CS48L32_MICBIAS_CTRL1:
	case CS48L32_MICBIAS_CTRL5:
	case CS48L32_IRQ1_CTRL_AOD:
	case CS48L32_INPUT_CONTROL:
	case CS48L32_INPUT_STATUS:
	case CS48L32_INPUT_RATE_CONTROL:
	case CS48L32_INPUT_CONTROL2:
	case CS48L32_INPUT_CONTROL3:
	case CS48L32_INPUT1_CONTROL1:
	case CS48L32_IN1L_CONTROL1 ... CS48L32_IN1L_CONTROL2:
	case CS48L32_IN1R_CONTROL1 ... CS48L32_IN1R_CONTROL2:
	case CS48L32_INPUT2_CONTROL1:
	case CS48L32_IN2L_CONTROL1 ... CS48L32_IN2L_CONTROL2:
	case CS48L32_IN2R_CONTROL1 ... CS48L32_IN2R_CONTROL2:
	case CS48L32_INPUT_HPF_CONTROL:
	case CS48L32_INPUT_VOL_CONTROL:
	case CS48L32_AUXPDM_CONTROL1:
	case CS48L32_AUXPDM_CONTROL2:
	case CS48L32_AUXPDM1_CONTROL1:
	case CS48L32_AUXPDM2_CONTROL1:
	case CS48L32_ADC1L_ANA_CONTROL1:
	case CS48L32_ADC1R_ANA_CONTROL1:
	case CS48L32_ASP1_ENABLES1 ... CS48L32_ASP1_DATA_CONTROL5:
	case CS48L32_ASP2_ENABLES1 ... CS48L32_ASP2_DATA_CONTROL5:
	case CS48L32_ASP1TX1_INPUT1 ... CS48L32_ASP1TX8_INPUT4:
	case CS48L32_ASP2TX1_INPUT1 ... CS48L32_ASP2TX4_INPUT4:
	case CS48L32_ISRC1INT1_INPUT1 ... CS48L32_ISRC1DEC4_INPUT1:
	case CS48L32_ISRC2INT1_INPUT1 ... CS48L32_ISRC2DEC2_INPUT1:
	case CS48L32_ISRC3INT1_INPUT1 ... CS48L32_ISRC3DEC2_INPUT1:
	case CS48L32_EQ1_INPUT1 ... CS48L32_EQ4_INPUT4:
	case CS48L32_DRC1L_INPUT1 ... CS48L32_DRC1R_INPUT4:
	case CS48L32_DRC2L_INPUT1 ... CS48L32_DRC2R_INPUT4:
	case CS48L32_LHPF1_INPUT1 ... CS48L32_LHPF1_INPUT4:
	case CS48L32_LHPF2_INPUT1 ... CS48L32_LHPF2_INPUT4:
	case CS48L32_LHPF3_INPUT1 ... CS48L32_LHPF3_INPUT4:
	case CS48L32_LHPF4_INPUT1 ... CS48L32_LHPF4_INPUT4:
	case CS48L32_DSP1RX1_INPUT1 ... CS48L32_DSP1RX8_INPUT4:
	case CS48L32_ISRC1_CONTROL1 ... CS48L32_ISRC1_CONTROL2:
	case CS48L32_ISRC2_CONTROL1 ... CS48L32_ISRC2_CONTROL2:
	case CS48L32_ISRC3_CONTROL1 ... CS48L32_ISRC3_CONTROL2:
	case CS48L32_FX_SAMPLE_RATE:
	case CS48L32_EQ_CONTROL1 ... CS48L32_EQ_CONTROL2:
	case CS48L32_EQ1_GAIN1 ... CS48L32_EQ1_BAND5_PG:
	case CS48L32_EQ2_GAIN1 ... CS48L32_EQ2_BAND5_PG:
	case CS48L32_EQ3_GAIN1 ... CS48L32_EQ3_BAND5_PG:
	case CS48L32_EQ4_GAIN1 ... CS48L32_EQ4_BAND5_PG:
	case CS48L32_LHPF_CONTROL1 ... CS48L32_LHPF_CONTROL2:
	case CS48L32_LHPF1_COEFF ... CS48L32_LHPF4_COEFF:
	case CS48L32_DRC1_CONTROL1 ... CS48L32_DRC1_CONTROL4:
	case CS48L32_DRC2_CONTROL1 ... CS48L32_DRC2_CONTROL4:
	case CS48L32_TONE_GENERATOR1 ... CS48L32_TONE_GENERATOR2:
	case CS48L32_COMFORT_NOISE_GENERATOR:
	case CS48L32_US_CONTROL:
	case CS48L32_US1_CONTROL:
	case CS48L32_US1_DET_CONTROL:
	case CS48L32_US2_CONTROL:
	case CS48L32_US2_DET_CONTROL:
	case CS48L32_DSP1_XM_SRAM_IBUS_SETUP_0 ... CS48L32_DSP1_XM_SRAM_IBUS_SETUP_24:
	case CS48L32_DSP1_YM_SRAM_IBUS_SETUP_0 ... CS48L32_DSP1_YM_SRAM_IBUS_SETUP_8:
	case CS48L32_DSP1_PM_SRAM_IBUS_SETUP_0 ... CS48L32_DSP1_PM_SRAM_IBUS_SETUP_7:
	case CS48L32_IRQ1_STATUS:
	case CS48L32_IRQ1_EINT_1 ... CS48L32_IRQ1_EINT_11:
	case CS48L32_IRQ1_STS_1 ... CS48L32_IRQ1_STS_11:
	case CS48L32_IRQ1_MASK_1 ... CS48L32_IRQ1_MASK_11:
	case CS48L32_DSP1_XMEM_PACKED_0 ... CS48L32_DSP1_XMEM_PACKED_LAST:
	case CS48L32_DSP1_SYS_INFO_ID ... CS48L32_DSP1_AHBM_WINDOW_DEBUG_1:
	case CS48L32_DSP1_XMEM_UNPACKED24_0 ... CS48L32_DSP1_XMEM_UNPACKED24_LAST:
	case CS48L32_DSP1_CLOCK_FREQ ... CS48L32_DSP1_SAMPLE_RATE_TX8:
	case CS48L32_DSP1_SCRATCH1 ... CS48L32_DSP1_SCRATCH4:
	case CS48L32_DSP1_CCM_CORE_CONTROL ... CS48L32_DSP1_STREAM_ARB_RESYNC_MSK1:
	case CS48L32_DSP1_YMEM_PACKED_0 ... CS48L32_DSP1_YMEM_PACKED_LAST:
	case CS48L32_DSP1_YMEM_UNPACKED24_0 ... CS48L32_DSP1_YMEM_UNPACKED24_LAST:
	case CS48L32_DSP1_PMEM_0 ... CS48L32_DSP1_PMEM_LAST:
		return true;
	default:
		return false;
	}
}

static bool cs48l32_volatile_register(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case CS48L32_DEVID:
	case CS48L32_REVID:
	case CS48L32_OTPID:
	case CS48L32_SFT_RESET:
	case CS48L32_CTRL_IF_DEBUG3:
	case CS48L32_MCU_CTRL1:
	case CS48L32_SYSTEM_CLOCK2:
	case CS48L32_FLL1_CONTROL5:
	case CS48L32_FLL1_CONTROL6:
	case CS48L32_INPUT_STATUS:
	case CS48L32_INPUT_CONTROL3:
	case CS48L32_DSP1_XM_SRAM_IBUS_SETUP_0 ... CS48L32_DSP1_XM_SRAM_IBUS_SETUP_24:
	case CS48L32_DSP1_YM_SRAM_IBUS_SETUP_0 ... CS48L32_DSP1_YM_SRAM_IBUS_SETUP_8:
	case CS48L32_DSP1_PM_SRAM_IBUS_SETUP_0 ... CS48L32_DSP1_PM_SRAM_IBUS_SETUP_7:
	case CS48L32_IRQ1_STATUS:
	case CS48L32_IRQ1_EINT_1 ... CS48L32_IRQ1_EINT_11:
	case CS48L32_IRQ1_STS_1 ... CS48L32_IRQ1_STS_11:
	case CS48L32_DSP1_XMEM_PACKED_0 ... CS48L32_DSP1_XMEM_PACKED_LAST:
	case CS48L32_DSP1_SYS_INFO_ID ... CS48L32_DSP1_AHBM_WINDOW_DEBUG_1:
	case CS48L32_DSP1_XMEM_UNPACKED24_0 ... CS48L32_DSP1_XMEM_UNPACKED24_LAST:
	case CS48L32_DSP1_CLOCK_FREQ ... CS48L32_DSP1_SAMPLE_RATE_TX8:
	case CS48L32_DSP1_SCRATCH1 ... CS48L32_DSP1_SCRATCH4:
	case CS48L32_DSP1_CCM_CORE_CONTROL ... CS48L32_DSP1_STREAM_ARB_RESYNC_MSK1:
	case CS48L32_DSP1_YMEM_PACKED_0 ... CS48L32_DSP1_YMEM_PACKED_LAST:
	case CS48L32_DSP1_YMEM_UNPACKED24_0 ... CS48L32_DSP1_YMEM_UNPACKED24_LAST:
	case CS48L32_DSP1_PMEM_0 ... CS48L32_DSP1_PMEM_LAST:
		return true;
	default:
		return false;
	}
}

/*
 * The bus bridge requires DSP packed memory registers to be accessed in
 * aligned block multiples.
 * Mark precious to prevent regmap debugfs causing an illegal bus transaction.
 */
static bool cs48l32_precious_register(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case CS48L32_DSP1_XMEM_PACKED_0 ... CS48L32_DSP1_XMEM_PACKED_LAST:
	case CS48L32_DSP1_YMEM_PACKED_0 ... CS48L32_DSP1_YMEM_PACKED_LAST:
	case CS48L32_DSP1_PMEM_0 ... CS48L32_DSP1_PMEM_LAST:
		return true;
	default:
		return false;
	}
}

static const struct regmap_config cs48l32_regmap = {
	.name = "cs48l32",
	.reg_bits = 32,
	.reg_stride = 4,
	.pad_bits = 32,
	.val_bits = 32,
	.reg_format_endian = REGMAP_ENDIAN_BIG,
	.val_format_endian = REGMAP_ENDIAN_BIG,

	.max_register = CS48L32_DSP1_PMEM_LAST,
	.readable_reg = &cs48l32_readable_register,
	.volatile_reg = &cs48l32_volatile_register,
	.precious_reg = &cs48l32_precious_register,

	.cache_type = REGCACHE_MAPLE,
	.reg_defaults = cs48l32_reg_default,
	.num_reg_defaults = ARRAY_SIZE(cs48l32_reg_default),
};

int cs48l32_create_regmap(struct spi_device *spi, struct cs48l32 *cs48l32)
{
	cs48l32->regmap = devm_regmap_init_spi(spi, &cs48l32_regmap);

	return PTR_ERR_OR_ZERO(cs48l32->regmap);
}
